/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.DuplicateKeyException;

import jp.co.intellilink.hinemos.importtool.conf.performance.CollectorItemCodeFactory;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTabelDefine;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.monitor.bean.MonitorPerfCheckInfo;
import com.clustercontrol.performance.monitor.ejb.session.MonitorPerfController;

/**
 * リソース監視情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMonitorPerformance extends ImportMonitorNumeric {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitorPerformance.class);

	@SuppressWarnings("unchecked")
	private ArrayList monitorList = null;

	private ArrayList<ArrayList<String>> monitorInfos = null;

	/**
	 * リソース監視情報をインポートする。<br>
	 */
	public void importPerformance() {
		log.info(Messages.getMsg("Resource.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//リソース監視情報のCSVファイルを読み込む
		monitorInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/RESOURCE.csv");

		//監視項目リスト取得
		monitorList = getMonitorList();

		for(ArrayList<String> info : monitorInfos){
			MonitorInfo monitorInfo = createMonitorInfo(info);
			registerMonitorInfo(monitorInfo);
		}

		log.info(Messages.getMsg("Resource.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorInfo(java.util.ArrayList)
	 */
	@Override
	protected MonitorInfo createMonitorInfo(ArrayList<String> info){
		MonitorInfo monitorInfo = new MonitorInfo();

		//監視基本情報
		createMonitorBase(info, monitorInfo);

		//監視条件情報
		createMonitorRule(info.subList(3, 5), monitorInfo);

		//監視情報
		createMonitorCheck(info.subList(5, 6), monitorInfo);

		//監視単位情報
		createMonitorBlock(info.subList(6, 7), monitorInfo);

		//監視判定情報
		createMonitorJudgement(info.subList(7, 12), monitorInfo);

		//監視通知情報
		createMonitorNotify(info.subList(12, 17), monitorInfo);

		//アプリケーション
		createMonitorApplication(info.subList(17, 18), monitorInfo);

		//有効/無効
		createMonitorVaild(info.subList(18, 19), monitorInfo);

		return monitorInfo;
	}

	/**
	 * プロセス監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorCheck(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		MonitorPerfCheckInfo perfCheckInfo = new MonitorPerfCheckInfo();

		List<CollectorItemInfo> itemInfoList = 
			CollectorItemCodeFactory.getAvailableCollectorItemList(monitorInfo.getFacilityId());

		for(CollectorItemInfo itemInfo : itemInfoList){
			String itemName = CollectorItemCodeFactory.getFullItemName(itemInfo);

			if(itemName.equals(info.get(0))){	
				perfCheckInfo.setItemCode(itemInfo.getItemCode());
				perfCheckInfo.setDeviceName(itemInfo.getDisplayName());
				break;
			}
		}

		monitorInfo.setCheckInfo(perfCheckInfo);

		super.createMonitorCheck(info, monitorInfo);
	}

	/**
	 * 監視情報を登録する。<br>
	 * 
	 * @param info 監視情報オブジェクト
	 */
	protected void registerMonitorInfo(MonitorInfo info) {
		MonitorPerfController perf = EjbConnectionManager.getConnectionManager().getMonitorPerfController();

		try {
			@SuppressWarnings("unchecked")
			ArrayList list = checkMonitorList(info.getMonitorId(), monitorList);
			if(list instanceof ArrayList){
				Date createTime = (Date)list.get(MonitorTabelDefine.CREATE_TIME);
				String createUser = (String)list.get(MonitorTabelDefine.CREATE_USER);
				info.setRegDate(new Timestamp(createTime.getTime()));
				info.setRegUser(createUser);

				MonitorInfo oldInfo = getPerformance(info.getMonitorId());
				String collectorId = ((MonitorPerfCheckInfo)oldInfo.getCheckInfo()).getCollectorId();
				((MonitorPerfCheckInfo)info.getCheckInfo()).setCollectorId(collectorId);

				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				perf.modifyPerformance(info);
			}
			else{
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("add") + ")");

				perf.addPerformance(info);
			}
		} catch (DuplicateKeyException e) {
			String args[] = {info.getMonitorId()};
			log.error(Messages.getMsg("ImportTool.Common.1", args));
			System.exit(20);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * リソース監視情報を取得する。<br>
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @return リソース監視情報
	 */
	protected MonitorInfo getPerformance(String monitorId) {

		MonitorPerfController perf = EjbConnectionManager.getConnectionManager().getMonitorPerfController();

		MonitorInfo info = null;
		try {
			info = perf.getPerformance(monitorId);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return info;
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getMonitorList() {

		MonitorPerfController perf = EjbConnectionManager.getConnectionManager().getMonitorPerfController();

		ArrayList records = null;
		try {
			records = perf.getPerformanceList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}