/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.repository;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * ノード情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportNode {
	
	// ログ出力
	private static Log log = LogFactory.getLog(ExportNode.class);

    /**
     * ノード情報をExportする。<br>
     */
    public void exportNode() {
    	log.info(Messages.getMsg("Node.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//ファシリティIDリスト取得
    	ArrayList idList = getNodeList();
    	
    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				String id = (String)itr.next();
				
				Object[] args = {id};
				log.info(Messages.getMsg("FacilityID", args));
				
				NodeInfo info = getNode(id);
				csvOutput.add(createCsvFormat(info));
			}
			
			//ファシリティIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
		String timestamp = "";
		if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			timestamp = "_" + format.format(new Date());
		}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/NODE" + timestamp + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("Node.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
    /**
	 * ノード情報をExportする。<br>
	 * 
	 * @param info ノード情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCsvFormat(NodeInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		//ファシリティID
		csvFormat.add(getStringValue(info.getFacilityId()));
		//ファシリティ名
		csvFormat.add(getStringValue(info.getFacilityName()));
		//説明
		csvFormat.add(getStringValue(info.getDescription()));
		//プラットフォーム
		csvFormat.add(getStringValue(info.getPlatformFamily()));
		//H/Wタイプ
		csvFormat.add(getStringValue(info.getHardwareType()));
		//画面アイコンイメージ
		csvFormat.add(getStringValue(info.getIconImage()));
		//管理対象
		csvFormat.add(getBooleanValue(info.isValid()));
		
		//SNMPポート番号
		csvFormat.add(getIntegerValue(info.getSnmpPort()));
		//SNMPコミュニティ名
		csvFormat.add(getStringValue(info.getSnmpCommunity()));
		//SNMPバージョン
		String snmpVersion = getStringValue(info.getSnmpVersion());
		if(!snmpVersion.equals(SnmpVersionConstant.STRING_V1) && !snmpVersion.equals(SnmpVersionConstant.STRING_V2))
			snmpVersion = "";
		csvFormat.add(snmpVersion);
		//SNMPタイムアウト
		csvFormat.add(getIntegerValue(info.getSnmpTimeout()));
		//SNMPリトライ回数
		csvFormat.add(getIntegerValue(info.getSnmpRetryCount()));
		//SNMPプロキシ
		csvFormat.add(getStringValue(info.getSnmpProxy()));
		
		//WBEM接続ユーザ名
		csvFormat.add(getStringValue(info.getWbemUser()));
		//WBEM接続ユーザパスワード
		csvFormat.add(getStringValue(info.getWbemUserPassword()));
		//WBEMポート番号
		csvFormat.add(getIntegerValue(info.getWbemPort()));
		//WBEMプロトコル
		csvFormat.add(getStringValue(info.getWbemProtocol()));
		//WBEMタイムアウト
		csvFormat.add(getIntegerValue(info.getWbemTimeout()));
		//WBEMリトライ回数
		csvFormat.add(getIntegerValue(info.getWbemRetryCount()));
		
		//DHCPクライアント
		csvFormat.add(getBooleanValue(info.getDhcpClient()));
		//IPアドレス種別
		csvFormat.add(getIntegerValue(info.getIpAddressType()));
		//IPアドレスバージョン
		csvFormat.add(getIntegerValue(info.getIpAddressVersion()));
		//IPアドレスV4
		csvFormat.add(getStringValue(info.getIpAddressV4()));
		//IPアドレスV6
		csvFormat.add(getStringValue(info.getIpAddressV6()));
		//ホスト名
		ArrayList<NodeHostnameInfo> hostnames = info.getNodeHostnameInfo();
		csvFormat.add(getStringValue(hostnames.get(0).getHostname()));
		
		//ノード名
		csvFormat.add(getStringValue(info.getNodeName()));
		//OS名
		csvFormat.add(getStringValue(info.getOsName()));
		//OSリリース
		csvFormat.add(getStringValue(info.getOsRelease()));
		//OSバージョン
		csvFormat.add(CheckString.checkReturn(getStringValue(info.getOsVersion())));
		//文字セット
		csvFormat.add(getStringValue(info.getCharacterSet()));
		
		//仮想化ノード種別
		csvFormat.add(getStringValue(info.getVirtualizationNodeType()));
		//仮想化管理ノード
		csvFormat.add(getStringValue(info.getVmManagementNode()));
		//仮想マシンINDEX
		csvFormat.add(getIntegerValue(info.getVmIndex()));
		//仮想マシン名
		csvFormat.add(getStringValue(info.getVmName()));
		//仮想化ソリューション
		csvFormat.add(getStringValue(info.getVirtualizationSolution()));
		//仮想マシンID
		csvFormat.add(getStringValue(info.getVmId()));
		//仮想化ソフト接続ユーザ
		csvFormat.add(getStringValue(info.getVmUser()));
		//仮想化ソフト接続ユーザパスワード
		csvFormat.add(getStringValue(info.getVmUserPassword()));
		//仮想化ソフト接続プロトコル
		csvFormat.add(getStringValue(info.getVmProtocol()));
		
		return csvFormat;
	}
	
	/**
	 * 文字列を取得する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getStringValue(String values) {
		return values;
	}
	
	/**
	 * 数値を文字列に変換する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getIntegerValue(Integer values) {
		return values.toString();
	}
	
	/**
	 * Booleanを文字列に変換する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getBooleanValue(Boolean values) {
		if(values.booleanValue())
			return YesNoConstant.typeToString(YesNoConstant.TYPE_YES);
		else
			return YesNoConstant.typeToString(YesNoConstant.TYPE_NO);
	}
	
    /**
     * ノード情報を取得する。<br>
     * 
     * @param facilityId 取得対象のファシリティID
     * @return ノード情報
     */
    protected NodeInfo getNode(String facilityId) {

    	RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

    	NodeInfo info = null;
        try {
        	info = repository.getNode(facilityId);
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return info;
    }
    
    /**
     * ファシリティIDリストを取得する。<br>
     * 
     * @return ファシリティID一覧
     */
    protected ArrayList getNodeList() {

    	RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

        ArrayList records = null;
        try {
            records = repository.getNodeFacilityIdList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
        return records;
    }
}