/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.util.ArrayList;
import java.util.List;

import jp.co.intellilink.hinemos.export.conf.monitor.JdbcDriverUtil;
import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.sql.bean.SqlCheckInfo;
import com.clustercontrol.sql.ejb.session.MonitorSqlController;

/**
 * SQL監視情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMonitorSql extends ImportMonitorNumeric {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitorSql.class);

	@SuppressWarnings("unchecked")
	private ArrayList monitorList = null;

	private ArrayList<ArrayList<String>> monitorInfo = null;
	
	private List<List<String>> driverList = JdbcDriverUtil.getJdbcDriver();

	/**
	 * SQL監視情報をインポートする。<br>
	 */
	public void importSql() {
		log.info(Messages.getMsg("Sql.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//SQL監視情報のCSVファイルを読み込む
		monitorInfo = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/SQL.csv");

		//監視項目リスト取得
		monitorList = getMonitorList();

		for(ArrayList<String> info : monitorInfo){
			MonitorInfo monitorInfo = createMonitorInfo(info);
			registerMonitorInfo(monitorInfo);
		}

		log.info(Messages.getMsg("Sql.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorInfo(java.util.ArrayList)
	 */
	@Override
	protected MonitorInfo createMonitorInfo(ArrayList<String> info){
		MonitorInfo monitorInfo = new MonitorInfo();

		//監視基本情報
		createMonitorBase(info, monitorInfo);

		//監視条件情報
		createMonitorRule(info.subList(3, 5), monitorInfo);

		//監視情報
		createMonitorCheck(info.subList(5, 10), monitorInfo);

		//監視判定情報
		createMonitorJudgement(info.subList(10, 15), monitorInfo);
		
		//監視通知情報
		createMonitorNotify(info.subList(15, 21), monitorInfo);

		//アプリケーション
		createMonitorApplication(info.subList(21, 22), monitorInfo);

		//メッセージ情報
		createMonitorMessage(info.subList(22, 28), monitorInfo);
		
		//有効/無効
		createMonitorVaild(info.subList(28, 29), monitorInfo);

		return monitorInfo;
	}

	/**
	 * SQL監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorCheck(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		SqlCheckInfo sqlInfo = new SqlCheckInfo();

		//接続先URL
		String url = CheckString.checkLength(info.get(0), 256);
		if(!url.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("connection.url"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		sqlInfo.setConnectionUrl(url);
		//接続先DB
		for(List<String> driver : driverList){
			String name = driver.get(0);
			if(name.equals(info.get(1))){
				sqlInfo.setJdbcDriver(driver.get(1));
				break;
			}
		}
		//ユーザID
		String user = CheckString.checkLength(info.get(2), 64);
		if(!user.equals(info.get(2))){
			String[] args = { com.clustercontrol.util.Messages.getString("user.id"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		sqlInfo.setUser(user);
		//パスワード
		String password = CheckString.checkLength(info.get(3), 64);
		if(!password.equals(info.get(3))){
			String[] args = { com.clustercontrol.util.Messages.getString("password"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		sqlInfo.setPassword(password);
		//SQL
		String sql = CheckString.checkLength(info.get(4), 1024);
		if(!sql.equals(info.get(4))){
			String[] args = { com.clustercontrol.util.Messages.getString("sql.string"), "1024" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		sqlInfo.setQuery(sql);

		monitorInfo.setSqlCheckInfo(sqlInfo);

		monitorInfo.setMonitorType(MonitorTypeConstant.TYPE_NUMERIC);

		super.createMonitorCheck(info, monitorInfo);
	}
	
	/**
	 * メッセージ情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * */
	protected void createMonitorMessage(List<String> info, MonitorInfo monitorInfo) {
		ArrayList<MonitorNumericValueInfo> list = monitorInfo.getNumericValueInfo();
		
		for (MonitorNumericValueInfo valueInfo : list) {
			int index = 0;
			if(valueInfo.getPriority() == PriorityConstant.TYPE_INFO){
				index = 0;
			}
			else if(valueInfo.getPriority() == PriorityConstant.TYPE_WARNING){
				index = 2;
			}
			else if(valueInfo.getPriority() == PriorityConstant.TYPE_CRITICAL){
				index = 4;
			}
			
			//メッセージID
			String id = CheckString.checkLength(info.get(index), 64);
			if(!id.equals(info.get(index))){
				String[] args = { com.clustercontrol.util.Messages.getString("message.id"), "64" };
				log.warn(Messages.getMsg("ImportTool.Common.2", args));
			}
			valueInfo.setMessageId(id);
			//メッセージ
			String message = CheckString.checkLength(info.get(index+1), 256);
			if(!message.equals(info.get(index+1))){
				String[] args = { com.clustercontrol.util.Messages.getString("message"), "256" };
				log.warn(Messages.getMsg("ImportTool.Common.2", args));
			}
			valueInfo.setMessage(message);
		}
	}

	/**
	 * 監視情報を登録する。<br>
	 * 
	 * @param info 監視情報オブジェクト
	 */
	protected void registerMonitorInfo(MonitorInfo info) {
		MonitorSqlController sql = EjbConnectionManager.getConnectionManager().getMonitorSqlController();

		try {
			@SuppressWarnings("unchecked")
			MonitorInfo list = checkMonitorList(info.getMonitorId(), monitorList);
			if(list instanceof MonitorInfo){
				info.setRegDate(list.getRegDate());
				info.setRegUser(list.getRegUser());
				
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				sql.modifySql(info);
			}
			else{
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("add") + ")");

				sql.addSql(info);
			}
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getMonitorList() {

		MonitorSqlController sql = EjbConnectionManager.getConnectionManager().getMonitorSqlController();

		ArrayList records = null;
		try {
			records = sql.getSqlList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}