/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.test.logtransfer;

import java.rmi.RemoteException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;
import jp.co.intellilink.hinemos.test.util.StringListComparator;
import jp.co.intellilink.hinemos.test.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.ejb.session.LogTransferController;
import com.clustercontrol.notify.monitor.ejb.entity.EventLog;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * ログ転送結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultLogTransfer {
	private Date fromDate;
	
	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultLogTransfer.class);
	
	public CheckResultLogTransfer(Date fromDate){
		this.fromDate = fromDate;
	}
	
    /**
     * 転送結果をチェックしCSVフォーマット文字列を作成する。<br>
     */
    public void checkResult() {
    	log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.LogTransfer.Start"));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//通知IDリスト取得
    	ArrayList idList = getLogTransferIdList();
    	
    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(0)};
				log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.TransferID", args));
				
				LogTransferInfo info = getLogTransfer((String)line.get(0));
				csvOutput.addAll(createCheckResult(info));
			}
			
			//ファイルパスでソート
			Collections.sort(csvOutput, new StringListComparator(2));
			//ファシリティIDでソート
			Collections.sort(csvOutput, new StringListComparator(1));
			//転送IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Check.Output.Dir") + "/LOGTRANSFER_" + format.format(new Date()) + ".csv", 
				csvOutput);
		
		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.LogTransfer.End"));
    }
    
	/**
	 * 転送結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param transferInfo 転送情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(LogTransferInfo transferInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
        //監視結果のファシリティIDリストを取得
		facilityIdList.addAll(getNodeList(transferInfo.getFacilityId()));
		
		for(String facilityId : facilityIdList){
			for(LogTransferFileInfo fileInfo : transferInfo.getFileInfo()){
				ArrayList<String> csvLine = new ArrayList<String>();
				//転送ID
				csvLine.add(transferInfo.getTransferId());
				//ファシリティID
				csvLine.add(facilityId);
				//ファイルパス
				csvLine.add(fileInfo.getFilePath());
				
				//イベントチェック（重要度別）
				csvLine.addAll(checkEvent(transferInfo, facilityId, fileInfo));
				
				csvFormat.add(csvLine);

				break;
			}

		}
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param transferInfo 転送情報
	 * @param facilityId ファシリティID
	 * @param fileInfo 転送ファイル情報
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(LogTransferInfo transferInfo, 
			String facilityId, LogTransferFileInfo fileInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";
		
		//監視が無効の場合
		if(transferInfo.getValid() == YesNoConstant.TYPE_NO || 
				fileInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(info);
			csvFormat.add(warning);
			csvFormat.add(critical);
			csvFormat.add(unknown);
			return csvFormat;
		}
		
		//通知
		info = checkEventByPriority(transferInfo, facilityId, fileInfo.getFilePath(), PriorityConstant.TYPE_INFO, fromDate);
		//警告
		warning = checkEventByPriority(transferInfo, facilityId, fileInfo.getFilePath(), PriorityConstant.TYPE_WARNING, fromDate);
		//危険
		critical = checkEventByPriority(transferInfo, facilityId, fileInfo.getFilePath(), PriorityConstant.TYPE_CRITICAL, fromDate);
		//不明
		unknown = checkEventByPriority(transferInfo, facilityId, fileInfo.getFilePath(), PriorityConstant.TYPE_UNKNOWN, fromDate);
		
		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param transferInfo 転送情報
	 * @param facilityId ファシリティID
	 * @param filePath ファイルパス
	 * @param priority 重要度
	 * @param fromDate 出力日時（開始）
	 * @return CSVフォーマット文字列
	 */
	protected String checkEventByPriority(LogTransferInfo transferInfo, String facilityId, String filePath, int priority, Date fromDate) {
		String result = "";

		Collection ct = 
			EjbConnectionManager.getConnectionManager().getEventByPriority(
					facilityId, priority, fromDate);
		Iterator itr = ct.iterator();
		while (itr.hasNext()) {
			EventLog event = (EventLog) itr.next();
			try {
				String orgMessage = event.getMessageOrg();
				if(orgMessage instanceof String && 
						orgMessage.matches(".*hinemos_log_agent\\(" + filePath + "\\).*")){
					result = DateFormat.getDateTimeInstance().format(event.getGenerationDate());
					break;
				}
			} catch (RemoteException e) {
				log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
				System.exit(14);
			}
		}
		
		return result;
	}

	/**
	 * ノードのファシリティIDを取得する。<br>
	 */
    protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
	
    /**
     * ログ転送情報を取得する。<br>
     * 
     * @param transferId 取得対象の転送ID
     * @return ログ転送情報
     */
    protected LogTransferInfo getLogTransfer(String transferId) {

    	LogTransferController transfer = EjbConnectionManager.getConnectionManager().getLogTransferController();

    	LogTransferInfo info = null;
        try {
        	info = transfer.getLogTransfer(transferId);
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
        return info;
    }
    
    /**
     * 転送IDリストを取得する。<br>
     * 
     * @return 転送ID一覧
     */
    protected ArrayList getLogTransferIdList() {

    	LogTransferController transfer = EjbConnectionManager.getConnectionManager().getLogTransferController();

        ArrayList records = null;
        try {
            records = transfer.getLogTransferList();
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}