/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.test.job;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;
import jp.co.intellilink.hinemos.test.util.StringListComparator;
import jp.co.intellilink.hinemos.test.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Schedule;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.jobmanagement.bean.JobHistory;
import com.clustercontrol.jobmanagement.bean.JobHistoryFilter;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.bean.ScheduleTableDefine;
import com.clustercontrol.jobmanagement.ejb.session.JobController;

/**
 * ジョブのスケジュール実行結果をチェックするクラス<br>
 * 
 * @version 1.1.0
 * @since 1.1.0
 */
public class CheckResultSchedule {
	private Date fromDate;
	private Date toDate;

	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultSchedule.class);

	public CheckResultSchedule(Date fromDate, Date toDate){
		this.fromDate = fromDate;
		this.toDate = toDate;
	}

	/**
	 * ジョブのスケジュール実行結果をチェックしCSVフォーマット文字列を作成する。<br>
	 */
	@SuppressWarnings("unchecked")
	public void checkResult() {
		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Job.Schedule.Start"));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//スケジュール一覧取得
		ArrayList list = getScheduleList();
		if(list instanceof ArrayList){
			Iterator itr = list.iterator();
			while(itr.hasNext()){
				JobSchedule line = (JobSchedule)itr.next();

				Object[] args = {line.getId()};
				log.info(Messages.getMsg("TestTool.ScheduleID", args));
				
				csvOutput.add(createCheckResult(line));
			}

			//スケジュールIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
		}

		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Check.Output.Dir") + "/JOB_SCHEDULE_" + format.format(new Date()) + ".csv", 
				csvOutput);

		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Job.Schedule.End"));
	}

	/**
	 * ジョブのスケジュール実行結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param info スケジュール情報
	 * @return CSVフォーマット文字列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> createCheckResult(JobSchedule info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		//スケジュールID
		csvFormat.add(info.getId());

		//スケジュール名設定
		csvFormat.add(info.getName());

		//ジョブID設定
		csvFormat.add(info.getJobId());

		//ジョブ名設定
		csvFormat.add(info.getJobName());

		//スケジュール設定
		csvFormat.add(createSchedule(info.getSchedule()));

		JobHistoryList jobHistoryList = getSessionList(info.getJobId());
		if(jobHistoryList.getList() instanceof ArrayList){
			Iterator itr = jobHistoryList.getList().iterator();
			while(itr.hasNext()){
				JobHistory line = (JobHistory)itr.next();
				String ｓtartTime = "";
				String EndTime = "";
				if(line.getStartDate() instanceof Date){
					ｓtartTime = DateFormat.getDateTimeInstance().format(line.getStartDate());
				}
				if(line.getEndDate() instanceof Date){
					EndTime = DateFormat.getDateTimeInstance().format(line.getEndDate());
				}

				//開始日時
				csvFormat.add(ｓtartTime);

				//終了日時
				csvFormat.add(EndTime);

				break;
			}
		}

		return csvFormat;
	}

	/**
	 * スケジュール情報を文字列に変換する。<br>
	 * 
	 * @param schedule スケジュール情報
	 * @return 文字列
	 */
	protected String createSchedule(Schedule schedule) {
		String scheduleString = "";

		if (schedule.getType() == ScheduleConstant.TYPE_EVERY_YEAR) {
			SimpleDateFormat formatter = new SimpleDateFormat("MM/dd HH:mm");
			if (schedule.getDate() instanceof Date) {
				scheduleString = formatter
				.format((Date) schedule.getDate());
			}
		} else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_MONTH) {
			SimpleDateFormat formatter = new SimpleDateFormat("dd'"
					+ com.clustercontrol.util.Messages.getString("monthday") + "' HH:mm");
			if (schedule.getDate() instanceof Date) {
				scheduleString = formatter
				.format((Date) schedule.getDate());
			}
		} else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_DAY) {
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm");
			if (schedule.getDate() instanceof Date) {
				scheduleString = formatter
				.format((Date) schedule.getDate());
			}
		} else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_HOUR) {
			SimpleDateFormat formatter = new SimpleDateFormat("mm'"
					+ com.clustercontrol.util.Messages.getString("minute") + "'");
			if (schedule.getDate() instanceof Date) {
				scheduleString = formatter
				.format((Date) schedule.getDate());
			}
		} else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK) {
			SimpleDateFormat formatter = new SimpleDateFormat(" HH:mm");
			if (schedule.getDate() instanceof Date) {
				scheduleString = DayOfWeekConstant.typeToString(schedule
						.getDayOfWeek())
						+ formatter.format((Date) schedule.getDate());
			}
		}

		return scheduleString;
	}

	/**
	 * 配列から文字列を取得する。<br>
	 * 
	 * @param values 配列
	 * @param index インデックス
	 * @return 文字列
	 */
	@SuppressWarnings("unchecked")
	public String getStringValue(ArrayList values, int index) {
		if(values.size() > index - 1 && values.get(index) instanceof String)
			return (String)values.get(index);
		else
			return "";
	}

	/**
	 * ジョブセッション一覧を取得する。<br>
	 * 
	 * @return ジョブセッション一覧
	 */
	protected JobHistoryList getSessionList(String jobId) {

		JobController job = EjbConnectionManager.getConnectionManager().getJobController();

		JobHistoryList info = null;
		try {
			JobHistoryFilter filter = new JobHistoryFilter();

			//ジョブIDを設定
			if(jobId instanceof String){
				filter.setJobId(jobId);
			}

			//開始・再実行日時（開始）を設定
			if(fromDate instanceof Date){
				filter.setStartFromDate(fromDate);
			}

			//開始・再実行日時（終了）を設定
			if(toDate instanceof Date){
				filter.setStartToDate(toDate);
			}

			int maxlist = Integer.parseInt(Config.getConfig("history.job.max.list"));

			info = job.getHistoryList(filter, maxlist);
		} catch (Exception e) {
			log.error(Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return info;
	}

	/**
	 * スケジュール一覧を取得する。<br>
	 * 
	 * @return スケジュール一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getScheduleList() {

		JobController job = EjbConnectionManager.getConnectionManager().getJobController();

		ArrayList list = null;
		try {
			list = job.getScheduleList();
		} catch (Exception e) {
			log.error(Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return list;
	}
}