/*

Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.test.maintenance;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;
import jp.co.intellilink.hinemos.test.util.StringListComparator;
import jp.co.intellilink.hinemos.test.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.maintenance.bean.MaintenanceInfoDataImp;
import com.clustercontrol.maintenance.ejb.session.MaintenanceController;
import com.clustercontrol.monitor.bean.ReportEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoData;

/**
 * メンテナンス結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultMaintenamce {
	private Date fromDate;
	private Date toDate;

	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultMaintenamce.class);

	public CheckResultMaintenamce(Date fromDate, Date toDate){
		this.fromDate = fromDate;
		this.toDate = toDate;
	}

	/**
	 * メンテナンス結果をチェックしCSVフォーマット文字列を作成する。<br>
	 */
	public void checkResult() {
		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Maintenance.Start"));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//メンテナンスIDリスト取得
		ArrayList idList = getMaintenanceIdList();

		if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();

				Object[] args = {(String)line.get(0)};
				log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.MaintenanceID", args));

				MaintenanceInfoDataImp info = getMaintenance((String)line.get(0));
				csvOutput.addAll(createCheckResult(info));
			}

			//メンテナンスIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
		}

		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Check.Output.Dir") + "/MAINTENANCE_" + format.format(new Date()) + ".csv", 
				csvOutput);

		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Maintenance.End"));
	}

	/**
	 * メンテナンス結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param maintenanceInfo メンテナンス情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(MaintenanceInfoDataImp maintenanceInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();

		ArrayList<String> csvLine = new ArrayList<String>();
		//メンテナンスID
		csvLine.add(maintenanceInfo.getMaintenance_id());

		//ステータスチェック
		csvLine.addAll(checkStatus(maintenanceInfo, "INTERNAL"));

		//イベントチェック（重要度別）
		csvLine.addAll(checkEvent(maintenanceInfo, "INTERNAL"));

		csvFormat.add(csvLine);

		return csvFormat;
	}

	/**
	 * ステータス出力結果をチェックする。<br>
	 * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkStatus(MaintenanceInfoDataImp maintenanceInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String result = "-";

		//監視が無効の場合
		if(maintenanceInfo.getValid_flg() == YesNoConstant.TYPE_NO){
			csvFormat.add(result);
			return csvFormat;
		}

		String notifyId = null;
		if(maintenanceInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = maintenanceInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}

		if(notifyId != null){
			result = "NG";

			Collection ct = 
				EjbConnectionManager.getConnectionManager().getStatus(facilityId);
			Iterator itr = ct.iterator();
			while (itr.hasNext()) {
				StatusInfoData status = (StatusInfoData) itr.next();
				if(status.getMonitorId().compareTo(maintenanceInfo.getMaintenance_id()) == 0){
					result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
					if(status.getExpirationFlg() != null && status.getExpirationFlg().intValue() != StatusExpirationConstant.TYPE_EXPIRATION)
						result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
					break;
				}
			}
		}
		else{
			result = "-";
		}

		csvFormat.add(result);

		return csvFormat;
	}

	/**
	 * イベント出力結果をチェックする。<br>
	 * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(MaintenanceInfoDataImp maintenanceInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";

		//監視が無効の場合
		if(maintenanceInfo.getValid_flg() == YesNoConstant.TYPE_NO){
			csvFormat.add(info);
			csvFormat.add(warning);
			csvFormat.add(critical);
			csvFormat.add(unknown);
			return csvFormat;
		}

		String notifyId = null;
		if(maintenanceInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = maintenanceInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}

		if(notifyId != null){
			//通知
			info = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_INFO, fromDate, toDate);
			//警告
			warning = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_WARNING, fromDate, toDate);
			//危険
			critical = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_CRITICAL, fromDate, toDate);
			//不明
			unknown = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_UNKNOWN, fromDate, toDate);
		}

		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);

		return csvFormat;
	}

	/**
	 * イベント出力結果をチェックする。<br>
	 * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @param priority 重要度
	 * @param fromDate 出力日時（開始）
	 * @return CSVフォーマット文字列
	 */
	protected String checkEventByPriority(MaintenanceInfoDataImp maintenanceInfo, String facilityId, int priority, Date fromDate, Date toDate) {
		String result = "";

		ArrayList<ReportEventInfo> eventList = 
			EjbConnectionManager.getConnectionManager().getEvent(facilityId, fromDate, toDate);
		Date newDate = null;
		for (ReportEventInfo event : eventList) {
			String monitorId = event.getMonitorId();
			int eventPriority = PriorityConstant.stringToType(event.getPriority());
			if(monitorId.compareTo(maintenanceInfo.getMaintenance_id()) == 0 && 
					eventPriority == priority){
				if(newDate instanceof Date){
					if(newDate.before(event.getGenerationDate()))
						newDate = event.getGenerationDate();
				}
				else{
					newDate = event.getGenerationDate();
				}
			}
		}
		if (newDate instanceof Date) {
			result = DateFormat.getDateTimeInstance().format(newDate);
		}

		return result;
	}

	/**
	 * 通知情報を取得する。<br>
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 */
	protected NotifyInfo getNotify(String notifyId) {
		return EjbConnectionManager.getConnectionManager().getNotifyId(notifyId);
	}

	/**
	 * メンテナンス情報を取得する。<br>
	 * 
	 * @param maintenanceId 取得対象のメンテナンスID
	 * @return メンテナンス情報
	 */
	protected MaintenanceInfoDataImp getMaintenance(String maintenanceId) {

		MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

		MaintenanceInfoDataImp info = null;
		try {
			info = maintenance.getMaintenanceInfo(maintenanceId);
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return info;
	}

	/**
	 * スケジュール情報を取得する。<br>
	 * 
	 * @param maintenanceId 取得対象のメンテナンスID
	 * @return スケジュール情報
	 */
	protected ArrayList getSchedule(String maintenanceId) {

		MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

		ArrayList info = null;
		try {
			info = maintenance.getSchedule(maintenanceId);
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return info;
	}

	/**
	 * メンテナンスIDリストを取得する。<br>
	 * 
	 * @return メンテナンスID一覧
	 */
	protected ArrayList getMaintenanceIdList() {

		MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

		ArrayList records = null;
		try {
			records = maintenance.getMaintenanceList();
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}