(ns test.math
  (:use clojure.contrib.test-is))

(require 'src.math)
(alias 'mt 'src.math)

(deftest test-swap
  (is (= '(d b c a)
         (mt/swap 0 3 '(a b c d))))
  (is (= '(b a c d)
         (mt/swap -3 0 '(a b c d))))
  (is (= '(a b c d)
         (mt/swap 1 1 '(a b c d)))))

(deftest test-search-non-zero
  (let [m [[0 0 1 2]
           [0 1 0 1]
           [0 0 0 3]]]
    (is (= (mt/search-non-zero m 0)
           [1 1]))
    (is (= (mt/search-non-zero m 2)
           nil))))

(deftest test-sweep1
  (let [m [[2 1 1 0]
           [1 2 4 6]
           [4 0 3 2]]]
    (is (= (mt/sweep1 m 0)
           [[1 1/2 1/2 0]
            [0 3/2 7/2 6]
	    [0  -2   1 2]]))
    (is (= (mt/sweep1 m 1)
           [[3/2 0 -1 -3]
            [1/2 1  2  3]
            [  4 0  3  2]]))))

(deftest test-lin-solve
  (let [m [[2 1 1 5]
           [1 2 1 4]
           [1 1 2 3]]]
    (is (= (mt/lin-solve m)
           [[1 0 0 2]
            [0 1 0 1]
            [0 0 1 0]
            ]))))

(deftest test-m*v
  (is (= (mt/m*v [[1 2] [3 4]] [5 6])
         [17 39]))
  (is (= (mt/m*v [[1 0] [0 1]] [3 4])
         [3 4]))
  (is (= (mt/m*v [[1 0 2] [2 1 3]] [1 1 2])
         [5 9])))

(deftest test-s*m
  (is (= (mt/s*m 2 [[1 2] [3 4]])
         [[2 4] [6 8]])))

(deftest test-m*m
  (is (= (mt/m*m [[2]] [[3]])
         [[6]]))
  (is (= (mt/m*m [[1 2 3]]
                 [[4]
                  [5]
                  [6]])
         [[32]]))
  (is (= (mt/m*m [[1]
                  [2]
                  [3]]
                 [[4 5 6]])
         [[ 4  5  6]
          [ 8 10 12]
          [12 15 18]]))
  (is (= (mt/m*m [[1 2]
                  [2 2]]
                 [[2 3]
                  [3 4]])
         [[ 8 11]
          [10 14]])))

(deftest test-m-m
  (is (= (mt/m-m [[1 2] [3 4]] [[5 6] [7 8]])
         [[-4 -4] [-4 -4]]))
  (is (= (mt/m-m [[1 0] [0 1]] [[1 2] [3 4]])
         [[0 -2] [-3 -3]])))

(deftest test-i-mat
  (is (= (mt/i-mat 0) []))
  (is (= (mt/i-mat 1) [[1]]))
  (is (= (mt/i-mat 2) [[1 0] [0 1]]))
  (is (= (mt/i-mat 3)
         [[1 0 0]
          [0 1 0]
          [0 0 1]])))

(deftest test-inv-mat
  (is (= (mt/inv-mat [[1 0 0]
                      [0 1 0]
                      [0 0 1]])
         [[1 0 0]
          [0 1 0]
          [0 0 1]
          ]))
  (is (= (mt/inv-mat [[1 2]
                      [3 4]])
         [[ -2    1]
	  [3/2 -1/2]
          ])))

(deftest test-tref
  (is (= (mt/tref [0 1 2])
         [0 1 2]
         ))
  (is (= (mt/tref [0 [1 2] [3 4]]
                  2)
         [3 4]))
  (is (= (mt/tref [0 [1 2] [3 4]]
                  2 1)
         4)))

(deftest test-tfassoc
  (is (= (mt/tfassoc 2 [] #(* % 3))
         6))
  (is (= (mt/tfassoc [0 [1 2] [3 4]]
                     [2 1]
                     #(+ % 4))
         [0 [1 2] [3 8]]
         )))

(deftest test-t+2
  (is (= (mt/t+2 1 2) 3))
  (is (= (mt/t+2 [1 2 3]
                 [4 5 6])
         [5 7 9]
         ))
  (is (= (mt/t+2 [0 [1 2] [3 4]]
                 [5 [6 7] [8 9]])
         [5 [7 9] [11 13]]
         )))

(deftest test-t+
  (is (= (mt/t+ [0 1 2])
         [0 1 2]
         ))
  (is (= (mt/t+ 1 2) 3))
  (is (= (mt/t+ [1 2 3]
                [4 5 6])
         [5 7 9]
         ))
  (is (= (mt/t+ [1 2 3]
                [4 5 6]
                [7 8 9])
         [12 15 18]
         )))

(deftest test-t-2
  (is (= (mt/t-2 1 3) -2))
  (is (= (mt/t-2 [1  2 3]
                 [4 -4 5])
         [-3 6 -2]
         ))
  (is (= (mt/t-2 [ 0 [ 1  2] [ 3  4]]
                 [-1 [-3 -5] [-7 -9]])
         [1 [4 7] [10 13]]
         )))

(deftest test-t-
  (is (= (mt/t- [3 [1 2]])
         [-3 [-1 -2]]
         ))
  (is (= (mt/t- 1 -3) 4))
  (is (= (mt/t- [1 2 3]
                [3 5 7])
         [-2 -3 -4]
         ))
  (is (= (mt/t- [1 2 3]
                [3 5 7]
                [2 5 8])
         [-4 -8 -12]
         )))

(deftest test-s*t
  (is (= (mt/s*t 2 [3]) [6]))
  (is (= (mt/s*t 2 [1 3]) [2 6]))
  (is (= (mt/s*t 2 [[1 2] 3]) [[2 4] 6]))
  )

(deftest test-diff-num
  (is (= (mt/diff-num (fn [x] (* 4 x))
                      [] 3 0.5)
         4
         ))
  (is (= (mt/diff-num (fn [[x y]]
                        [(* 2 x y)
                         (+ (* 5 x) (* 3 y))
                         ])
                      [0] [1 2] 0.5)
         [4 5]
         ))
  (is (= (mt/diff-num (fn [[x y]]
                        [(* 2 x y)
                         (+ (* 5 x) (* 3 y))
                         ])
                      [] [1 2] 0.5)
         [[4 5]
          [2 3]
          ]))
  (is (= (mt/diff-num (fn [[x [y]]]
                        [(* 2 x y)
                         [(+ (* 5 x) (* 3 y))]
                         ])
                      [] [1 [2]] 0.5)
         [[4 [5]]
          [[2 [3]]]
          ])))

(deftest test-L->accer
  (let [faccer (mt/L->accer
                 0.5 0.5 0.5
                 (fn [[t [x y] [vx vy]]]
                   (- (+ (* vx vx) (* 2 vy vy))
                      (+ (* x x) (* 2 y y))))
                 (fn [_] 0))]
    (is (= (faccer [0 [1 2] [2 1]])
           [-1.0 -2.0]))))

(deftest test-eular-implicit
  (let [f (fn [[x vx]] [vx (- x)])
        [x-end vx-end]
        (loop [i 100 val [1 0]]
          (if (<= i 0)
            val
            (recur (dec i)
                   (mt/eular-implicit f val 0.5 0.5)
                   )))]
    (is (< (Math/abs x-end) 0.01))
    (is (< (Math/abs vx-end) 0.01))
    ))

(run-tests)
