//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		test_color.cpp
 * @brief		FndColor eXg t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define _FILE_test_color_CPP_

//======================================================================
// include
#include "unit/gt/gt_inchead.h"
#include "fnd/image/FndColor.h"
#include "c++0x/cpp0x_type_traits.hpp"
#include "iris_using.h"

TEST(FndColorTest, Construct)
{
	for( int i=0; i < 10000; ++i )
	{
		int R = rand() & 0xFF;
		int G = rand() & 0xFF;
		int B = rand() & 0xFF;
		COLORREF color = RGB(R, G, B);
		CCOLORREF clrref = color;
		ASSERT_EQ( R, clrref.r() );
		ASSERT_EQ( G, clrref.g() );
		ASSERT_EQ( B, clrref.b() );
		ASSERT_EQ( 0, clrref.a() );
	}
}

TEST(FndColorTest, Copy)
{
	COLORREF color = RGB(0xFF, 0xCC, 0x66);
	CCOLORREF clr1 = color;
	CBGRA8888 clr2 = clr1;
	CRGBA8888 clr3 = clr2;

	IrisFColor fclr1 = (IrisFColor)clr1;
	IrisFColor fclr2 = (IrisFColor)clr2;
	IrisFColor fclr3 = (IrisFColor)clr3;
	ASSERT_FLOAT_EQ( 1.0f, fclr1.r );
	ASSERT_FLOAT_EQ( 0.8f, fclr1.g );
	ASSERT_FLOAT_EQ( 0.4f, fclr1.b );
	ASSERT_FLOAT_EQ( 1.0f, fclr1.a );

	ASSERT_FLOAT_EQ( 1.0f, fclr2.r );
	ASSERT_FLOAT_EQ( 0.8f, fclr2.g );
	ASSERT_FLOAT_EQ( 0.4f, fclr2.b );
	ASSERT_FLOAT_EQ( 1.0f, fclr2.a );

	ASSERT_FLOAT_EQ( 1.0f, fclr3.r );
	ASSERT_FLOAT_EQ( 0.8f, fclr3.g );
	ASSERT_FLOAT_EQ( 0.4f, fclr3.b );
	ASSERT_FLOAT_EQ( 1.0f, fclr3.a );
}

typedef testing::Types< CRGBA8888, CCOLORREF, CRGB565, CRGB888 > FormatTypeParams;

template<typename T>
class FndColorTypesTest : public testing::TestWithParam<T>
{
};

TYPED_TEST_CASE(FndColorTypesTest, FormatTypeParams);

TYPED_TEST(FndColorTypesTest, construct)
{
	ParamType color;
	ASSERT_EQ(0, color.r());
	ASSERT_EQ(0, color.g());
	ASSERT_EQ(0, color.b());
	ASSERT_EQ(0, color.a());
}

TYPED_TEST(FndColorTypesTest, size)
{
	ASSERT_EQ(sizeof(ParamType::value_type), sizeof(ParamType));
}

TYPED_TEST(FndColorTypesTest, white)
{
	ParamType color(CRGBA8888(0xFFFFFFFFu));
	IrisFColor frgb = color;
	ASSERT_FLOAT_EQ(1.0f, frgb.r);
	ASSERT_FLOAT_EQ(1.0f, frgb.g);
	ASSERT_FLOAT_EQ(1.0f, frgb.b);
	ASSERT_FLOAT_EQ(1.0f, frgb.a);

	CRGBA8888 rgba8888 = color;
	ASSERT_EQ(0xFF, rgba8888.r());
	ASSERT_EQ(0xFF, rgba8888.g());
	ASSERT_EQ(0xFF, rgba8888.b());
	ASSERT_EQ(0xFF, rgba8888.a());
}

TYPED_TEST(FndColorTypesTest, convert)
{
	u8 r = static_cast<u8>((rand()>>4) & 0xFF);
	u8 g = static_cast<u8>((rand()>>4) & 0xFF);
	u8 b = static_cast<u8>((rand()>>4) & 0xFF);
	u8 a = static_cast<u8>((rand()>>4) & 0xFF);
	CRGBA8888 rgba( r, g, b, a );
	ParamType color(rgba);
	CRGBA8888 rgba_dst = color;

	// xႤ
	int ar = 1<<abs(CRGBA8888::BSZ_R - ParamType::BSZ_R);
	int ag = 1<<abs(CRGBA8888::BSZ_G - ParamType::BSZ_G);
	int ab = 1<<abs(CRGBA8888::BSZ_B - ParamType::BSZ_B);
	int aa = 1<<abs(CRGBA8888::BSZ_A - ParamType::BSZ_A);
	ASSERT_NEAR(rgba.r(), rgba_dst.r(), ar);
	ASSERT_NEAR(rgba.g(), rgba_dst.g(), ag);
	ASSERT_NEAR(rgba.b(), rgba_dst.b(), ab);
	ASSERT_NEAR(rgba.a(), rgba_dst.a(), aa);
}

TYPED_TEST(FndColorTypesTest, convert_component1)
{
	u8 r = static_cast<u8>((rand()>>4) & 0xFF);
	u8 g = static_cast<u8>((rand()>>4) & 0xFF);
	u8 b = static_cast<u8>((rand()>>4) & 0xFF);
	u8 a = static_cast<u8>((rand()>>4) & 0xFF);
	CRGBA8888 rgba( r, g, b, a );
	ParamType color(rgba);
	CRGBA8888 rgba_dst = color;

	ASSERT_EQ(rgba_dst.r(), color.r<u8>());
	ASSERT_EQ(rgba_dst.g(), color.g<u8>());
	ASSERT_EQ(rgba_dst.b(), color.b<u8>());
	ASSERT_EQ(rgba_dst.a(), color.a<u8>());
}

TYPED_TEST(FndColorTypesTest, convert_component2)
{
	u8 r = static_cast<u8>((rand()>>4) & 0xFF);
	u8 g = static_cast<u8>((rand()>>4) & 0xFF);
	u8 b = static_cast<u8>((rand()>>4) & 0xFF);
	u8 a = static_cast<u8>((rand()>>4) & 0xFF);
	CRGBA8888 rgba( r, g, b, a );
	ParamType color(rgba);
	CRGB565 rgba_dst = rgba;

	ASSERT_EQ(rgba_dst.r(), (color.r<u8, 5>()));
	ASSERT_EQ(rgba_dst.g(), (color.g<u8, 6>()));
	ASSERT_EQ(rgba_dst.b(), (color.b<u8, 5>()));
	ASSERT_EQ(rgba_dst.a(), (color.a<u8, 0>()));
}

TYPED_TEST(FndColorTypesTest, convert_from_float)
{
	IrisFColor rgba; 
	rgba.r = ((rand()>>4) & 0xFF) / 255.0f;
	rgba.g = ((rand()>>4) & 0xFF) / 255.0f;
	rgba.b = ((rand()>>4) & 0xFF) / 255.0f;
	rgba.a = ((rand()>>4) & 0xFF) / 255.0f;
	ParamType color(rgba);
	IrisFColor rgba_dst = color;

	// xႤ
	float ar = ParamType::BSZ_R ? ((1<<ParamType::BSZ_R)-1) : 1.0f;
	float ag = ParamType::BSZ_G ? ((1<<ParamType::BSZ_G)-1) : 1.0f;
	float ab = ParamType::BSZ_B ? ((1<<ParamType::BSZ_B)-1) : 1.0f;
	float aa = ParamType::BSZ_A ? ((1<<ParamType::BSZ_A)-1) : 1.0f;
	ASSERT_NEAR(rgba.r, rgba_dst.r, 1.0f/ar);
	ASSERT_NEAR(rgba.g, rgba_dst.g, 1.0f/ag);
	ASSERT_NEAR(rgba.b, rgba_dst.b, 1.0f/ab);
	ASSERT_NEAR(rgba.a, rgba_dst.a, 1.0f/aa);
}

TYPED_TEST(FndColorTypesTest, buffer_read)
{
	/*
	 * ze[uȂ̂ŁAobt@ƂĂ̂܂܎g 
	*/
	const int bufsize = 100;
	ParamType color[bufsize];
	ParamType::value_type buf[bufsize];

	for( int i=0; i < bufsize; ++i )
	{
		u8 r = static_cast<u8>((rand()>>4) & 0xFF);
		u8 g = static_cast<u8>((rand()>>4) & 0xFF);
		u8 b = static_cast<u8>((rand()>>4) & 0xFF);
		u8 a = static_cast<u8>((rand()>>4) & 0xFF);
		CRGBA8888 rgba( r, g, b, a );
		color[i] = rgba;
		buf[i] = color[i];
	}

	ParamType* pc = pointer_cast<ParamType*>(buf);
	for( int i=0; i < bufsize; ++i, ++pc )
	{
		ASSERT_EQ( color[i], *pc );
	}
}

TYPED_TEST(FndColorTypesTest, buffer_write)
{
	/*
	 * ze[uȂ̂ŁAobt@ƂĂ̂܂܎g 
	*/
	const int bufsize = 100;
	ParamType color[bufsize];
	ParamType::value_type buf[bufsize];

	ParamType* pc = pointer_cast<ParamType*>(buf);
	for( int i=0; i < bufsize; ++i, ++pc )
	{
		u8 r = static_cast<u8>((rand()>>4) & 0xFF);
		u8 g = static_cast<u8>((rand()>>4) & 0xFF);
		u8 b = static_cast<u8>((rand()>>4) & 0xFF);
		u8 a = static_cast<u8>((rand()>>4) & 0xFF);
		CRGBA8888 rgba( r, g, b, a );
		color[i] = rgba;
		*pc = rgba;
	}

	for( int i=0; i < bufsize; ++i, ++pc )
	{
		ASSERT_EQ( color[i], buf[i] );
	}
}

TYPED_TEST(FndColorTypesTest, is_pod)
{
	ASSERT_FALSE(cpp0x::is_pod<ParamType>::value);
}

