//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndIEEE.h
 * @brief		IEEE754 t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndIEEE_H_
#define INCG_IRIS_FndIEEE_H_

//======================================================================
// include
#include "../../iris_object.h"
#include "../../iris_debug.h"
#include "../../ml/iml_math.hpp"

namespace iris {
namespace fnd
{

//======================================================================
// declare
template<typename ttTN, int tiSIGN, int tiEXP, int tiFRAC, int tiBIAS>class CIEEE754;

//======================================================================
// typedef
typedef CIEEE754<u32, 1, 8, 23, 127>				CIEEEBinary32;
typedef CIEEE754<u64, 1, 11, 52, 1023>				CIEEEBinary64;


//======================================================================
// class
/**
 * @brief	IEEE 754 _NX
 * @tparam ttTN		= i[鐮^
 * @tparam tiSIGN	= ̃rbg(0 or 1)
 * @tparam tiEXP	= w
 * @tparam tiFRAC	= 
 * @tparam tiBIAS	= w̃oCAX
*/
template<typename ttTN, int tiSIGN, int tiEXP, int tiFRAC, int tiBIAS = ::iml::static_pow<2, tiEXP-1>::value-1 >
class CIEEE754 : public IIrisObject
{
	typedef	ttTN	value_type;
	typedef CIEEE754<ttTN, tiSIGN, tiEXP, tiFRAC, tiBIAS>	_Myt;
	typedef CIEEEBinary32									floatType;
	typedef CIEEEBinary64									doubleType;

	IRIS_STATIC_ASSERT( tiSIGN == 0 || tiSIGN == 1 );
public:
	static const int SIGN = tiSIGN;	//!< ̃rbg
	static const int EXP  = tiEXP;	//!< w̃rbg
	static const int FRAC = tiFRAC;	//!< ̃rbg
	static const int BIAS = tiBIAS;	//!< w̃oCAXl
	static const size_t BITSIZE = SIGN + EXP + FRAC;
	IRIS_STATIC_ASSERT( BITSIZE <= (sizeof(value_type)*8) );

public:
	static const value_type	PINF	= static_cast<value_type>(iml::static_mask<EXP			, FRAC  >::value);		//!< + 
	static const value_type	NINF	= static_cast<value_type>(iml::static_mask<EXP+SIGN		, FRAC  >::value);		//!< - 
	static const value_type	PNAN	= static_cast<value_type>(iml::static_mask<EXP			, FRAC>::value | 0x1);	//!< + NAN
	static const value_type	NNAN	= static_cast<value_type>(iml::static_mask<EXP+SIGN		, FRAC>::value | 0x1);	//!< - NAN
	static const value_type	PQNAN	= static_cast<value_type>(iml::static_mask<EXP+1		, FRAC-1>::value);		//!< + QNAN
	static const value_type	NQNAN	= static_cast<value_type>(iml::static_mask<EXP+SIGN+1	, FRAC-1>::value);		//!< - QNAN

private:
	value_type	m_value;	//!< l

public:
	/**
	 * @name RXgN^
	 * @{
	*/
	CIEEE754(void) : m_value(0) {}
	CIEEE754(float f) : m_value(0) { cast_from_float(f); }
	CIEEE754(const _Myt& rhs) : m_value(rhs.m_value) {}
	template<typename ttXTN, int tiXSIGN, int tiXEXP, int tiXFRAC, int tiXBIAS>
	CIEEE754(const CIEEE754<ttXTN, tiXSIGN, tiXEXP, tiXFRAC, tiXBIAS>& ieee) : m_value(0) { cast_from_ieee(ieee); }
	/**
	 * @}
	*/
public:
	/**
	 * @name	operator
	 * @{
	*/
	_Myt& operator = (float rhs)	{ cast_from_float(rhs); return *this; }
	template<typename _XTN, int _XSIGN, int _XEXP, int _XFRAC, int _XBIAS>
	_Myt& operator = (const CIEEE754<_XTN, _XSIGN, _XEXP, _XFRAC, _XBIAS>& ieee) { cast_from_ieee(ieee); return *this; }

	operator float (void)	const		{ return cast_to_float(); }
	operator double (void)	const		{ return cast_to_double(); }
	/**
	 * @}
	*/

public:
	//! l̎擾
	value_type&		value(void)			{ return m_value; }
	value_type		value(void)	const	{ return m_value; }

private:
	// LXg
	void	cast_from_float(float f)
	{
		IrisFInt fi = f;

		value_type exp = (fi.iv >> floatType::FRAC) & iml::static_mask<floatType::EXP>::value;
		if( exp == iml::static_mask<floatType::EXP>::value )
		{
			exp = iml::static_mask<EXP>::value;
		}
		else if( exp != 0 )
		{
			exp = exp - floatType::BIAS + BIAS;
			if( exp >= iml::static_lshift<1, EXP>::value )
			{
				// INF
				m_value = (((exp>>(sizeof(exp)-1))&0x1)<<EXP) | iml::static_mask<EXP>::value;
				m_value <<= FRAC;
				return;
			}
		}
		m_value  = (value_type)(iml::signed_rshift<(floatType::FRAC+floatType::EXP)-(FRAC+EXP)>::shift(fi.iv) & iml::static_mask<1, EXP+FRAC>::value);
		m_value |= exp << FRAC;
		m_value |= iml::signed_rshift<floatType::FRAC-FRAC>::shift( static_cast<IrisU32>(fi.iv & iml::static_mask<floatType::FRAC>::value) );
	}
	template<typename XTN_, int XSIGN_, int XEXP_, int XFRAC_, int XBIAS_>
	void	cast_from_ieee(const CIEEE754<XTN_, XSIGN_, XEXP_, XFRAC_, XBIAS_>& ieee)
	{
		XTN_ value = ieee.value();
		value_type exp = static_cast<value_type>((value >> XFRAC_) & iml::static_mask<XEXP_>::value);
		if( exp == iml::static_mask<XEXP_>::value )
		{
			exp = iml::static_mask<EXP>::value;
		}
		else if( exp != 0 )
		{
			exp = exp - ieee.BIAS + BIAS;
			if( exp >= static_cast<value_type>(iml::static_lshift<1, EXP>::value) )
			{
				// INF
				m_value = (((exp>>(sizeof(exp)-1))&0x1)<<EXP) | iml::static_mask<EXP>::value;
				m_value <<= FRAC;
				return;
			}
		}
		m_value  = static_cast<value_type>(iml::signed_rshift<(XFRAC_+XEXP_)-(FRAC+EXP)>::shift(static_cast<value_type>(value)) & iml::static_mask<1, EXP+FRAC>::value);
		m_value |= exp << FRAC;
		m_value |= iml::signed_rshift<XFRAC_-FRAC>::shift( static_cast<value_type>(value & iml::static_mask<XFRAC_>::value) );
	}

	float	cast_to_float(void)	const
	{
		floatType ieee(*this);
		IrisFInt fi = ieee.value();
		return fi.fv;
	}
	double	cast_to_double(void) const
	{
		doubleType ieee(*this);
		IrisF64Int fi = ieee.value();
		return fi.fv;
	}

public:
	// 
	value_type		sign(void)				{ return (m_value >> (EXP+FRAC)) & iml::static_mask<SIGN>::value; }
	void			sign(value_type s)		{ m_value &= iml::static_mask<EXP+FRAC>::value; m_value |= ((s&1) << (EXP+FRAC)); }

	// w
	value_type		exp(void)				{ return (m_value >> FRAC) & iml::static_mask<EXP>::value; }
	void			exp(value_type e)		{ m_value &= ~iml::static_mask<EXP, FRAC>::value; m_value |= (e&iml::static_mask<EXP>::value) << FRAC; }

	// 
	value_type		frac(void)				{ return m_value & iml::static_mask<FRAC>::value; }
	void			frac(value_type f)		{ m_value &= ~iml::static_mask<FRAC>::value; m_value |= f&iml::static_mask<FRAC>::value; }
};

}	// end of namespace fnd
}	// end of namespace iris

#endif
