//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndBitmap.cpp
 * @brief		bitmapNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndBitmap_CPP_

//======================================================================
// include
#include "FndBitmap.h"
#include "../memory/FndMemory.h"
#include "../io/FndFile.h"
#include <string.h>
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CBitmap::CBitmap(void)
: m_bAllocated(false)
{
	m_bmp.file = nullptr;
	m_bmp.info = nullptr;
	m_bmp.data = nullptr;
	m_bmp.pixel = nullptr;
	m_bmp.pallete = nullptr;
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CBitmap::~CBitmap(void)
{
	Release();
}

/**********************************************************************//**
 *
 * t@C̐`FbN
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CBitmap::CheckFile(void)
{
	return CheckFile(m_bmp.file, m_bmp.info);
}

/**********************************************************************//**
 *
 * t@C̐`FbN
 *
 -----------------------------------------------------------------------
 * @param [in]	bfh	= t@Cwb_
 * @param [in]	bih	= wb_
 * @return	^Ul
*//***********************************************************************/
bool CBitmap::CheckFile(LPBMPFILEHEADER bfh, LPBMPINFOHEADER bih)
{
	if( bfh == nullptr ) return false;
	if( bih == nullptr ) return false;

	switch( bih->biCompression )
	{
	case 0:
		break;
	case 1:	// 4bit OXk
		if( bih->biBitCount != 4 )	return false;
		//break;
	case 2:	// 8bit OXk
		if( bih->biBitCount != 8 )	return false;
		//break;
	case 3:	// rbgtB[h
		if( bih->biBitCount != 32 ) return false;
		//break;
	default:
		IRIS_WARNING("Compressed Bitmap is not supported.");
		return false;
	}

	if( bih->biSizeImage == 0 )
	{
		// ̃tB[hɏo͂Ȃy
		bih->biSizeImage = GetImageSize(bih);
	}
	else
	{
		// t@Cƃwb_񂪈vȂ
		if( bih->biSizeImage != GetImageSize(bih) )
		{
#if 0	// ꉞǂݍ݂͉\Ȃ̂ŁAfalseԂȂ悤ɂ
			return false;
#endif
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * t@Cǂݍ
 *
 -----------------------------------------------------------------------
 * @param [in]	lpFileName	= t@C
 * @return	
*//***********************************************************************/
template<>
bool CBitmap::ReadFile<CHAR>(LPCSTR lpFileName)
{
	return ReadFileA(lpFileName);
}
template<>
bool CBitmap::ReadFile<WCHAR>(LPCWSTR lpFileName)
{
	return ReadFileW(lpFileName);
}
/// CBitmap::ReadFile Q
bool CBitmap::ReadFileA(LPCSTR  lpFileName)
{
	CFile file;
	if( !file.OpenA(lpFileName, FOPEN_READ) ) return false;
	// t@Cwb_̓ǂݍ
	BMPFILEHEADER fh;
	file.Read(&fh, sizeof(BMPFILEHEADER));
	if( fh.bfType != BITMAP_FILEHEAD_TYPE ) return false;
	BMPINFOHEADER ih;
	// infowb_̓ǂݍ
	file.Read(&ih, sizeof(BMPINFOHEADER));
	if( !CheckFile(&fh, &ih) ) return false;

	u32 size = ih.biSizeImage + BMP_HEADSIZE;
	if( !Alloc(size) ) return false;

	*m_bmp.file = fh;
	*m_bmp.info = ih;
	file.Read(m_bmp.data, fh.bfSize-BMP_HEADSIZE);
	u32 ofs = m_bmp.file->bfOffBits - BMP_HEADSIZE;
	m_bmp.pixel = m_bmp.data + ofs;
	m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(ofs == 0 ? nullptr : m_bmp.data);
	return true;
}
/// CBitmap::ReadFile Q
bool CBitmap::ReadFileW(LPCWSTR lpFileName)
{
	CFile file;
	if( !file.OpenW(lpFileName, FOPEN_READ) ) return false;
	// t@Cwb_̓ǂݍ
	BMPFILEHEADER fh;
	file.Read(&fh, sizeof(BMPFILEHEADER));
	if( fh.bfType != BITMAP_FILEHEAD_TYPE ) return false;
	BMPINFOHEADER ih;
	// infowb_̓ǂݍ
	file.Read(&ih, sizeof(BMPINFOHEADER));
	if( !CheckFile(&fh, &ih) ) return false;

	u32 size = ih.biSizeImage + BMP_HEADSIZE;
	if( !Alloc(size) ) return false;

	*m_bmp.file = fh;
	*m_bmp.info = ih;
	file.Read(m_bmp.data, fh.bfSize-BMP_HEADSIZE);
	u32 ofs = m_bmp.file->bfOffBits - BMP_HEADSIZE;
	m_bmp.pixel = m_bmp.data + ofs;
	m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(ofs == 0 ? nullptr : m_bmp.data);
	return true;
}

/**********************************************************************//**
 *
 * t@C
 *
 -----------------------------------------------------------------------
 * @param [in]	lpFileName	= t@C
 * @return	
*//***********************************************************************/
template<>
bool CBitmap::WriteFile<CHAR>(LPCSTR lpFileName)
{
	return WriteFileA(lpFileName);
}
template<>
bool CBitmap::WriteFile<WCHAR>(LPCWSTR lpFileName)
{
	return WriteFileW(lpFileName);
}
/// CBitmap::WriteFile Q
bool CBitmap::WriteFileA(LPCSTR  lpFileName)
{
	CFile file;
	if( !IsValid() ) return false;
	if( !file.OpenA(lpFileName, FOPEN_WRITE|FOPEN_CREATE) ) return false;
	file.Write(m_bmp.file, m_bmp.file->bfSize);
	return true;
}
/// CBitmap::WriteFile Q
bool CBitmap::WriteFileW(LPCWSTR lpFileName)
{
	CFile file;
	if( !IsValid() ) return false;
	if( !file.OpenW(lpFileName, FOPEN_WRITE|FOPEN_CREATE) ) return false;
	file.Write(m_bmp.file, m_bmp.file->bfSize);
	return true;
}

/**********************************************************************//**
 *
 * obt@ǂݍ
 *
 -----------------------------------------------------------------------
 * @param [in]	lpBuffer	= ̓obt@
 * @return	
*//***********************************************************************/
bool CBitmap::ReadOnMemory(const void* lpBuffer)
{
	u8* buf = static_cast<u8*>(const_cast<void*>(lpBuffer));
	LPBMPFILEHEADER fh = pointer_cast<LPBMPFILEHEADER>(buf);
	LPBMPINFOHEADER ih = pointer_cast<LPBMPINFOHEADER>(buf+sizeof(BMPFILEHEADER));
	if( !CheckFile(fh, ih) ) return false;
	Release();
	m_bmp.file		= fh;
	m_bmp.info		= ih;
	m_bmp.data		= buf+BMP_HEADSIZE;
	m_bmp.pallete	= ((m_bmp.file->bfOffBits - BMP_HEADSIZE) == 0) ? nullptr : pointer_cast<IrisRGBQUAD*>(m_bmp.data);
	m_bmp.pixel		= m_bmp.data + m_bmp.file->bfOffBits - BMP_HEADSIZE;
	m_bAllocated	= false;
	return true;
}

/**********************************************************************//**
 *
 * obt@̊m
 *
 -----------------------------------------------------------------------
 * @param [in]	size	= mۂTCY
 * @return	
*//***********************************************************************/
bool CBitmap::Alloc(u32 size)
{
	if( m_bmp.file != nullptr )
	{
		IRIS_WARNING("bitmap data was allocated.");
		irisFree(m_bmp.file);
		memset(&m_bmp, 0, sizeof(m_bmp));
		m_bAllocated = false;
	}
	u8* data = pointer_cast<u8*>(irisAlloc(size, 0));
	if( data == nullptr ) return false;
	m_bmp.file = pointer_cast<LPBMPFILEHEADER>(data);
	m_bmp.info = pointer_cast<LPBMPINFOHEADER>(data+sizeof(BMPFILEHEADER));
	m_bmp.data = data+BMP_HEADSIZE;
	m_bAllocated = true;
	return true;
}

/**********************************************************************//**
 *
 * obt@̉
 *
*//***********************************************************************/
void CBitmap::Release(void)
{
	if( m_bAllocated )
	{
		if( m_bmp.file != nullptr )
		{
			irisFree(m_bmp.file);
		}
	}
	m_bAllocated = false;
	memset(&m_bmp, 0, sizeof(m_bmp));
}

/**********************************************************************//**
 *
 * rbg}bv̍쐬
 *
 -----------------------------------------------------------------------
 * @param [in]	nBitCount	= rbg[x
 * @param [in]	nWidht		= 
 * @param [in]	nHeight		= 
 * @return	
*//***********************************************************************/
bool CBitmap::Create(u16 nBitCount, s32 nWidth, s32 nHeight)
{
	Release();
	u32 plsize = 0;
	switch(nBitCount)
	{
	case 1:
		{
			plsize = sizeof(IrisRGBQUAD)*2;
			u32 size = IRIS_RoundUp4B((nWidth+7)>>3) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(m_bmp.data);
		}
		break;
	case 4:
		{
			plsize = sizeof(IrisRGBQUAD)*16;
			u32 size = IRIS_RoundUp4B((nWidth+1)>>1) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(m_bmp.data);
		}
		break;
	case 8:
		{
			plsize = sizeof(IrisRGBQUAD)*256;
			u32 size = IRIS_RoundUp4B(nWidth) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(m_bmp.data);
		}
		break;
	case 16:
		{
			plsize = sizeof(IrisRGBQUAD)*65536;
			u32 size = IRIS_RoundUp4B(nWidth*2) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(m_bmp.data);
		}
		break;
	case 24:
		{
			u32 size = IRIS_RoundUp4B(nWidth*3) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = nullptr;
		}
		break;
	case 32:
		{
			u32 size = IRIS_RoundUp4B(nWidth*4) * nHeight;
			if( !Alloc(size + BMP_HEADSIZE + plsize) ) return false;
			m_bmp.pallete = nullptr;
		}
		break;
	default:
		return false;;
	}

	m_bmp.file->bfOffBits = BMP_HEADSIZE + plsize;
	m_bmp.pixel			  = m_bmp.data + plsize;
	if( !CreateHeader(&m_bmp, nBitCount, nWidth, nHeight) ) return false;
	m_bmp.file->bfSize = m_bmp.file->bfOffBits + m_bmp.info->biSizeImage;
	memset(m_bmp.data, 0, m_bmp.file->bfSize - BMP_HEADSIZE);
	m_bAllocated = true;
	return true;
}

/**********************************************************************//**
 *
 * rbg}bṽRs[
 *
 -----------------------------------------------------------------------
 * @param [in]	rBitmap	= Rs[rbg}bv
 * @return	
*//***********************************************************************/
bool CBitmap::Duplicate(CBitmap& rBitmap)
{
	if( !rBitmap.IsValid() ) return false;
	u32 size = rBitmap.m_bmp.file->bfSize;
	if( !Alloc(size) ) return false;
	memcpy(m_bmp.file, rBitmap.m_bmp.file, size);

	m_bmp.pallete	= pointer_cast<IrisRGBQUAD*>(rBitmap.m_bmp.pallete != nullptr ? m_bmp.data : nullptr);
	m_bmp.pixel		= m_bmp.data + m_bmp.file->bfOffBits - BMP_HEADSIZE;
	return true;
}

/**********************************************************************//**
 *
 * C[W̃Rs[
 *
 -----------------------------------------------------------------------
 * @param [in]	rBitmap	= Rs[C[W
 * @return	
*//***********************************************************************/
bool CBitmap::Duplicate(IImage& rImage)
{
	if( !rImage.IsValid() ) return false;
	s32 w = rImage.GetWidth();
	s32 h = rImage.GetHeight();
	Release();
	if( !Create(24, w, h) ) return false;
	if( !rImage.CreateImage(m_bmp.pixel, PF_BGR8, 4, PPD_LEFT_TOP) )
	{
		Release();
		return false;
	}
	return true;
}

/**********************************************************************//**
 *
 * obt@Rs[
 *
 -----------------------------------------------------------------------
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	size		= obt@TCY
 * @return	
*//***********************************************************************/
bool CBitmap::DuplicateOnMemory(const void* lpBuffer, u32 size)
{
	Release();
	if( !Alloc(size) ) return false;
	u8* buf = pointer_cast<u8*>(m_bmp.file);
	memcpy(buf, lpBuffer, size);
	m_bmp.info = pointer_cast<LPBMPINFOHEADER>(buf+sizeof(BMPFILEHEADER));
	m_bmp.data = buf + BMP_HEADSIZE;
	u32 ofs = m_bmp.file->bfOffBits - BMP_HEADSIZE;
	m_bmp.pixel = m_bmp.data + ofs;
	m_bmp.pallete = pointer_cast<IrisRGBQUAD*>(ofs == 0 ? nullptr : m_bmp.data);
	return true;
}

/**********************************************************************//**
 *
 * wb_ɏ]ĕϊRs[
 *
 -----------------------------------------------------------------------
 * @param [in]	rBitmap	= Rs[rbg}bv
 * @return	
*//***********************************************************************/
bool CBitmap::Transfer(CBitmap& rBitmap)
{
	if( &rBitmap == this ) return false;
	if( !IsValid() ) return false;
	if( !rBitmap.IsValid() ) return false;
	BMPFMT& rbmp = rBitmap.m_bmp;
	u16 dbit = m_bmp.info->biBitCount;
	u16 sbit = rbmp.info->biBitCount;
	s32 dw = m_bmp.info->biWidth;
	s32 dh = m_bmp.info->biHeight;
	s32 sw = rbmp.info->biWidth;
	s32 sh = rbmp.info->biHeight;
	s32 tw = dw;
	s32 th = dh;
	if( tw > sw ) tw = sw;
	if( th > sh ) th = sh;
	u32 ofs = 0;
	int i=0;
	if( !IsSupportedBitCoount(dbit) ) return false;

	switch( sbit )
	{
	case 1:
		ofs = IRIS_RoundUp4B((sw+7)>>3);
		switch( dbit )
		{
		case 1:
			{
				for(i=0; i < 2; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h )
				{
					memcpy(m_bmp.pixel + IRIS_RoundUp4B((dw+7)>>3)*(dh-h-1)
						, rbmp.pixel + ofs * (sh-h-1), IRIS_RoundUp4B((tw+7)>>3));
				}
			}
			break;
		case 4:
			{
				for(i=0; i < 2; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; i < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u8* otpx = m_bmp.pixel + IRIS_RoundUp4B((dw+1)>>1) * (dh-h-1);
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 8 && w < tw; ++i, ++w)
						{
							int p1 = (*inpx & (0x01 << i)) >> i;
							++i;
							int p2 = 0;
							if( w+i < tw )	p2 = (*inpx & (0x01 << i)) >> i;
							*otpx = static_cast<u8>( ((p1 << 4) & 0xF0) | (p2 & 0x0F) );
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 8:
			{
				for(i=0; i < 2; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u8* otpx = m_bmp.pixel + IRIS_RoundUp4B(dw) * (dh-h-1);
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 8 && w < tw; ++i, ++w)
						{
							u8 p1 = (u8)((*inpx & (0x01 << i)) >> i);
							*otpx = p1;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 16:
			{
				for(i=0; i < 2; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u16* otpx = (u16*)(m_bmp.pixel + IRIS_RoundUp4B(dw*2) * (dh-h-1));
					for(s32 w=0; w < tw;)
					{
						for(i=0; i < 8 && w < tw; ++i, ++w)
						{
							u16 p1 = (u16)((*inpx & (0x01 << i)) >> i);
							*otpx = p1;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 24:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisBGR888* otpx = (IrisBGR888*)(m_bmp.pixel + IRIS_RoundUp4B(dw*3) * (dh-h-1));
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 8 && w < tw; ++i, ++w)
						{
							int p1 = (*inpx & (0x01 << i)) >> i;
							otpx->b = rbmp.pallete[p1].b;
							otpx->g = rbmp.pallete[p1].g;
							otpx->r = rbmp.pallete[p1].r;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 32:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisRGBQUAD* otpx = (IrisRGBQUAD*)(m_bmp.pixel + IRIS_RoundUp4B(dw*4) * (dh-h-1));
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 8 && w < tw; ++i, ++w)
						{
							int p1 = (*inpx & (0x01 << i)) >> i;
							*otpx = rbmp.pallete[p1];
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		default:
			return false;
		}
		break;
	case 4:
		ofs = IRIS_RoundUp4B((sw+1)>>1);
		switch( dbit )
		{
		//case 1:
		//	break;
		case 4:
			{
				for(i=0; i < 16; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h )
				{
					memcpy(m_bmp.pixel + IRIS_RoundUp4B((dw+1)>>1)*(dh-h-1)
						, rbmp.pixel + ofs * (sh-h-1), IRIS_RoundUp4B((tw+1)>>1));
				}
			}
			break;
		case 8:
			{
				for(i=0; i < 16; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u8* otpx = m_bmp.pixel + IRIS_RoundUp4B(dw) * (dh-h-1);
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 2 && w < tw; ++i, ++w)
						{
							u8 p1 = (u8)((*inpx & (0x0F << (i<<2))) >> (i<<2));
							*otpx = p1;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 16:
			{
				for(i=0; i < 16; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u16* otpx = (u16*)(m_bmp.pixel + IRIS_RoundUp4B(dw*2) * (dh-h-1));
					for(s32 w=0; w < tw; )
					{
						for(i=0; i < 2 && w < tw; ++i, ++w)
						{
							u16 p1 = (u16)((*inpx & (0x0F << (i<<2))) >> (i<<2));
							*otpx = p1;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 24:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisBGR888* otpx = (IrisBGR888*)(m_bmp.pixel + IRIS_RoundUp4B(dw*3) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						for(i=0; i < 2 && w < tw; ++i, ++w)
						{
							int p1 = (*inpx & (0x0F << (i<<2))) >> (i<<2);
							otpx->b = rbmp.pallete[p1].b;
							otpx->g = rbmp.pallete[p1].g;
							otpx->r = rbmp.pallete[p1].r;
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		case 32:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisRGBQUAD* otpx = (IrisRGBQUAD*)(m_bmp.pixel + IRIS_RoundUp4B(dw*4) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						for(i=0; i < 2 && w < tw; ++i, ++w)
						{
							int p1 = (*inpx & (0x0F << (i<<2))) >> (i<<2);
							*otpx = rbmp.pallete[p1];
							++otpx;
						}
						++inpx;
					}
				}
			}
			break;
		default:
			return false;
		}
		break;
	case 8:
		ofs = IRIS_RoundUp4B(sw);
		switch( dbit )
		{
		//case 1:
		//	break;
		//case 4:
		//	break;
		case 8:
			{
				for(i=0; i < 256; ++i)
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h )
				{
					memcpy(m_bmp.pixel + IRIS_RoundUp4B(dw)*(dh-h-1)
						, rbmp.pixel + ofs * (sh-h-1), IRIS_RoundUp4B(tw));
				}
			}
			break;
		case 16:
			{
				for(i=0; i < 256; ++i )
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					u16* otpx = (u16*)(m_bmp.pixel + IRIS_RoundUp4B(dw*2) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						u16 p1 = *inpx;
						*otpx = p1;
						++otpx;
						++inpx;
					}
				}
			}
			break;
		case 24:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisBGR888* otpx = (IrisBGR888*)(m_bmp.pixel + IRIS_RoundUp4B(dw*3) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						int p1 = *inpx;
						otpx->b = rbmp.pallete[p1].b;
						otpx->g = rbmp.pallete[p1].g;
						otpx->r = rbmp.pallete[p1].r;
						++otpx;
						++inpx;
					}
				}
			}
			break;
		case 32:
			{
				for(s32 h=0; h < th; ++h)
				{
					u8*	inpx = rbmp.pixel + ofs * (sh-h-1);
					IrisRGBQUAD* otpx = (IrisRGBQUAD*)(m_bmp.pixel + IRIS_RoundUp4B(dw*4) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						int p1 = *inpx;
						*otpx = rbmp.pallete[p1];
						++otpx;
						++inpx;
					}
				}
			}
			break;
		default:
			return false;
		}
		break;
	case 16:
		ofs = IRIS_RoundUp4B(sw*2);
		switch( dbit )
		{
		//case 1:
		//	break;
		//case 4:
		//	break;
		//case 8:
		//	break;
		case 16:
			{
				for(i=0; i < 65536; ++i )
					m_bmp.pallete[i] = rbmp.pallete[i];
				for(s32 h=0; h < th; ++h )
				{
					memcpy(m_bmp.pixel + IRIS_RoundUp4B(dw*2)*(dh-h-1)
						, rbmp.pixel + ofs * (sh-h-1), IRIS_RoundUp4B(tw*2));
				}
			}
			break;
		case 24:
			{
				for(s32 h=0; h < th; ++h)
				{
					u16*	inpx = (u16*)rbmp.pixel + ofs * (sh-h-1);
					IrisBGR888* otpx = (IrisBGR888*)(m_bmp.pixel + IRIS_RoundUp4B(dw*3) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						int p1 = *inpx;
						otpx->b = rbmp.pallete[p1].b;
						otpx->g = rbmp.pallete[p1].g;
						otpx->r = rbmp.pallete[p1].r;
						++otpx;
						++inpx;
					}
				}
			}
			break;
		case 32:
			{
				for(s32 h=0; h < th; ++h)
				{
					u16*	inpx = (u16*)rbmp.pixel + ofs * (sh-h-1);
					IrisRGBQUAD* otpx = (IrisRGBQUAD*)(m_bmp.pixel + IRIS_RoundUp4B(dw*4) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						int p1 = *inpx;
						*otpx = rbmp.pallete[p1];
						++otpx;
						++inpx;
					}
				}
			}
			break;
		default:
			return false;
		}
		break;
	case 24:
		ofs = IRIS_RoundUp4B(sw*3);
		switch( dbit )
		{
		//case 1:
		//	break;
		//case 4:
		//	break;
		//case 8:
		//	break;
		//case 16:
		//	break;
		case 24:
			for(s32 h=0; h < th; ++h )
			{
				memcpy(m_bmp.data + IRIS_RoundUp4B(dw*3)*(dh-h-1)
					, rbmp.data + ofs * (sh-h-1), IRIS_RoundUp4B(tw*3));
			}
			break;
		case 32:
			{
				for(s32 h=0; h < th; ++h)
				{
					IrisBGR888* inpx = (IrisBGR888*)(rbmp.data + ofs * (sh-h-1));
					IrisRGBQUAD* otpx = (IrisRGBQUAD*)(m_bmp.data + IRIS_RoundUp4B(dw*4) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						otpx->b = inpx->b;
						otpx->g = inpx->g;
						otpx->r = inpx->r;
						++otpx;
						++inpx;
					}
				}
			}
			break;
		default:
			return false;
		}
		break;
	case 32:
		ofs = IRIS_RoundUp4B(sw*4);
		switch( dbit )
		{
		//case 1:
		//	break;
		//case 4:
		//	break;
		//case 8:
		//	break;
		//case 16:
		//	break;
		case 24:
			{
				for(s32 h=0; h < th; ++h)
				{
					IrisRGBQUAD* inpx = (IrisRGBQUAD*)(rbmp.data + ofs * (sh-h-1));
					IrisBGR888* otpx = (IrisBGR888*)(m_bmp.data + IRIS_RoundUp4B(dw*3) * (dh-h-1));
					for(s32 w=0; w < tw; ++w)
					{
						otpx->b = inpx->b;
						otpx->g = inpx->g;
						otpx->r = inpx->r;
						++otpx;
						++inpx;
					}
				}
			}
			break;
		case 32:
			for(s32 h=0; h < th; ++h )
			{
				memcpy(m_bmp.data + IRIS_RoundUp4B(dw*4)*(dh-h-1)
					, rbmp.data + ofs * (sh-h-1), IRIS_RoundUp4B(tw*4));
			}
			break;
		default:
			return false;
		}
		break;
	default:
		return false;
	}

	return true;
}

/**********************************************************************//**
 *
 * f[^Lǂ
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CBitmap::IsValid(void) const
{
	return m_bmp.file != nullptr;
}

/**********************************************************************//**
 *
 * ΉĂBitCountǂ
 *
 -----------------------------------------------------------------------
 * @param [in]	nBitCount	= rbg[x
 * @return	^Ul
*//***********************************************************************/
bool CBitmap::IsSupportedBitCoount(u16 nBitCount)
{
	if( nBitCount == 1
		|| nBitCount == 4
		|| nBitCount == 8
		|| nBitCount == 16
		|| nBitCount == 24
		|| nBitCount == 32
	)
	{
		return true;
	}
	return false;
}

/**********************************************************************//**
 *
 * rbgJEg擾
 *
 -----------------------------------------------------------------------
 * @return	rbgJEg
*//***********************************************************************/
u16 CBitmap::GetBitCount(void) const
{
	if( m_bmp.info == nullptr ) return 0;
	return m_bmp.info->biBitCount;
}

/**********************************************************************//**
 *
 * sNZf[^̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	x	= xW
 * @param [in]	y	= yW
 * @return	sNZf[^
*//***********************************************************************/
void* CBitmap::GetPixel(s32 x, s32 y) const
{
	if( !IsValid() ) return nullptr;
	s32 w = m_bmp.info->biWidth;
	s32 h = m_bmp.info->biHeight;
	if( x >= w || y >= h ) return nullptr;
	s32 dy = h-y-1;
	u32 lw;
	switch( m_bmp.info->biBitCount )
	{
	case 1:
		lw = IRIS_RoundUp4B((w+7)>>3);
		x >>= 3;
		break;
	case 4:
		lw = IRIS_RoundUp4B((w+1)>>1);
		x >>= 1;
		break;
	case 8:
		lw = IRIS_RoundUp4B(w);
		break;
	case 16:
		lw = IRIS_RoundUp4B(w*2);
		x <<= 1;
		break;
	case 24:
		lw = IRIS_RoundUp4B(w*3);
		x *= 3;
		break;
	case 32:
		lw = IRIS_RoundUp4B(w*4);
		x <<= 2;
		break;
	default:
		return nullptr;
	}
	return m_bmp.pixel + lw * dy + x;
}

/**********************************************************************//**
 *
 * sNZobt@̎擾
 *
 -----------------------------------------------------------------------
 * @return	sNZobt@
*//***********************************************************************/
void* CBitmap::GetPixel(void) const
{
	return m_bmp.pixel;
}

/**********************************************************************//**
 *
 * sNZ̐F̎擾
 *
 -----------------------------------------------------------------------
 * @param [out]	pDst	= o
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @return	
*//***********************************************************************/
bool CBitmap::GetPixelRGBQUAD(s32 x, s32 y, IrisRGBQUAD& rgbq) const
{
	void* pixel = GetPixel(x, y);
	if( pixel == nullptr ) return false;
	int idx = 0;
	switch( m_bmp.info->biBitCount )
	{
	case 1:
		idx = (*(u8*)(pixel) >> (x & 0x7)) & 0x1;
		rgbq = *(m_bmp.pallete + idx);
		break;
	case 4:
		idx = (*(u8*)(pixel) >> ((x & 0x1) << 2)) & 0xF;
		rgbq = *(m_bmp.pallete + idx);
		break;
	case 8:
		idx = *(u8*)pixel;
		rgbq = *(m_bmp.pallete + idx);
		break;
	case 16:
		idx = *(u16*)pixel;
		rgbq = *(m_bmp.pallete + idx);
		break;
	case 24:
		{
			IrisBGR888* tmp = (IrisBGR888*)pixel;
			rgbq.r = tmp->r;
			rgbq.g = tmp->g;
			rgbq.b = tmp->b;
			rgbq.Re = 0;
		}
		break;
	case 32:
		{
			rgbq = *(IrisRGBQUAD*)(pixel);
		}
		break;
	default:
		return false;
	}
	return true;
}

/**********************************************************************//**
 *
 * sNZ̐F̎擾
 *
 -----------------------------------------------------------------------
 * @param [out]	pDst	= o
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @return	
*//***********************************************************************/
bool CBitmap::SetPixelRGBQUAD(s32 x, s32 y, const IrisRGBQUAD& rgbq)
{
	void* pixel = GetPixel(x, y);
	if( pixel == nullptr ) return false;
	int idx = 0;
	switch( m_bmp.info->biBitCount )
	{
	case 1:
		idx = (*(u8*)(pixel) >> (x & 0x7)) & 0x1;
		*(m_bmp.pallete + idx) = rgbq;
		break;
	case 4:
		idx = (*(u8*)(pixel) >> ((x & 0x1) << 2)) & 0xF;
		*(m_bmp.pallete + idx) = rgbq;
		break;
	case 8:
		idx = *(u8*)pixel;
		*(m_bmp.pallete + idx) = rgbq;
		break;
	case 16:
		idx = *(u16*)pixel;
		*(m_bmp.pallete + idx) = rgbq;
		break;
	case 24:
		{
			IrisBGR888* tmp = (IrisBGR888*)pixel;
			tmp->r = rgbq.r;
			tmp->g = rgbq.g;
			tmp->b = rgbq.b;
		}
		break;
	case 32:
		{
			*(IrisRGBQUAD*)(pixel) = rgbq;
		}
		break;
	default:
		return false;
	}
	return true;
}

/**********************************************************************//**
 *
 * pbg̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	idx = pbgCfbNX
 * @return	sNZf[^
*//***********************************************************************/
IrisRGBQUAD* CBitmap::GetPallete(u32 idx) const
{
	if( m_bmp.pallete == nullptr ) return nullptr;
	u32 max = GetPalleteNum();
	if( idx >= max ) return nullptr;
	return &m_bmp.pallete[idx];
}

/**********************************************************************//**
 *
 * pbgobt@̎擾
 *
 -----------------------------------------------------------------------
 * @return	pbgobt@
*//***********************************************************************/
IrisRGBQUAD* CBitmap::GetPallete(void) const
{
	return m_bmp.pallete;
}

/**********************************************************************//**
 *
 * pbǧ擾
 *
 -----------------------------------------------------------------------
 * @return	pbg
*//***********************************************************************/
u32 CBitmap::GetPalleteNum(void) const
{
	//IRIS_ASSERT( IsValid() );
	if( m_bmp.info == nullptr ) return (u32)-1;
	return m_bmp.info->biClrUsed;
}

/**********************************************************************//**
 *
 * pbg̃TCY擾
 *
 -----------------------------------------------------------------------
 * @return	pbgTCY
*//***********************************************************************/
u32 CBitmap::GetPalleteSize(void) const
{
	//IRIS_ASSERT( IsValid() );
	if( m_bmp.info == nullptr ) return 0;
	return m_bmp.info->biClrUsed * sizeof(IrisRGBQUAD);
}

/**********************************************************************//**
 *
 * C[Wf[^TCY擾
 *
 -----------------------------------------------------------------------
 * @return	C[Wf[^TCY
*//***********************************************************************/
u32 CBitmap::GetImageSize(void) const
{
	//IRIS_ASSERT( IsValid() );
	return GetImageSize(m_bmp.info);
}

/**********************************************************************//**
 *
 * C[Wf[^TCY擾
 *
 -----------------------------------------------------------------------
 * @return	C[Wf[^TCY
*//***********************************************************************/
u32 CBitmap::GetImageSize(LPBMPINFOHEADER lpih)
{
	if( lpih == nullptr ) return 0;
	u32 size = 0;
	s32 nWidth = lpih->biWidth;
	s32 nHeight = lpih->biHeight;
	switch(lpih->biBitCount)
	{
	case 1:
		size = IRIS_RoundUp4B((nWidth+7)>>3) * nHeight;
		break;
	case 4:
		size = IRIS_RoundUp4B((nWidth+1)>>1) * nHeight;
		break;
	case 8:
		size = IRIS_RoundUp4B(nWidth) * nHeight;
		break;
	case 16:
		size = IRIS_RoundUp4B(nWidth*2) * nHeight;
		break;
	case 24:
		size = IRIS_RoundUp4B(nWidth*3) * nHeight;
		break;
	case 32:
		size = IRIS_RoundUp4B(nWidth*4) * nHeight;
		break;
	}
	return size;
}

/**********************************************************************//**
 *
 * obt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CBitmap::CreateImage(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	switch( GetBitCount() )
	{
	case 1:
		return CreateImage1(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 4:
		return CreateImage4(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 8:
		return CreateImage8(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 16:
		return CreateImage16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 24:
		return CreateImage24(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 32:
		return CreateImage32(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return false;
}

bool CBitmap::CreateImage1(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

bool CBitmap::CreateImage4(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

bool CBitmap::CreateImage8(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

bool CBitmap::CreateImage16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

bool CBitmap::CreateImage24(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

bool CBitmap::CreateImage32(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( pBuffer == nullptr ) return false;
	if( !IsValid() ) return false;
	IRIS_UNUSED_VAR(nPixelFormat);
	IRIS_UNUSED_VAR(nLineAlign);
	IRIS_UNUSED_VAR(nOrder);
	return false;
}

/**********************************************************************//**
 *
 * sNZJ[̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	x	= xW
 * @param [in]	y	= yW
 * @return	sNZRGBA
*//***********************************************************************/
bool CBitmap::GetPixelRGBA8888(s32 x, s32 y, IrisRGBA8888& rgba) const
{
	IrisRGBQUAD tmp;
	if( !GetPixelRGBQUAD(x, y, tmp) ) return false;
	rgba.r = tmp.r;
	rgba.g = tmp.g;
	rgba.b = tmp.b;
	rgba.a = 0xFF;
	return true;
}

/**********************************************************************//**
 *
 * sNZJ[̐ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [in]	rgba	= sNZRGBA
*//***********************************************************************/
bool CBitmap::SetPixelRGBA8888(s32 x, s32 y, const IrisRGBA8888& rgba)
{
	IrisRGBQUAD tmp;
	tmp.r = rgba.r;
	tmp.g = rgba.g;
	tmp.b = rgba.b;
	return SetPixelRGBQUAD(x, y, tmp);
}

/**********************************************************************//**
 *
 * pbg̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	r = Rl
 * @param [in]	g = Gl
 * @param [in]	b = Bl
 * @return	Px
*//***********************************************************************/
u8 CBitmap::GetLuminance(u8 r, u8 g, u8 b)
{
	return CLuminance::Calc<0>(r, g, b);
}

/**********************************************************************//**
 *
 * rbg}bvwb_[̂ݍ쐬
 *
 -----------------------------------------------------------------------
 * @param [out]	pBmpFmt		= o͍\̃AhX
 * @param [in]	nBitCount	= rbg[x
 * @param [in]	nWidth		= 
 * @param [in]	nHeight		= 
 * @return	
*//***********************************************************************/
bool CBitmap::CreateHeader(LPBMPFMT pBmpFmt, u16 nBitCount, s32 nWidth, s32 nHeight)
{
	if( pBmpFmt == nullptr ) return false;
	if( !IsSupportedBitCoount(nBitCount) ) return false;
	pBmpFmt->file->bfType = BITMAP_FILEHEAD_TYPE;

	pBmpFmt->info->biSize			= sizeof(BMPINFOHEADER);
	pBmpFmt->info->biBitCount		= nBitCount;
	pBmpFmt->info->biPlanes			= 1;
	pBmpFmt->info->biClrImportant	= 0;
	pBmpFmt->info->biXPelsPerMeter	= 0;
	pBmpFmt->info->biYPelsPerMeter	= 0;
	pBmpFmt->info->biCompression	= 0;
	pBmpFmt->info->biWidth			= nWidth;
	pBmpFmt->info->biHeight			= nHeight;

	u32 size = 0;
	u32 clr = 0;
	switch(nBitCount)
	{
	case 1:
		size = IRIS_RoundUp4B((nWidth+7)>>3) * nHeight;
		clr = 2;
		break;
	case 4:
		size = IRIS_RoundUp4B((nWidth+1)>>1) * nHeight;
		clr = 16;
		break;
	case 8:
		size = IRIS_RoundUp4B(nWidth) * nHeight;
		clr = 256;
		break;
	case 16:
		size = IRIS_RoundUp4B(nWidth*2) * nHeight;
		clr = 65536;
		break;
	case 24:
		size = IRIS_RoundUp4B(nWidth*3) * nHeight;
		break;
	case 32:
		size = IRIS_RoundUp4B(nWidth*4) * nHeight;
		break;
	}
	pBmpFmt->info->biSizeImage		= size;
	pBmpFmt->info->biClrUsed		= clr;
	return true;
}

}	// end of namespace fnd
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../../unit/UnitCore.h"
#include "iris_using.h"
#include "../../iris_iostream.h"
#include <stdio.h>
#include <string.h>
#include <tchar.h>

//======================================================================
// test
IRIS_UNITTEST(CFndBitmapUnitTest, Func)
{
	CBitmap bitmap, out;
	TCHAR path[MAX_PATH] = TEXT("../../../data/image/sample.bmp");
	TCHAR out_path[MAX_PATH] = TEXT("");
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "Jbitmapt@C͂ĂB" << std::endl;
	std::tcin >> path;
#endif

	if( !bitmap.ReadFile(path) )
	{
		std::cout << "t@CI[vɎs܂B" << std::endl;
		return;
	}

	std::cout << "rbg[xƃTCYύXāAo͂܂B" << std::endl;

	u16 bit[] = { 1, 4, 8, 16, 24, 32 };
	for( int i=0, sz=IRIS_NumOfElements(bit); i < sz; ++i )
	{
		TCHAR temp[16];
		out.Create(bit[i], 100, 100);
		if( out.Transfer(bitmap) )
		{
			_tcscpy_s(out_path, MAX_PATH, path);
			wsprintf(temp, IRIS_TEXT("%d.bmp"), bit[i]);
			_tcscat_s(out_path, MAX_PATH, temp);
			out.WriteFile(out_path);
			std::tcout << "write file ->" << out_path << std::endl;
		}
	}
}

#endif
