//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndEdgeDetector.h
 * @brief		GbWot@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndEdgeDetector_H_
#define INCG_IRIS_FndEdgeDetector_H_

//======================================================================
// include
#include "FndImage.h"
#include "../../iris_debug.h"
#include "../../iris_xf.h"

namespace iris {
namespace fnd
{

//======================================================================
// class
/// GbWoNX
class CEdgeDetector : public IIrisObject
{
public:
	static const int MAX_KERNEL	= 2;	//!< J[l

	/// RA
	typedef class IKernel : public IIrisObject
	{
	public:
		virtual s32*	GetCore(void) = 0;
		virtual s32		GetAperture(void) = 0;
	} KERNEL, *LPKERNEL;

	template<int APERTURE_=3>
	class CKernel : public IKernel
	{
	public:
		enum
		{
			APERTURE	= APERTURE_,			//!< 
			BOXNUM		= APERTURE_*APERTURE_	//!< RA̗vf
		};
		IRIS_STATIC_ASSERT( APERTURE & 0x1 );

	protected:
		union
		{
			struct {
				s32	val[BOXNUM];				//!< l
			};
			struct {
				s32	box[APERTURE][APERTURE];	//!< l
			};
		};
	public:
		virtual s32*	GetCore(void)		{ return val; }
		virtual s32		GetAperture(void)	{ return APERTURE; }
	};

protected:
	LPKERNEL	m_pKernel[MAX_KERNEL];	//!< RA
public:
	// RXgN^
	CEdgeDetector(void);

public:
	// o
	bool	Detect(IImage* lpDst, const IImage* lpSrc, xf32 rate);
	// o
	bool	DetectFast(IImage* lpDst, const IImage* lpSrc, xf32 rate, void* lpWork, u32 WorkSize);

protected:
	// ZxvZ
	s32		Convolution(s32 x, s32 y, const IImage* lpSrc, LPKERNEL lpKernel);
public:
	/// RA̐ݒ
	void	SetKernel(s32 index, LPKERNEL pKernel)	{ IRIS_ASSERT( index >= 0 && index < MAX_KERNEL ); m_pKernel[index] = pKernel; }

public:
	// RAev[g
	/// Prewitt 
	template<int APERTURE_=3>
	struct PREWITT_KERNEL_H : public CKernel<APERTURE_>
	{
		PREWITT_KERNEL_H(void)
		{
			for( int i=0; i<APERTURE_; ++i )
			{
				this->box[i][0] = 1;
				for( int j=1; j<APERTURE_-1; ++j )
				{
					this->box[i][j] = 0;
				}
				this->box[i][APERTURE_-1] = -1;
			}
		}
	};
	/// Prewitt 
	template<int APERTURE_=3>
	struct PREWITT_KERNEL_V : public CKernel<APERTURE_>
	{
		PREWITT_KERNEL_V(void)
		{
			for( int i=0; i<APERTURE_; ++i )
			{
				this->box[0][i] = 1;
			}
			for( int i=1; i<APERTURE_-1; ++i )
			{
				for( int j=0; j<APERTURE_; ++j )
				{
					this->box[i][j] = 0;
				}
			}
			for( int i=0; i<APERTURE_; ++i )
			{
				this->box[APERTURE_-1][i] = -1;
			}
		}
	};

	/// Sobel 
	template<int APERTURE_=3>
	struct SOBEL_KERNEL_H : public CKernel<APERTURE_>
	{
		SOBEL_KERNEL_H(void)
		{
			int level=1, i=0;
			for( i=0; i<APERTURE_/2; ++i, ++level )
			{
				this->box[i][0] = level;
				for( int j=1; j<APERTURE_-1; ++j )
				{
					this->box[i][j] = 0;
				}
				this->box[i][APERTURE_-1] = -level;
			}
			for( ; i<APERTURE_; ++i, --level )
			{
				this->box[i][0] = level;
				for( int j=1; j<APERTURE_-1; ++j )
				{
					this->box[i][j] = 0;
				}
				this->box[i][APERTURE_-1] = -level;
			}
		}
	};

	/// Sobel 
	template<int APERTURE_=3>
	struct SOBEL_KERNEL_V : public CKernel<APERTURE_>
	{
		SOBEL_KERNEL_V(void)
		{
			int i=0, level=1;
			for( i=0; i<APERTURE_/2; ++i, ++level )
			{
				this->box[0][i] = 1;
			}
			for( ; i<APERTURE_; ++i, --level )
			{
				this->box[0][i] = 1;
			}
			for( i=1; i<APERTURE_-1; ++i )
			{
				for( int j=0; j<APERTURE_; ++j )
				{
					this->box[i][j] = 0;
				}
			}
			for( i=0; i<APERTURE_/2; ++i, ++level )
			{
				this->box[APERTURE_-1][i] = -1;
			}
			for( ; i<APERTURE_; ++i, --level )
			{
				this->box[APERTURE_-1][i] = -1;
			}
		}
	};

	/// Laplacian 4
	template<int APERTURE_=3>
	struct LAPLACIAN4_KERNEL : public CKernel<APERTURE_>
	{
		LAPLACIAN4_KERNEL(void)
		{
			int i=0;
			for( i=0; i < APERTURE_/2; ++i )
			{
				int j=0;
				for( j=0; j < APERTURE_/2; ++j )
				{
					this->box[i][j] = 0;
				}
				this->box[i][j++] = -1;
				for( ; j < APERTURE_; ++j )
				{
					this->box[i][j] = 0;
				}
			}

			{
				int j=0;
				for( j=0; j < APERTURE_/2; ++j )
				{
					this->box[i][j] = -1;
				}
				this->box[i][j++] = 4;
				for( ; j < APERTURE_; ++j )
				{
					this->box[i][j] = -1;
				}
			}

			for( ++i; i < APERTURE_; ++i )
			{
				int j=0;
				for( j=0; j < APERTURE_/2; ++j )
				{
					this->box[i][j] = 0;
				}
				this->box[i][j++] = -1;
				for( ; j < APERTURE_; ++j )
				{
					this->box[i][j] = 0;
				}
			}
		}
	};

	/// Laplacian 8
	template<int APERTURE_=3>
	struct LAPLACIAN8_KERNEL : public CKernel<APERTURE_>
	{
		LAPLACIAN8_KERNEL(void)
		{
			int i=0;
			for( i=0; i < APERTURE_/2; ++i )
			{
				for( int j=0; j < APERTURE_; ++j )
				{
					this->box[i][j] = -1;
				}
			}

			{
				int j=0;
				for( j=0; j < APERTURE_/2; ++j )
				{
					this->box[i][j] = -1;
				}
				this->box[i][j++] = 8;
				for( ; j < APERTURE_; ++j )
				{
					this->box[i][j] = -1;
				}
			}

			for( ++i; i < APERTURE_; ++i )
			{
				for( int j=0; j < APERTURE_; ++j )
				{
					this->box[i][j] = -1;
				}
			}
		}
	};
};


}	// end of namespace fnd
}	// end of namespace iris

#endif
