//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathBezierCurve.cpp
 * @brief		xWGȐt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_MathBezierCurve_CPP_

//======================================================================
// include
#include "MathBezierCurve.h"
#include "../MathPractical.h"
#include "../MathPower.h"
#include "../../iris_debug.h"
#include "../../iris_math.h"

namespace iris {
namespace math
{

//======================================================================
// function
/**********************************************************************//**
 *
 * xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [in]	t	= (0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	...	= n̓_
 * @return o͒l
*//***********************************************************************/
f32	BezierCurve(f32 t, s32 n, ...)
{
	IRIS_ASSERT( n > 0 );
	f32 ret = 0.0f;
	va_list va;
	va_start(va, n);
	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	if( t <= 0.0f ) { ret = (f32)va_arg(va, xf32_va); goto func_end; }
	if( t >= 1.0f )
	{
		for( s32 i=0; i < n-1; ++i ) (void)va_arg(va, f32_va);
		ret = (f32)va_arg(va, f32_va);
		goto func_end;
	}

	for( s32 i=0; i < n; ++i )
	{
		f32 B = (f32)va_arg(va, f32_va);
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		ret += F32_Mul(B, J);
		tt = F32_Mul(tt, t);
	}
func_end:
	va_end(va);
	return ret;
}

/**********************************************************************//**
 *
 * xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [in]	t	= ( 0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	pt	= _̔z(n)
 * @return o͒l
*//***********************************************************************/
f32	BezierCurveArray(f32 t, s32 n, f32 pt[])
{
	IRIS_ASSERT( pt != nullptr );
	IRIS_ASSERT( n >= 2 );
	f32 ret = 0.0f;
	if( t <= 0.0f )	{ return *pt; }
	if( t >= 1.0f )	{ return pt[n-1]; }

	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	f32* pp = pt;
	for( s32 i=0; i < n; ++i, ++pp )
	{
		f32 B = *pp;
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		ret += F32_Mul(B, J);
		tt = F32_Mul(tt, t);
	}
	return ret;
}

/**********************************************************************//**
 *
 * NxNg̃xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [out]	pV	= xNg
 * @param [in]	N	= 
 * @param [in]	t	= (0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	...	= n̃xNg(f32*)
 * @return o͒l
*//***********************************************************************/
f32*	BezierCurveN(f32* pV, s32 N, f32 t, s32 n, ...)
{
	IRIS_ASSERT( pV != nullptr );
	IRIS_ASSERT( N > 0 );
	IRIS_ASSERT( n > 0 );
	for( s32 i=0; i < N; ++i ) pV[i] = 0.0f;
	va_list va;
	va_start(va, n);
	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	if( t <= 0.0f )
	{
		f32* pv = va_arg(va, f32*);
		for( s32 i=0; i < N; ++i, ++pv )
			pV[i] = *pv;
		goto func_end;
	}
	if( t >= 1.0f )
	{
		for( s32 i=0; i < n-1; ++i )
			(void)va_arg(va, f32*);

		f32* pv = va_arg(va, f32*);
		for( s32 i=0; i < N; ++i, ++pv )
			pV[i] = *pv;
		goto func_end;
	}

	for( s32 i=0; i < n; ++i )
	{
		f32* pv = va_arg(va, f32*);
		//f32 J = BernsteinPolynomial(nn, i, t);
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		for( s32 j=0; j < N; ++j, ++pv )
		{
			pV[j] += F32_Mul(*pv, J);
		}
		tt = F32_Mul(tt, t);
	}
func_end:
	va_end(va);
	return pV;
}

/**********************************************************************//**
 *
 * NxNg̃xWGȐ
 *
 -----------------------------------------------------------------------
 * @param [out]	pV	= xNg
 * @param [in]	N	= 
 * @param [in]	t	= ( 0 <= t <= 1)
 * @param [in]	n	= _̐
 * @param [in]	pt	= _̔z(n)
 * @return o͒l
*//***********************************************************************/
f32*	BezierCurveArrayN(f32* pV, s32 N, f32 t, s32 n, f32 pt[])
{
	IRIS_ASSERT( pV != nullptr );
	IRIS_ASSERT( N > 0 );
	IRIS_ASSERT( pt != nullptr );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0.0f )	{ for(s32 i=0; i < N; ++i) pV[i] = pt[i];			return pV; }
	if( t >= 1.0f )	{ for(s32 i=0; i < N; ++i) pV[i] = pt[(n-1)*N+i];	return pV; }

	for( s32 j=0; j < N; ++j )	{ pV[j] = 0.0f; }

	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	f32* pp = pt;
	for( s32 i=0; i < n; ++i)
	{
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		//f32 J = BernsteinPolynomial(nn, i, t);
		for( s32 j=0; j < N; ++j, ++pp )
		{
			pV[j] += F32_Mul(*pp, J);
		}
		tt = F32_Mul(tt, t);
	}
	return pV;
}

}	// end of namespace math
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_using.h"
#include "../../iris_iostream.h"

//======================================================================
// test
IRIS_UNITTEST(CMathBezierCurveUnitTest, Func)
{
	f32 P[] = {
		    0.0f
		,  50.0f
		, 100.0f
		, 150.0f
	};
	IrisS32 N = sizeof(P)/sizeof(f32);

	IrisFVec2 V[] =
	{
		  {   0.0f, 150.0f, }
		, {  50.0f, 100.0f, }
		, { 100.0f,  50.0f, }
		, { 150.0f,   0.0f, }
	};
	IrisS32 NV = sizeof(V)/sizeof(IrisFVec2);

	for( int i=0; i < 11; ++i )
	{
		std::cout << BezierCurveArray(i/10.0f, N, P) << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		std::cout << BezierCurve(i/10.0f, N, P[0], P[1], P[2], P[3]) << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		IrisFVec2 tmp;
		f32* p = (f32*)&V;
		BezierCurveArrayN((f32*)&tmp
			, 2, i/10.0f, NV, p);

		std::cout << tmp.x << ", " << tmp.y  << std::endl;
	}

	std::cout << "------------------------------" << std::endl;
	for( int i=0; i < 11; ++i )
	{
		IrisFVec2 tmp;
		BezierCurveN((f32*)&tmp
			, 2, i/10.0f, NV
			, (f32*)&V[0], (f32*)&V[1], (f32*)&V[2], (f32*)&V[3]);

		std::cout << tmp.x << ", " << tmp.y  << std::endl;
	}
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
