//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathColor.inl
 * @brief		J[Zt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathColor_inl_
#define INCG_IRIS_MathColor_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	J[^̏
 * @param [out]	pc	= o̓J[
 * @param [in]	r	= Rl
 * @param [in]	g	= Gl
 * @param [in]	b	= Bl
 * @param [in]	a	= Al
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorSet(IrisFColor* pc, f32 r, f32 g, f32 b, f32 a)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = r;
	pc->g = g;
	pc->b = b;
	pc->a = a;
	return pc;
}

/**
 * @brief	J[^̏
 * @param [out]	pc	= o̓J[
 * @param [in]	r	= Rl
 * @param [in]	g	= Gl
 * @param [in]	b	= Bl
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorSetRGB(IrisFColor* pc, f32 r, f32 g, f32 b)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = r;
	pc->g = g;
	pc->b = b;
	return pc;
}

/**
 * @brief	J[^̃V[Rs[
 * @param [out]	pc0	= o̓J[
 * @param [in]	pc1	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorCopy(IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = pc1->r;
	pc0->g = pc1->g;
	pc0->b = pc1->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃[
 * @param [out]	pc0	= o̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorZero(IrisFColor* pc0)
{
	MATH_FPU_NULLASSERT( pc0 );
	pc0->r = 0.0f;
	pc0->g = 0.0f;
	pc0->b = 0.0f;
	pc0->a = 0.0f;
	return pc0;
}

/**
 * @brief	J[^RGBA8888ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisRGBA8888	FpuColorToRGBA8888(const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pc );
	IrisRGBA8888 rgba;
	rgba.r = (u8)(( pc->r < 0.0f ) ? 0x00 : ( pc->r > 255.0f ) ? 0xFF : (pc->r * 0xFF) ); 
	rgba.g = (u8)(( pc->g < 0.0f ) ? 0x00 : ( pc->g > 255.0f ) ? 0xFF : (pc->g * 0xFF) ); 
	rgba.b = (u8)(( pc->b < 0.0f ) ? 0x00 : ( pc->b > 255.0f ) ? 0xFF : (pc->b * 0xFF) ); 
	rgba.a = (u8)(( pc->a < 0.0f ) ? 0x00 : ( pc->a > 255.0f ) ? 0xFF : (pc->a * 0xFF) ); 
	return rgba;
}

/**
 * @brief	J[^ARGB8888ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisBGRA8888	FpuColorToBGRA8888(const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pc );
	IrisBGRA8888 argb;
	argb.r = (u8)(( pc->r < 0.0f ) ? 0x00 : ( pc->r > 255.0f ) ? 0xFF : (pc->r * 0xFF) ); 
	argb.g = (u8)(( pc->g < 0.0f ) ? 0x00 : ( pc->g > 255.0f ) ? 0xFF : (pc->g * 0xFF) ); 
	argb.b = (u8)(( pc->b < 0.0f ) ? 0x00 : ( pc->b > 255.0f ) ? 0xFF : (pc->b * 0xFF) ); 
	argb.a = (u8)(( pc->a < 0.0f ) ? 0x00 : ( pc->a > 255.0f ) ? 0xFF : (pc->a * 0xFF) ); 
	return argb;
}

/**
 * @brief	J[^RGBA4444ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisRGBA4444	FpuColorToRGBA4444(const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pc );
	IrisRGBA4444 rgba;
	rgba.r = (u16)(( pc->r < 0.0f ) ? 0x0 : ( pc->r > 1.0f ) ? 0xF : (pc->r * 0xF)); 
	rgba.g = (u16)(( pc->g < 0.0f ) ? 0x0 : ( pc->g > 1.0f ) ? 0xF : (pc->g * 0xF)); 
	rgba.b = (u16)(( pc->b < 0.0f ) ? 0x0 : ( pc->b > 1.0f ) ? 0xF : (pc->b * 0xF)); 
	rgba.a = (u16)(( pc->a < 0.0f ) ? 0x0 : ( pc->a > 1.0f ) ? 0xF : (pc->a * 0xF)); 
	return rgba;
}

/**
 * @brief	J[^RGBA5551ɕϊ
 * @param [in]	pc	= ̓J[
 * @param [in]	q	= 臒l
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisRGBA5551	FpuColorToRGBA5551(const IrisFColor* pc, f32 q)
{
	MATH_FPU_NULLASSERT( pc );
	IrisRGBA5551 rgba;
	rgba.r = (u16)(( pc->r < 0.0f ) ? 0x0 : ( pc->r > 1.0f ) ? 0x1F : (pc->r * 0x1F));
	rgba.g = (u16)(( pc->g < 0.0f ) ? 0x0 : ( pc->g > 1.0f ) ? 0x1F : (pc->g * 0x1F));
	rgba.b = (u16)(( pc->b < 0.0f ) ? 0x0 : ( pc->b > 1.0f ) ? 0x1F : (pc->b * 0x1F));
	rgba.a = ( pc->a <= q ) ? 0u : 1u; 
	return rgba;
}

/**
 * @brief	J[^RGB565ɕϊ
 * @param [in]	pc	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisRGB565		FpuColorToRGB565(const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pc );
	IrisRGB565 rgb;
	rgb.r = (u16)(( pc->r < 0.0f ) ? 0x0u : ( pc->r > 1.0f ) ? 0xFu : (pc->r * 0x1F)); 
	rgb.g = (u16)(( pc->g < 0.0f ) ? 0x0u : ( pc->g > 1.0f ) ? 0xFu : (pc->g * 0x3F)); 
	rgb.b = (u16)(( pc->b < 0.0f ) ? 0x0u : ( pc->b > 1.0f ) ? 0xFu : (pc->b * 0x1F)); 
	return rgb;
}

/**
 * @brief	J[^CMYɕϊ
 * @param [out]	pcmy	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFCMY*		FpuColorToCMY(IrisFCMY*  pcmy, const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pcmy );
	MATH_FPU_NULLASSERT( pc );
	pcmy->c = 1.0f - pc->r;
	pcmy->m = 1.0f - pc->g;
	pcmy->y = 1.0f - pc->b;
	return pcmy;
}

/**
 * @brief	J[^CMYKɕϊ
 * @param [out]	pcmyk	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFCMYK*		FpuColorToCMYK(IrisFCMYK*  pcmyk, const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pcmyk );
	MATH_FPU_NULLASSERT( pc );
	f32 c = 1.0f - pc->r;
	f32 m = 1.0f - pc->g;
	f32 y = 1.0f - pc->b;
	f32 k = min(c, min(m, y));
	f32 tk = 1.0f - k;
	pcmyk->c = F32_Div((c - k) , tk);
	pcmyk->m = F32_Div((m - k) , tk);
	pcmyk->y = F32_Div((y - k) , tk);
	pcmyk->k = k;
	return pcmyk;
}

/**
 * @brief	J[^HSVɕϊ
 * @param [out]	phsv	= o̓J[
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFHSV*		FpuColorToHSV(IrisFHSV* phsv, const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( phsv );
	MATH_FPU_NULLASSERT( pc );
	f32 fmax = max(max(pc->r, pc->g), pc->b);
	f32 fmin = min(min(pc->r, pc->g), pc->b);
	f32 diff = fmax - fmin;
	f32 h = 0.0f;
	f32 s = F32_Div(diff, fmax);
	if( diff != 0.0f )
	{
		if( fmax == pc->r )			h = F32_Mul(60.0f, F32_Div((pc->g - pc->b), diff));
		else if( fmax == pc->g )	h = F32_Mul(60.0f, F32_Div((pc->b - pc->r), diff)) + 120.0f;
		else if( fmax == pc->b )	h = F32_Mul(60.0f, F32_Div((pc->r - pc->g), diff)) + 240.0f;
		if( h < 0.0f ) h += 360.0f;
		h = F32_Div(h, 360.0f);	// 0 ` 1 ɐK
	}
	phsv->h = h;
	phsv->s = s;
	phsv->v = fmax;
	return phsv;
}

/**
 * @brief	RGBA8888J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromRGBA8888(IrisFColor* pc, IrisRGBA8888 rgba)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = (f32)(rgba.r) / 0xFF;
	pc->g = (f32)(rgba.g) / 0xFF;
	pc->b = (f32)(rgba.b) / 0xFF;
	pc->a = (f32)(rgba.a) / 0xFF;
	return pc;
}

/**
 * @brief	ARGB8888J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromBGRA8888(IrisFColor* pc, IrisBGRA8888 argb)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = (f32)(argb.r) / 0xFF;
	pc->g = (f32)(argb.g) / 0xFF;
	pc->b = (f32)(argb.b) / 0xFF;
	pc->a = (f32)(argb.a) / 0xFF;
	return pc;
}

/**
 * @brief	RGBA4444J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromRGBA4444(IrisFColor* pc, IrisRGBA4444 rgba)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = (f32)(rgba.r) / 0xF;
	pc->g = (f32)(rgba.g) / 0xF;
	pc->b = (f32)(rgba.b) / 0xF;
	pc->a = (f32)(rgba.a) / 0xF;
	return pc;
}

/**
 * @brief	RGBA5551J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgba	= ̓J[
 * @param [in]	q		= 臒l
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromRGBA5551(IrisFColor* pc, IrisRGBA5551 rgba, f32 q)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = (f32)(rgba.r) / 0x1F;
	pc->g = (f32)(rgba.g) / 0x1F;
	pc->b = (f32)(rgba.b) / 0x1F;
	pc->a = rgba.a ? q : 0.0f;
	return pc;
}

/**
 * @brief	RGB555J[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	rgb		= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromRGB565(IrisFColor* pc, IrisRGB565 rgb, f32 q)
{
	MATH_FPU_NULLASSERT( pc );
	pc->r = (f32)(rgb.r) / 0x1F;
	pc->g = (f32)(rgb.g) / 0x3F;
	pc->b = (f32)(rgb.b) / 0x1F;
	pc->a = q;
	return pc;
}

/**
 * @brief	CMYJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	pcmy	= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromCMY(IrisFColor* pc, const IrisFCMY* pcmy, f32 q)
{
	MATH_FPU_NULLASSERT( pc );
	MATH_FPU_NULLASSERT( pcmy );
	pc->r = 1.0f - pcmy->c;
	pc->g = 1.0f - pcmy->m;
	pc->b = 1.0f - pcmy->y;
	pc->a = q;
	return pc;
}

/**
 * @brief	CMYKJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	pcmy	= ̓J[
 * @param [in]	q		= At@
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromCMYK(IrisFColor* pc, const IrisFCMYK* pcmyk, f32 q)
{
	MATH_FPU_NULLASSERT( pc );
	MATH_FPU_NULLASSERT( pcmyk );
	f32 tk = 1.0f - pcmyk->k;
	pc->r = 1.0f - min(1.0f, F32_Mul(pcmyk->c, tk) + pcmyk->k);
	pc->g = 1.0f - min(1.0f, F32_Mul(pcmyk->m, tk) + pcmyk->k);
	pc->b = 1.0f - min(1.0f, F32_Mul(pcmyk->y, tk) + pcmyk->k);
	pc->a = q;
	return pc;
}

/**
 * @brief	HSVJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	phsv	= ̓J[
 * @param [in]	qa		= At@
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromHSV(IrisFColor* pc, const IrisFHSV* phsv, f32 qa)
{
	MATH_FPU_NULLASSERT( pc );
	MATH_FPU_NULLASSERT( phsv );
	if( phsv->s == 0.0f )
	{
		pc->r = pc->g = pc->b = phsv->v;
	}
	else
	{
		f32 V = phsv->v;
		f32 th = phsv->h * 6;
		s32 iH = (s32)th;
		f32 f = th - (f32)(iH);
		f32 p = F32_Mul(V, 1.0f - phsv->s);
		f32 q = F32_Mul(V, 1.0f - F32_Mul(f, phsv->s));
		f32 t = F32_Mul(V, 1.0f - F32_Mul(1.0f - f, phsv->s));
		f32 r=0, g=0, b=0;
		switch(iH)
		{
		case 0:
			r = V; g = t; b = p;	break;
		case 1:
			r = q; g = V; b = p;	break;
		case 2:
			r = p; g = V; b = t;	break;
		case 3:
			r = p; g = q; b = V;	break;
		case 4:
			r = t; g = p; b = V;	break;
		case 5:
			r = V; g = p; b = q;	break;
		default:
			IRIS_ASSERT( iH >= 0 && iH < 6 );		break;
		}
		pc->r = r;
		pc->g = g;
		pc->b = b;
	}
	pc->a = qa;
	return pc;
}

/**
 * @brief	HSVJ[^ɕϊ
 * @param [out]	pc		= o̓J[
 * @param [in]	phsv	= ̓J[
 * @param [in]	qa		= At@
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorFromMunsell(IrisFColor* pc, const IrisMunsellColor* phsv, f32 qa)
{
	MATH_FPU_NULLASSERT( pc );
	MATH_FPU_NULLASSERT( phsv );
	IrisFHSV hsv;
	hsv.h = (phsv->hg * 0.1f - 0.05f) + phsv->h * 0.01f;
	hsv.s = phsv->s * 0.1f;
	hsv.v = phsv->v * 0.1f;
	return FpuColorFromHSV(pc, &hsv, qa);
}

/**
 * @brief	J[^̉Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorAdd(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + pc2->r;
	pc0->g = pc1->g + pc2->g;
	pc0->b = pc1->b + pc2->b;
	pc0->a = pc1->a + pc2->a;
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorAddRGB(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + pc2->r;
	pc0->g = pc1->g + pc2->g;
	pc0->b = pc1->b + pc2->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̌Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorSub(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r - pc2->r;
	pc0->g = pc1->g - pc2->g;
	pc0->b = pc1->b - pc2->b;
	pc0->a = pc1->a - pc2->a;
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorSubRGB(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r - pc2->r;
	pc0->g = pc1->g - pc2->g;
	pc0->b = pc1->b - pc2->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̏Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorMul(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = F32_Mul(pc1->r, pc2->r);
	pc0->g = F32_Mul(pc1->g, pc2->g);
	pc0->b = F32_Mul(pc1->b, pc2->b);
	pc0->a = F32_Mul(pc1->a, pc2->a);
	return pc0;
}

/**
 * @brief	J[^RGBZ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorMulRGB(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = F32_Mul(pc1->r, pc2->r);
	pc0->g = F32_Mul(pc1->g, pc2->g);
	pc0->b = F32_Mul(pc1->b, pc2->b);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^1̕␔(RGBA)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorNeg(IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = 1.0f - pc1->r;
	pc0->g = 1.0f - pc1->g;
	pc0->b = 1.0f - pc1->b;
	pc0->a = 1.0f - pc1->a;
	return pc0;
}

/**
 * @brief	J[^1̕␔(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorNegRGB(IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = 1.0f - pc1->r;
	pc0->g = 1.0f - pc1->g;
	pc0->b = 1.0f - pc1->b;
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	2̃J[Ԃ̓}
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @param [in]	t		= 
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorLerp(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2, f32 t)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + F32_Mul(t, (pc2->r - pc1->r));
	pc0->g = pc1->g + F32_Mul(t, (pc2->g - pc1->g));
	pc0->b = pc1->b + F32_Mul(t, (pc2->b - pc1->b));
	pc0->a = pc1->a + F32_Mul(t, (pc2->a - pc1->a));
	return pc0;
}

/**
 * @brief	2̃J[Ԃ̓}(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @param [in]	t		= 
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorLerpRGB(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2, f32 t)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = pc1->r + F32_Mul(t, (pc2->r - pc1->r));
	pc0->g = pc1->g + F32_Mul(t, (pc2->g - pc1->g));
	pc0->b = pc1->b + F32_Mul(t, (pc2->b - pc1->b));
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃XJ[l̏Z
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	s		= XJ[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorScale(IrisFColor* pc0, const IrisFColor* pc1, f32 s)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = F32_Mul(pc1->r, s);
	pc0->g = F32_Mul(pc1->g, s);
	pc0->b = F32_Mul(pc1->b, s);
	pc0->a = F32_Mul(pc1->a, s);
	return pc0;
}

/**
 * @brief	J[^̃XJ[l̏Z(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	s		= XJ[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorScaleRGB(IrisFColor* pc0, const IrisFColor* pc1, f32 s)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = F32_Mul(pc1->r, s);
	pc0->g = F32_Mul(pc1->g, s);
	pc0->b = F32_Mul(pc1->b, s);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̃Nv
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	min		= ŏ
 * @param [in]	max		= ő
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorClamp(IrisFColor* pc0, const IrisFColor* pc1, f32 min, f32 max)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = (pc1->r < min) ? min : ((pc1->r > max) ? max : pc1->r);
	pc0->g = (pc1->g < min) ? min : ((pc1->g > max) ? max : pc1->g);
	pc0->b = (pc1->b < min) ? min : ((pc1->b > max) ? max : pc1->b);
	pc0->a = (pc1->a < min) ? min : ((pc1->a > max) ? max : pc1->a);
	return pc0;
}

/**
 * @brief	J[^̃Nv(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	min		= ŏ
 * @param [in]	max		= ő
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorClampRGB(IrisFColor* pc0, const IrisFColor* pc1, f32 min, f32 max)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->r = (pc1->r < min) ? min : ((pc1->r > max) ? max : pc1->r);
	pc0->g = (pc1->g < min) ? min : ((pc1->g > max) ? max : pc1->g);
	pc0->b = (pc1->b < min) ? min : ((pc1->b > max) ? max : pc1->b);
	pc0->a = pc1->a;
	return pc0;
}

/**
 * @brief	J[^̊evf̑傫Ԃ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorMax(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = (pc1->r > pc2->r) ? pc1->r : pc2->r;
	pc0->g = (pc1->g > pc2->g) ? pc1->g : pc2->g;
	pc0->b = (pc1->b > pc2->b) ? pc1->b : pc2->b;
	pc0->a = (pc1->a > pc2->a) ? pc1->a : pc2->a;
	return pc0;
}

/**
 * @brief	J[^̊evf̏Ԃ
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @param [in]	pc2		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorMin(IrisFColor* pc0, const IrisFColor* pc1, const IrisFColor* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->r = (pc1->r < pc2->r) ? pc1->r : pc2->r;
	pc0->g = (pc1->g < pc2->g) ? pc1->g : pc2->g;
	pc0->b = (pc1->b < pc2->b) ? pc1->b : pc2->b;
	pc0->a = (pc1->a < pc2->a) ? pc1->a : pc2->a;
	return pc0;
}

/**
 * @brief	J[^ǂԂ
 * @param [in]	pc0		= ̓J[
 * @param [in]	pc1		= ̓J[
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool		FpuColorIsEqual(const IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	if( pc0->r == pc1->r && pc0->g == pc1->g && pc0->b == pc1->b && pc0->a == pc1->a )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	J[^ǂԂ
 * @param [in]	pc		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisBool		FpuColorIsZero(const IrisFColor* pc)
{
	MATH_FPU_NULLASSERT( pc );
	const IrisIVec4* pv = (const IrisIVec4*)pc;
	if( ((pv->x | pv->y | pv->z | pv->w) & 0x7FFFFFFF)==0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	J[^̐K
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorNormalize(IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	f32 q = pc1->r;
	if(q < pc1->g) q = pc1->g;
	if(q < pc1->b) q = pc1->b;
	if(q < pc1->a) q = pc1->a;
	if(q != 0.0f) q = F32_Div(1.0f, q);

	pc0->r = F32_Mul(pc1->r, q);
	pc0->g = F32_Mul(pc1->g, q);
	pc0->b = F32_Mul(pc1->b, q);
	pc0->a = F32_Mul(pc1->a, q);
	return pc0;
}

/**
 * @brief	J[^̐K(RGB)
 * @param [out]	pc0		= o̓J[
 * @param [in]	pc1		= ̓J[
 * @return	o̓J[
*/
IRIS_FPU_INLINE IrisFColor*	FpuColorNormalizeRGB(IrisFColor* pc0, const IrisFColor* pc1)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	f32 q = pc1->r;
	if(q < pc1->g) q = pc1->g;
	if(q < pc1->b) q = pc1->b;
	if(q != 0.0f) q = F32_Div(1.0f, q);

	pc0->r = F32_Mul(pc1->r, q);
	pc0->g = F32_Mul(pc1->g, q);
	pc0->b = F32_Mul(pc1->b, q);
	pc0->a = pc1->a;
	return pc0;
}


}	// end of namespace math
}	// end of namespace iris

#endif
