//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathComplex.inl
 * @brief		fZt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathComplex_inl_
#define INCG_IRIS_MathComplex_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	f̐
 * @param [out]	pc0	= o͕f
 * @param [in]	re	= 
 * @param [in]	im	= 
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex* FpuComplexSet(IrisFComplex* pc0, f32 re, f32 im)
{
	MATH_FPU_NULLASSERT( pc0 );
	pc0->re = re;
	pc0->im = im;
	return pc0;
}

/**
 * @brief	[f̐
 * @param [out]	pc0	= o͕f
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex* FpuComplexZero(IrisFComplex* pc0)
{
	MATH_FPU_NULLASSERT( pc0 );
	pc0->re = 0.0f;
	pc0->im = 0.0f;
	return pc0;
}

/**
 * @brief	f̉Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexAdd(IrisFComplex* pc0, const IrisFComplex* pc1, const IrisFComplex* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->re = pc1->re + pc2->re;
	pc0->im = pc1->im + pc2->im;
	return pc0;
}

/**
 * @brief	f̌Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񌸕f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexSub(IrisFComplex* pc0, const IrisFComplex* pc1, const IrisFComplex* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->re = pc1->re - pc2->re;
	pc0->im = pc1->im - pc2->im;
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 敡f
 * @param [in]	pc2	= 敡f
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexMul(IrisFComplex* pc0, const IrisFComplex* pc1, const IrisFComplex* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	pc0->re = F32_Mul(pc1->re, pc2->re) - F32_Mul(pc1->im, pc2->im);
	pc0->im = F32_Mul(pc1->im, pc2->re) + F32_Mul(pc1->re, pc2->im);
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񏜕f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexDiv(IrisFComplex* pc0, const IrisFComplex* pc1, const IrisFComplex* pc2)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	MATH_FPU_NULLASSERT( pc2 );
	f32 re = pc2->re;
	f32 im = pc2->im;
	if( FpuIsNaN(re) || FpuIsNaN(im) )
	{
		pc0->re = FpuPositiveSNaNF(pc2->re);
		pc0->im = pc1->re;
	}
	else if( (re < 0.0f ? -re : +re) < (im < 0.0f ? -im : +im) )
	{
		f32 wr = F32_Div(pc2->im, pc2->re);
		f32 wd = pc2->re + F32_Mul(wr, pc2->im);

		if( FpuIsNaN(wd) || wd == 0.0f )
		{
			pc0->re = FpuPositiveSNaNF(pc2->re);
			pc0->im = pc1->re;
		}
		else
		{
			f32 tmp = F32_Div(pc1->re + F32_Mul(pc1->im, wr), wd);
			pc0->im = F32_Div(pc1->im - F32_Mul(pc1->re, wr), wd);
			pc0->re = tmp;
		}
	}
	else if( im == 0.0f )
	{
		pc0->re = FpuPositiveSNaNF(pc2->re);
		pc0->im = pc1->re;
	}
	else
	{
		f32 wr = F32_Div(pc2->re, pc2->im);
		f32 wd = pc2->im + F32_Mul(wr, pc2->re);

		if( FpuIsNaN(wd) || wd == 0.0f )
		{
			pc0->re = FpuPositiveSNaNF(pc2->re);
			pc0->im = pc1->re;
		}
		else
		{
			f32 tmp = F32_Div(F32_Mul(pc1->re, wr) + pc1->im, wd);
			pc0->im = F32_Div(F32_Mul(pc1->im, wr) - pc1->re, wd);
			pc0->re = tmp;
		}
	}
	return pc0;
}

/**
 * @brief	f̉Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexAddReal(IrisFComplex* pc0, const IrisFComplex* pc1, f32 re)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->re = pc1->re + re;
	return pc0;
}

/**
 * @brief	f̌Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񌸕f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexSubReal(IrisFComplex* pc0, const IrisFComplex* pc1, f32 re)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->re = pc1->re - re;
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 敡f
 * @param [in]	re	= 搔
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexMulReal(IrisFComplex* pc0, const IrisFComplex* pc1, f32 re)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->re = F32_Div(pc1->re, re);
	pc0->im = F32_Div(pc1->im, re);
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񏜕f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_FPU_INLINE IrisFComplex*	FpuComplexDivReal(IrisFComplex* pc0, const IrisFComplex* pc1, f32 re)
{
	MATH_FPU_NULLASSERT( pc0 );
	MATH_FPU_NULLASSERT( pc1 );
	pc0->re = F32_Div(pc1->re, re);
	pc0->im = F32_Div(pc1->im, re);
	return pc0;
}

}	// end of namespace math
}	// end of namespace iris

#endif
