//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathFpu.inl
 * @brief		lZt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathFpu_inl_
#define INCG_IRIS_MathFpu_inl_

namespace iris {
namespace math
{

//======================================================================
// function

/**
 * @brief	_̐ΒlԂ
 * @param [in]	fs	= l
 * @return	Βl
*/
IRIS_FPU_INLINE f32		FpuAbs(f32 fs)
{
	return fabsf(fs);
}

/**
 * @brief	XJ[l̐؂グ
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE s32			FpuCeil(f32 fs)
{
	return (s32)ceilf(fs);
}

/**
 * @brief	XJ[l̐؂̂
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE s32			FpuFloor(f32 fs)
{
	return (s32)floorf(fs);
}

/**
 * @brief	XJ[ől
 * @param [in]	a	= l
 * @param [in]	b	= l
 * @return	ől
*/
IRIS_FPU_INLINE f32			FpuMax(f32 a, f32 b)
{
	return (a > b) ? a : b;
}

/**
 * @brief	XJ[ŏl
 * @param [in]	a	= l
 * @param [in]	b	= l
 * @return	ŏl
*/
IRIS_FPU_INLINE f32			FpuMin(f32 a, f32 b)
{
	return (a < b) ? a : b;
}

/**
 * @brief	̕Ԃ
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE f32			FpuNeg(f32 fs)
{
	return -fs;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE s32			FpuRound(f32 fs)
{
	return (s32)roundf(fs);
}

/**
 * @brief	̋t
 * @param [in]	fs	= l
 * @return	̋t
*/
IRIS_FPU_INLINE f32			FpuRsqrt(f32 fs)
{
	return 1.0f / FpuSqrt(fs);
}

/**
 * @brief	
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE f32			FpuSqrt(f32 fs)
{
	return sqrtf(fs);
}

/**
 * @brief	0ۂ
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE s32			FpuTrunc(f32 fs)
{
	return (s32)fs;
}

/**
 * @brief	̏]
 * @param [in]	fs	= l
 * @param [in]	fd	= l
 * @return	]
*/
IRIS_FPU_INLINE f32			FpuFmod(f32 fs, f32 fd)
{
	f32 v = (f32)FpuTrunc( fs / fd );
	return fs - v * fd;
}

/**
 * @brief	̎擾
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE f32			FpuFrac(f32 fs)
{
	f32 v = (f32)FpuTrunc(fs);
	return fs - v;
}

/**
 * @brief	^Ƃĕ]
 * @param [in]	ui	= rbgl
 * @return	
*/
IRIS_FPU_INLINE f32			FpuReinterpretFloat(u32 ui)
{
	IrisFInt fi; fi.iv = ui;
	return fi.fv;
}

/**
 * @brief	̃rbg\Ԃ
 * @param [in]	fs	= l
 * @return	rbg
*/
IRIS_FPU_INLINE u32			FpuReinterpretUint(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	return fi.iv;
}

/**
 * @brief	_lł邩Ԃ
 * @param [in]	fs1	= l
 * @param [in]	fs2	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool	FpuIsEqual(f32 fs1, f32 fs2)
{
	return fs1 == fs2 ? IRIS_TRUE : IRIS_FALSE;
}

/**
 * @brief	,̕f32ŕԂ
 * @param [in]	fs	= l
 * @return	,̕
*/
IRIS_FPU_INLINE f32			FpuSignFloat(f32 fs)
{
	return (fs < 0.0f) ? -1.0f : (fs > 0.0f) ? 1.0f : 0.0f;
}

/**
 * @brief	,̕s32ŕԂ
 * @param [in]	fs	= l
 * @return	,̕
*/
IRIS_FPU_INLINE s32			FpuSignInt(f32 fs)
{
	return (fs < 0.0f) ? -1 : (fs > 0.0f) ? 1 : 0;
}

/**
 * @brief	+0擾
 * @return	+0
*/
IRIS_FPU_INLINE f32			FpuPositiveZero(void)
{
	return 0.0f;
}

/**
 * @brief	-0擾
 * @return	-0
*/
IRIS_FPU_INLINE f32			FpuNegativeZero(void)
{
	IrisFInt fi; fi.iv = 0x80000000;
	return fi.fv;
}

/**
 * @brief	0ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool	FpuIsZero(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	return (fi.iv & 0x7FFFFFFFU) == 0 ? IRIS_TRUE : IRIS_FALSE;
}

/**
 * @brief	+0ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool	FpuIsPositiveZero(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	return (fi.iv == 0) ? IRIS_TRUE : IRIS_FALSE;
}

/**
 * @brief	-0ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool	FpuIsNegativeZero(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	return (fi.iv == 0x80000000) ? IRIS_TRUE : IRIS_FALSE;
}

/**
 * @brief	񐳋Kł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE s32			FpuIsDenormal(f32 fs)
{
	s32 ret = 0;
	IrisFInt fi; fi.fv = fs;
	u32 t0 = fi.iv;
	u32 t1 = t0 & 0x7F800000;
	fi.iv  = t0 | 0x7F800000;
	ret = fi.iv >> 30 ? +1 : -1;
	t0 = t0 << 9;
	ret = t1 != 0 ? 0 : ret;
	ret = t0 == 0 ? 0 : ret;
	return ret;
}

/**
 * @brief	0܂͔񐳋Kł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE s32			FpuIsZeroOrDenormal(f32 fs)
{
	s32 ret = 0;
	IrisFInt fi; fi.fv = fs;
	u32 t0 = fi.iv;
	u32 t1 = t0 & 0x7F800000;
	fi.iv  = t0 | 0x7F800000;
	ret = fi.iv >> 30 ? +1 : -1;
	t0 = t0 << 9;
	ret = t1 != 0 ? 0 : ret;
	return ret;
}

/**
 * @brief	+擾
 * @return	+
*/
IRIS_FPU_INLINE f32			FpuPositiveInf(void)
{
	IrisFInt fi; fi.iv = MATH_F32_P_INF_BITS;
	return fi.fv;
}

/**
 * @brief	-擾
 * @return	-
*/
IRIS_FPU_INLINE f32			FpuNegativeInf(void)
{
	IrisFInt fi; fi.iv = MATH_F32_N_INF_BITS;
	return fi.fv;
}

/**
 * @brief	ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE s32			FpuIsInf(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	if( fi.iv == MATH_F32_P_INF_BITS ) return 1;
	if( fi.iv == MATH_F32_N_INF_BITS )	return -1;
	return 0;
}

/**
 * @brief	+NaN擾
 * @return	+NaN
*/
IRIS_FPU_INLINE f32			FpuPositiveNaN(void)
{
	IrisFInt fi; fi.iv = MATH_F32_P_NAN_BITS;
	return fi.fv;
}

/**
 * @brief	+NaN擾
 * @return	+NaN
*/
IRIS_FPU_INLINE f32			FpuNegativeNaN(void)
{
	IrisFInt fi; fi.iv = MATH_F32_N_NAN_BITS;
	return fi.fv;
}

/**
 * @brief	+QNaN擾
 * @return	+QNaN
*/
IRIS_FPU_INLINE f32			FpuPositiveQNaN(void)
{
	IrisFInt fi; fi.iv = MATH_F32_P_QNAN_BITS;
	return fi.fv;
}

/**
 * @brief	-QNaN擾
 * @return	-QNaN
*/
IRIS_FPU_INLINE f32			FpuNegativeQNaN(void)
{
	IrisFInt fi; fi.iv = MATH_F32_N_QNAN_BITS;
	return fi.fv;
}

/**
 * @brief	+SNaN擾
 * @param [in]	signal = 
 * @return	+SNaN
*/
IRIS_FPU_INLINE f32			FpuPositiveSNaN(u32 signal)
{
	IrisFInt fi; fi.iv = MATH_F32_P_SNAN_BITS(signal);
	return fi.fv;
}

/**
 * @brief	+SNaN擾
 * @param [in]	signal = 
 * @return	+SNaN
*/
IRIS_FPU_INLINE f32			FpuPositiveSNaNF(f32 signal)
{
	IrisFInt fi; fi.fv = signal; fi.iv = MATH_F32_N_SNAN_BITS(fi.iv);
	return fi.fv;
}

/**
 * @brief	-SNaN擾
 * @param [in]	signal = 
 * @return	-SNaN
*/
IRIS_FPU_INLINE f32			FpuNegativeSNaN(u32 signal)
{
	IrisFInt fi; fi.iv = MATH_F32_N_SNAN_BITS(signal);
	return fi.fv;
}

/**
 * @brief	-SNaN擾
 * @param [in]	signal = 
 * @return	-SNaN
*/
IRIS_FPU_INLINE f32			FpuNegativeSNaNF(f32 signal)
{
	IrisFInt fi; fi.fv = signal; fi.iv = MATH_F32_N_SNAN_BITS(fi.iv);
	return fi.fv;
}

/**
 * @brief	񐔂ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE s32			FpuIsNaN(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	s32 ret = 0x807FFFFF;
	u32 t0 = fi.iv;
	u32 t1 = fi.iv << 1;
	ret = (s32)(ret & t0);
	t1 >>= 24;
	t0 <<= 9;
	t1 = (t1<0xFF);
	ret = (t0==0) ? 0 : ret;
	ret = (t1!=0) ? 0 : ret;
	return ret;
}

/**
 * @brief	or񐔂ł邩Ԃ
 * @param [in]	fs	= l
 * @return	^Ul
*/
IRIS_FPU_INLINE s32			FpuIsInfOrNaN(f32 fs)
{
	IrisFInt fi; fi.fv = fs;
	s32 ret = (s32)fi.iv;
	u32 t0 = fi.iv << 1;
	ret >>= 30;
	t0 >>= 24;
	ret |= 0x00000001;
	t0 = (t0<0xFF);
	ret = (t0!=0) ? 0 : ret;
	return ret;
}

/**
 * @brief	ʑ-΁`΂ɐK
 * @param [in]	fs	= l
 * @return	
*/
IRIS_FPU_INLINE f32			FpuNormalizePhase(f32 fs)
{
	f32 fd = fs / MATH_F32_2PI;
	s32  f0 = (s32)( fd + 0.5f );
	fd = (f32)f0;
	fd = fd * MATH_F32_2PI;
	fd = fs - fd;
	return fd;
}

/**
 * @brief	(TC)
 * @param [in]	fs	= WAl
 * @return	
*/
IRIS_FPU_INLINE f32			FpuSin(f32 fs)
{
	return sinf(fs);
}

/**
 * @brief	](RTC)vZ
 * @param [in]	fs	= WAl
 * @return	]
*/
IRIS_FPU_INLINE f32			FpuCos(f32 fs)
{
	return cosf(fs);
}

/**
 * @brief	t(A[N^WFg)vZ
 * @param [in]	fs	= l
 * @return	t
*/
IRIS_FPU_INLINE f32			FpuAtan(f32 fs)
{
	return atanf(fs);
}

/**
 * @brief	t(A[NTC)vZ
 * @param [in]	fs	= l
 * @return	t
*/
IRIS_FPU_INLINE f32			FpuAsin(f32 fs)
{
	return asinf(fs);
}

/**
 * @brief	t](A[NRTC)vZ
 * @param [in]	fs	= l
 * @return	t]
*/
IRIS_FPU_INLINE f32			FpuAcos(f32 fs)
{
	return acosf(fs);
}

/**
 * @brief	Rΐ
 * @param [in]	fs	= l
 * @return	Rΐ
*/
IRIS_FPU_INLINE f32			FpuLog(f32 fs)
{
	return logf(fs);
}

/**
 * @brief	2Ƃΐ
 * @param [in]	fs	= l
 * @return	2Ƃΐ
*/
IRIS_FPU_INLINE f32			FpuLog2(f32 fs)
{
#if 1
	return logf(fs) / logf(2.0f);
#else
	static const f32 log2Poly[] = {
		 6051103.0f / (1024.0f * 1024.0f *  2.0f), /* arround 2.0 */
		 8065844.0f / (1024.0f * 1024.0f *  8.0f), /* arround 2.0/3.0 */
		20040025.0f / (1024.0f * 1024.0f * 32.0f), /* arround 2.0/5.0 */
	};
	union {
		f32 f;
		s32 i;
	} fi;
	s32 a, b, exponent;
	f32 y, z;
	fi.f = x;
	if(fi.i <= 0) 
	{
		if ((fi.i & 0x7fffffff) == 0) 
		{
			fi.i = 0xff800000; /* -INF */
			return fi.f;
		}
		fi.i = 0xff80ffff; /* Signaling NaN */
		return fi.f;
	}

	b = (fi.i & 0x007fffff);
	a = b - (1<<23);
	exponent = fi.i;
	if (b <= 0x3504F3)
	{
		/*J x ̉ SQRT2/2 ȉ */
		a = b;
		b = b - (1 << 23);
		exponent -= (1<<23);
	}
	b = b - (1 << 23) + (1<<25);

	x = (f32)(a) / (f32)(b);
	z = (f32)((exponent>>23)-126);
	y = x * x;
	return z + x * (((log2Poly[2] * y
		+ log2Poly[1]) * y)
		+ log2Poly[0]);
#endif
}

/**
 * @brief	pΐ
 * @param [in]	fs	= l
 * @return	pΐ
*/
IRIS_FPU_INLINE f32			FpuLog10(f32 fs)
{
	return log10f(fs);
}

/**
 * @brief	eƂw
 * @param [in]	fs	= l
 * @return	eƂw
*/
IRIS_FPU_INLINE f32			FpuExp(f32 fs)
{
	return expf(fs);
}

/**
 * @brief	2Ƃw
 * @param [in]	fs	= l
 * @return	eƂw
*/
IRIS_FPU_INLINE f32			FpuExp2(f32 fs)
{
	return powf(2.0f, fs);
}

/**
 * @brief	10Ƃw
 * @param [in]	fs	= l
 * @return	eƂw
*/
IRIS_FPU_INLINE f32			FpuExp10(f32 fs)
{
	return powf(10.0f, fs);
}

/**
 * @brief	w
 * @param [in]	x	= l
 * @param [in]	y	= l
 * @return	w
*/
IRIS_FPU_INLINE f32			FpuPow(f32 x, f32 y)
{
	return powf(x, y);
}

/**
 * @brief	f32  f64ɕϊ
 * @param [in]	fs	= l
 * @return	f64
*/
IRIS_FPU_INLINE f64			FpuF32ToF64(f32 fs)
{
	return (f64)fs;
}

/**
 * @brief	f64  f32ɕϊ
 * @param [in]	fs	= l
 * @return	f32
*/
IRIS_FPU_INLINE f32			FpuF64ToF32(f64 fs)
{
	return (f32)fs;
}

}	// end of namespace math
}	// end of namespace iris

#endif
