//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathVector4.inl
 * @brief		4DxNgt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathVector4_inl_
#define INCG_IRIS_MathVector4_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	vfݒ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @param [in]	z	= zl
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Set(IrisFVec4* pv0, f32 x, f32 y, f32 z, f32 w)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y; pv0->z = z; pv0->w = w;
	return pv0;
}

/**
 * @brief	vfݒ(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @param [in]	z	= zl
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4SetXYZ(IrisFVec4* pv0, f32 x, f32 y, f32 z)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y; pv0->z = z;
	return pv0;
}

/**
 * @brief	vfRs[
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Copy(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x; pv0->y = pv1->y; pv0->z = pv1->z; pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	+0.0fɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4PositiveZero(IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = pv0->z = pv0->w = 0.0f;
	return pv0;
}

/**
 * @brief	-0.0fɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4NegativeZero(IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	IrisVec4* v = (IrisVec4*)pv0;
	v->iv.x = v->iv.y = v->iv.z = v->iv.w = 0x80000000;
	return pv0;
}

/**
 * @brief	؂グ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisIVec4*	FpuVec4Ceil(IrisIVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F32_Ceil(pv1->x) );
	pv0->y = (s32)( F32_Ceil(pv1->y) );
	pv0->z = (s32)( F32_Ceil(pv1->z) );
	pv0->w = (s32)( F32_Ceil(pv1->w) );
	return pv0;
}

/**
 * @brief	0ۂ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisIVec4*	FpuVec4Trunc(IrisIVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F32_Trunc(pv1->x) );
	pv0->y = (s32)( F32_Trunc(pv1->y) );
	pv0->z = (s32)( F32_Trunc(pv1->z) );
	pv0->w = (s32)( F32_Trunc(pv1->w) );
	return pv0;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisIVec4*	FpuVec4Round(IrisIVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F32_Round(pv1->x) );
	pv0->y = (s32)( F32_Round(pv1->y) );
	pv0->z = (s32)( F32_Round(pv1->z) );
	pv0->w = (s32)( F32_Round(pv1->w) );
	return pv0;
}

/**
 * @brief	؂̂
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisIVec4*	FpuVec4Floor(IrisIVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F32_Floor(pv1->x) );
	pv0->y = (s32)( F32_Floor(pv1->y) );
	pv0->z = (s32)( F32_Floor(pv1->z) );
	pv0->w = (s32)( F32_Floor(pv1->w) );
	return pv0;
}

/**
 * @brief	xNgϊ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4FromIVec4(IrisFVec4* pv0, const IrisIVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (f32)( pv1->x );
	pv0->y = (f32)( pv1->y );
	pv0->z = (f32)( pv1->z );
	pv0->w = (f32)( pv1->w );
	return pv0;
}

/**
 * @brief	xNg̉Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Add(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	pv0->z = pv1->z + pv2->z;
	pv0->w = pv1->w + pv2->w;
	return pv0;
}

/**
 * @brief	xNg̉Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4AddXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	pv0->z = pv1->z + pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Sub(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	pv0->z = pv1->z - pv2->z;
	pv0->w = pv1->w - pv2->w;
	return pv0;
}

/**
 * @brief	xNǧZ(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4SubXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	pv0->z = pv1->z - pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Mul(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Mul(pv1->x, pv2->x);
	pv0->y = F32_Mul(pv1->y, pv2->y);
	pv0->z = F32_Mul(pv1->z, pv2->z);
	pv0->w = F32_Mul(pv1->w, pv2->w);
	return pv0;
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4MulXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Mul(pv1->x, pv2->x);
	pv0->y = F32_Mul(pv1->y, pv2->y);
	pv0->z = F32_Mul(pv1->z, pv2->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Div(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Div(pv1->x, pv2->x);
	pv0->y = F32_Div(pv1->y, pv2->y);
	pv0->z = F32_Div(pv1->z, pv2->z);
	pv0->w = F32_Div(pv1->w, pv2->w);
	return pv0;
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4DivXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Div(pv1->x, pv2->x);
	pv0->y = F32_Div(pv1->y, pv2->y);
	pv0->z = F32_Div(pv1->z, pv2->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̕]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Neg(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
#if 0
	pv0->x = -pv1->x;
	pv0->y = -pv1->y;
	pv0->z = -pv1->z;
	pv0->w = -pv1->w;
#else
	IrisVec4* v0 = (IrisVec4*)pv0;
	const IrisVec4* v1 = (const IrisVec4*)pv1;
	v0->iv.x = (s32)(v1->iv.x ^ 0x80000000U);
	v0->iv.y = (s32)(v1->iv.y ^ 0x80000000U);
	v0->iv.z = (s32)(v1->iv.z ^ 0x80000000U);
	v0->iv.w = (s32)(v1->iv.w ^ 0x80000000U);
#endif
	return pv0;
}

/**
 * @brief	xNg̊evf̐Βl擾
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Abs(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
#if 0
	pv0->x = F32_ABS(pv1->x);
	pv0->y = F32_ABS(pv1->y);
	pv0->z = F32_ABS(pv1->z);
	pv0->w = F32_ABS(pv1->w);
#else
	IrisVec4* v0 = (IrisVec4*)pv0;
	const IrisVec4* v1 = (const IrisVec4*)pv1;
	v0->iv.x = (s32)(v1->iv.x & 0x7FFFFFFFU);
	v0->iv.y = (s32)(v1->iv.y & 0x7FFFFFFFU);
	v0->iv.z = (s32)(v1->iv.z & 0x7FFFFFFFU);
	v0->iv.w = (s32)(v1->iv.w & 0x7FFFFFFFU);
#endif
	return pv0;
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Lerp(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + F32_Mul(t, (pv2->x - pv1->x));
	pv0->y = pv1->y + F32_Mul(t, (pv2->y - pv1->y));
	pv0->z = pv1->z + F32_Mul(t, (pv2->z - pv1->z));
	pv0->w = pv1->w + F32_Mul(t, (pv2->w - pv1->w));
	return pv0;
}

/**
* @brief	xNg̊Ԃ̓}(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4LerpXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + F32_Mul(t, (pv2->x - pv1->x));
	pv0->y = pv1->y + F32_Mul(t, (pv2->y - pv1->y));
	pv0->z = pv1->z + F32_Mul(t, (pv2->z - pv1->z));
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Scale(IrisFVec4* pv0, const IrisFVec4* pv1, f32 s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = F32_Mul(pv1->x, s);
	pv0->y = F32_Mul(pv1->y, s);
	pv0->z = F32_Mul(pv1->z, s);
	pv0->w = F32_Mul(pv1->w, s);
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4ScaleXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, f32 s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = F32_Mul(pv1->x, s);
	pv0->y = F32_Mul(pv1->y, s);
	pv0->z = F32_Mul(pv1->z, s);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4ScaleAdd(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, f32 s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Mul(pv1->x, s) + pv2->x;
	pv0->y = F32_Mul(pv1->y, s) + pv2->y;
	pv0->z = F32_Mul(pv1->z, s) + pv2->z;
	pv0->w = F32_Mul(pv1->w, s) + pv2->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4ScaleAddXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, f32 s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Mul(pv1->x, s) + pv2->x;
	pv0->y = F32_Mul(pv1->y, s) + pv2->y;
	pv0->z = F32_Mul(pv1->z, s) + pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Hermite(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pt1
									  , const IrisFVec4* pv2, const IrisFVec4* pt2, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pt1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pt2 );
	f32 t2 = F32_Mul(t, t);
	f32 t3 = F32_Mul(t, t2);
	IrisFVec4 v[4];
	FpuVec4Scale(&v[0], pv1, t);
	FpuVec4Scale(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1.0f));
	FpuVec4Scale(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	FpuVec4Scale(&v[2], pt2, ((     t3) - (    t2)           ));
	FpuVec4Scale(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	FpuVec4Add(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	FpuVec4Add(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	FpuVec4Add(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4HermiteXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pt1
									  , const IrisFVec4* pv2, const IrisFVec4* pt2, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pt1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pt2 );
	f32 t2 = F32_Mul(t, t);
	f32 t3 = F32_Mul(t, t2);
	IrisFVec4 v[4];
	FpuVec4ScaleXYZ(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1.0f));
	FpuVec4ScaleXYZ(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	FpuVec4ScaleXYZ(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	FpuVec4ScaleXYZ(&v[2], pt2, ((     t3) - (    t2)           ));
	FpuVec4AddXYZ(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	FpuVec4AddXYZ(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	FpuVec4AddXYZ(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Bezier(IrisFVec4* pv0, s32 n, const IrisFVec4* pva, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0.0f ) { FpuVec4Copy(pv0, &pva[0]); return pv0; }
	if( t >= 1.0f  ) { FpuVec4Copy(pv0, &pva[n-1]); return pv0; }
	FpuVec4PositiveZero(pv0);
	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisFVec4* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		pv0->x += F32_Mul(pv->x, J);
		pv0->y += F32_Mul(pv->y, J);
		pv0->z += F32_Mul(pv->z, J);
		pv0->w += F32_Mul(pv->w, J);
		tt = F32_Mul(tt, t);
	}
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4BezierXYZ(IrisFVec4* pv0, s32 n, const IrisFVec4* pva, f32 t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0.0f ) { FpuVec4Copy(pv0, &pva[0]); return pv0; }
	if( t >= 1.0f  ) { FpuVec4Copy(pv0, &pva[n-1]); return pv0; }
	FpuVec4PositiveZero(pv0);
	pv0->w = pva[0].w;
	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisFVec4* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		f32 J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (f32)(nn-i)));
		pv0->x += F32_Mul(pv->x, J);
		pv0->y += F32_Mul(pv->y, J);
		pv0->z += F32_Mul(pv->z, J);
		tt = F32_Mul(tt, t);
	}
	return pv0;
}

/**
 * @brief	xNg̃Nv
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Clamp(IrisFVec4* pv0, const IrisFVec4* pv1, f32 min, f32 max)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	pv0->z = (pv1->z < min) ? min : ((pv1->z > max) ? max : pv1->z);
	pv0->w = (pv1->w < min) ? min : ((pv1->w > max) ? max : pv1->w);
	return pv0;
}

/**
 * @brief	xNg̃Nv(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4ClampXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, f32 min, f32 max)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	pv0->z = (pv1->z < min) ? min : ((pv1->z > max) ? max : pv1->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̑傫Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Max(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x > pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y > pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z > pv2->z) ? pv1->z : pv2->z;
	pv0->w = (pv1->w > pv2->w) ? pv1->w : pv2->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̏Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Min(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y < pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z < pv2->z) ? pv1->z : pv2->z;
	pv0->w = (pv1->w < pv2->w) ? pv1->w : pv2->w;
	return pv0;
}

/**
 * @brief	
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
IRIS_FPU_INLINE	f32		FpuVec4InnerProduct(const IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	return F32_Mul(pv0->x, pv1->x) + F32_Mul(pv0->y, pv1->y) + F32_Mul(pv0->z, pv1->z) + F32_Mul(pv0->w, pv1->w);
}

/**
 * @brief	(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
IRIS_FPU_INLINE	f32		FpuVec4InnerProductXYZ(const IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	return F32_Mul(pv0->x, pv1->x) + F32_Mul(pv0->y, pv1->y) + F32_Mul(pv0->z, pv1->z);
}

/**
 * @brief	O(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4OuterProductXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = F32_Mul(pv1->y, pv2->z) - F32_Mul(pv2->y, pv1->z);
	pv0->y = F32_Mul(pv1->z, pv2->x) - F32_Mul(pv2->z, pv1->x);
	pv0->z = F32_Mul(pv1->x, pv2->y) - F32_Mul(pv2->x, pv1->y);
	pv0->w = 0.0f;
	return pv0;
}

/**
 * @brief	xNg̊evf̑a擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̑a
*/
IRIS_FPU_INLINE	f32		FpuVec4Funnel(const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z + pv0->w);
}

/**
 * @brief	xNg̊evf̕ς擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̕
*/
IRIS_FPU_INLINE	f32		FpuVec4Average(const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z + pv0->w) / 4;
}

/**
 * @brief	xNgǂԂ
 * @param [in]	pv0	= rxNg
 * @param [in]	pv1	= rxNg
 * @return	^Ul
*/
IRIS_FPU_INLINE	IrisBool	FpuVec4IsEqual(const IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	if(    (pv0->x != pv1->x)
		|| (pv0->y != pv1->y) 
		|| (pv0->z != pv1->z) 
		|| (pv0->w != pv1->w) ) return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	[xNgǂԂ
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
IRIS_FPU_INLINE	IrisBool	FpuVec4IsZero(const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	const IrisVec4* v = (const IrisVec4*)pv0;
	if( ((v->iv.x | v->iv.y | v->iv.z | v->iv.w) & 0x7FFFFFFFU) == 0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	[xNgǂԂ(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
IRIS_FPU_INLINE	IrisBool	FpuVec4IsZeroXYZ(const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	const IrisVec4* v = (const IrisVec4*)pv0;
	if( ((v->iv.x | v->iv.y | v->iv.z) & 0x7FFFFFFFU) == 0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	,̕IrisFVec4ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4SignFloat(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0.0f) ? -1.0f : (pv1->x > 0.0f) ? 1.0f : 0.0f;
	pv0->y = (pv1->y < 0.0f) ? -1.0f : (pv1->y > 0.0f) ? 1.0f : 0.0f;
	pv0->z = (pv1->z < 0.0f) ? -1.0f : (pv1->z > 0.0f) ? 1.0f : 0.0f;
	pv0->w = (pv1->w < 0.0f) ? -1.0f : (pv1->w > 0.0f) ? 1.0f : 0.0f;
	return (pv0);
}

/**
 * @brief	,̕IrisIVec4ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisIVec4*	FpuVec4SignInt(IrisIVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0.0f) ? -1 : (pv1->x > 0.0f) ? 1 : 0;
	pv0->y = (pv1->y < 0.0f) ? -1 : (pv1->y > 0.0f) ? 1 : 0;
	pv0->z = (pv1->z < 0.0f) ? -1 : (pv1->z > 0.0f) ? 1 : 0;
	pv0->w = (pv1->w < 0.0f) ? -1 : (pv1->w > 0.0f) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	xNg̐K
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4Normalize(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	f32 x = pv1->x, y = pv1->y, z = pv1->z, w = pv1->w;
	f32 q = F32_Sqrt( F32_Mul(x, x) + F32_Mul(y, y) + F32_Mul(z, z) + F32_Mul(w, w) );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0.0f );
	q = F32_Div(1.0f, q);
#else
	if( q != 0.0f ) q = F32_Div(1.0f, q);
#endif
	pv0->x = F32_Mul(x, q);
	pv0->y = F32_Mul(y, q);
	pv0->z = F32_Mul(z, q);
	pv0->w = F32_Mul(w, q);
	return pv0;
}

/**
 * @brief	xNg̐K(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4NormalizeXYZ(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	f32 x = pv1->x, y = pv1->y, z = pv1->z;
	f32 q = F32_Sqrt( F32_Mul(x, x) + F32_Mul(y, y) + F32_Mul(z, z) );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0.0f );
	q = F32_Div(1.0f, q);
#else
	if( q != 0.0f ) q = F32_Div(1.0f, q);
#endif
	pv0->x = F32_Mul(x, q);
	pv0->y = F32_Mul(y, q);
	pv0->z = F32_Mul(z, q);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊el-΁`+΂Ɏ߂
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4NormalizePhase(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	f32 x = pv1->x, y = pv1->y, z = pv1->z;
	f32 q = F32_Sqrt( F32_Mul(x, x) + F32_Mul(y, y) + F32_Mul(z, z) );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0.0f );
	q = F32_Div(F32_PI, q);
#else
	if( q != 0.0f ) q = F32_Div(F32_PI, q);
#endif
	pv0->x = F32_Mul(x, q);
	pv0->y = F32_Mul(y, q);
	pv0->z = F32_Mul(z, q);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̒Ԃ(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @return	xNg̒
*/
IRIS_FPU_INLINE	f32	FpuVec4LengthXYZ(const IrisFVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (F32_Sqrt( F32_Mul(pv0->x, pv0->x) + F32_Mul(pv0->y, pv0->y) + F32_Mul(pv0->z, pv0->z) ));
}

/**
 * @brief	xNg̊Ԃ̋(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	xNg̒
*/
IRIS_FPU_INLINE	f32	FpuVec4DistanceXYZ(const IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	f32 x = pv0->x - pv1->x;
	f32 y = pv0->y - pv1->y;
	f32 z = pv0->z - pv1->z;
	return (F32_Sqrt( F32_Mul(x, x) + F32_Mul(y, y) + F32_Mul(z, z) ));
}

/**
 * @brief	IuWFNg\ʂJɌ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pv3	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4FaceForwardXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, const IrisFVec4* pv3)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pv3 );
	f32 d = F32_Mul(pv2->x, pv3->x) + F32_Mul(pv2->y, pv3->y) + F32_Mul(pv2->z, pv3->z);
	if(d < 0.0f)
	{
		pv0->x = pv1->x;
		pv0->y = pv1->y;
		pv0->z = pv1->z;
		pv0->w = pv1->w;
	} 
	else
	{
		pv0->x = -pv1->x;
		pv0->y = -pv1->y;
		pv0->z = -pv1->z;
		pv0->w = pv1->w;
	}
	return pv0;
}

/**
 * @brief	˃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4ReflectXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	f32 d = F32_Mul(pv1->x, pv2->x) + F32_Mul(pv1->y, pv2->y) + F32_Mul(pv1->z, pv2->z);
	pv0->x = pv1->x - F32_Mul(2 * d, pv2->x);
	pv0->y = pv1->y - F32_Mul(2 * d, pv2->y);
	pv0->z = pv1->z - F32_Mul(2 * d, pv2->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	܃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	eta	= ܗ
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4RefractXYZ(IrisFVec4* pv0, const IrisFVec4* pv1, const IrisFVec4* pv2, f32 eta)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	f32 d, e, f;
	d = F32_Mul(pv1->x, pv2->x) + F32_Mul(pv1->y, pv2->y) + F32_Mul(pv1->z, pv2->z);
	e = 1.0f - F32_Mul(F32_Mul(eta, eta), (1.0f - F32_Mul(d, d)));
	if(e < 0.0f)
	{
		pv0->x = 0.0f;
		pv0->y = 0.0f;
		pv0->z = 0.0f;
	} 
	else
	{
		f = F32_Mul(eta, d) - F32_Sqrt(e);
		pv0->x = F32_Mul(eta, pv1->x) - F32_Mul(f, pv2->x);
		pv0->y = F32_Mul(eta, pv1->y) - F32_Mul(f, pv2->y);
		pv0->z = F32_Mul(eta, pv1->z) - F32_Mul(f, pv2->z);
	}
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE	IrisFVec4*	FpuVec4TruncatePrecision24(IrisFVec4* pv0, const IrisFVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	IrisVec4* v0 = (IrisVec4*)pv0;
	const IrisVec4* v1 = (const IrisVec4*)pv1;
	v0->iv.x = (s32)(v1->iv.x & 0xFFFFFF00);
	v0->iv.y = (s32)(v1->iv.y & 0xFFFFFF00);
	v0->iv.z = (s32)(v1->iv.z & 0xFFFFFF00);
	v0->iv.w = (s32)(v1->iv.w & 0xFFFFFF00);
	return pv0;
}

}	// end of namespace math
}	// end of namespace iris

#endif
