//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathSHA1.cpp
 * @brief		SHA1 t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_MathSHA1_CPP_
#define INCG_IRIS_MathSHAPrivate_CPP_	// private֐gp

//======================================================================
// include
#include "MathSHA1.h"
#include "MathSHACommon.h"
#include "../../misc/iris_allegrex.h"
#include "../../iris_debug.h"

namespace iris {
namespace math
{

//======================================================================
// define
#define COUNT_HIGH			0
#define COUNT_LOW			1

#define MESSAGE_BLK_SIZE	SHA1_MSGBLKSIZE
#define MESSAGE_PAD_SIZE	56	// 64-8

// ]
#define ROTL	iris_allegrex_rotl

#define SHA1AddLength(ctx, len)					\
	(ctx)->uCount[COUNT_LOW] += len				\
	, ((ctx)->uCount[COUNT_LOW] != 0 ) ? 0 :	\
	(++(ctx)->uCount[COUNT_HIGH]				\
	, ((ctx)->uCount[COUNT_HIGH] != 0) ? 0 : 1)

//======================================================================
// declare
// gXtH[֐
static void _SHA1Transform(u32* lpState, const u8* lpBuffer);

//======================================================================
// function
/**********************************************************************//**
 *
 * SHA1pReLXg̏
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA1InitContext(LPSHA1CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	lpContext->uCount[0] = lpContext->uCount[1] = 0;
	lpContext->uIndex = 0;
	lpContext->uState[0] = 0x67452301;
	lpContext->uState[1] = 0xefcdab89;
	lpContext->uState[2] = 0x98badcfe;
	lpContext->uState[3] = 0x10325476;
	lpContext->uState[4] = 0xC3D2E1F0;
	//ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA1pReLXg̃NA
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA1ClearContext(LPSHA1CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	SHA1InitContext(lpContext);
	ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA1vZ
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA1Update(LPSHA1CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	IRIS_ASSERT( lpContext != nullptr );
	IRIS_ASSERT( lpBuffer != nullptr );
	const u8* buf = lpBuffer;
	u16 index = lpContext->uIndex;
	while( uLength-- )
	{
		lpContext->uBuffer[index++] = *buf;
		if( index == MESSAGE_BLK_SIZE )
		{
			_SHA1Transform(lpContext->uState, lpContext->uBuffer);
			index = 0;
		}
		if( SHA1AddLength(lpContext, 8) )
		{
			IRIS_WARNING("message is too long.");
			break;
		}
		++buf;
	}
	lpContext->uIndex = index;
}

/**********************************************************************//**
 *
 * ŏIISHA1vZ
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
*//***********************************************************************/
void	SHA1Final(LPSHA1CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	u16 index = lpContext->uIndex;
	if( index >= MESSAGE_PAD_SIZE )
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_BLK_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
		_SHA1Transform(lpContext->uState, lpContext->uBuffer);
		index = 0;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	else
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	_SHAEncode32(lpContext->uBuffer+MESSAGE_PAD_SIZE, lpContext->uCount, 8);
	_SHA1Transform(lpContext->uState, lpContext->uBuffer);
	lpContext->uIndex = 0;
}

/**********************************************************************//**
 *
 * SHA1o
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [out]	lpBuffer	= o̓obt@
*//***********************************************************************/
void	SHA1Output(LPCSHA1CONTEXT lpContext, u8* lpBuffer)
{
	_SHAEncode32(lpBuffer, lpContext->uState, SHA1_HASHSIZE);
}

/**********************************************************************//**
 *
 * SHA1܂Ƃ߂ČvZ(Init ` Final)
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA1Encode(LPSHA1CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	SHA1InitContext(lpContext);
	SHA1Update(lpContext, lpBuffer, uLength);
	SHA1Final(lpContext);
}

/**********************************************************************//**
 *
 * SHA1܂Ƃ߂ČvZ(Init ` Output)
 *
 ----------------------------------------------------------------------
 * @param [io]	lpDst		= o̓obt@
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA1Encode(u8* lpDst, const u8* lpBuffer, size_t uLength)
{
	SHA1CONTEXT ctx;
	SHA1InitContext(&ctx);
	SHA1Update(&ctx, lpBuffer, uLength);
	SHA1Final(&ctx);
	SHA1Output(&ctx, lpDst);
}

/**********************************************************************//**
 *
 * SHA1o͒l𕶎ɕϊ
 *
 ----------------------------------------------------------------------
 * @param [out]	lpString	= o̓obt@
 * @param [in]	uSize		= o̓obt@TCY
 * @param [in]	lpMD5		= ̓obt@(MD5)
*//***********************************************************************/
LPTSTR	SHA1ToString (LPTSTR lpString, size_t uSize, const u8* lpSHA1)
{
	return SHAToString(lpString, uSize, lpSHA1, SHA1_HASHSIZE);
}
LPSTR	SHA1ToStringA(LPSTR  lpString, size_t uSize, const u8* lpSHA1)
{
	return SHAToStringA(lpString, uSize, lpSHA1, SHA1_HASHSIZE);
}
LPWSTR	SHA1ToStringW(LPWSTR lpString, size_t uSize, const u8* lpSHA1)
{
	return SHAToStringW(lpString, uSize, lpSHA1, SHA1_HASHSIZE);
}

// transform
void _SHA1Transform(u32* lpState, const u8* lpBuffer)
{
#define K1	0x5A827999
#define K2	0x6ED9EBA1
#define K3	0x8F1BBCDC
#define K4	0xCA62C1D6

#define F1(b, c, d)	_SHA_Ch(b, c, d)
#define F2(b, c, d)	((b) ^ (c) ^ (d))
#define F3(b, c, d)	_SHA_Maj(b, c, d)
#define F4(b, c, d)	((b) ^ (c) ^ (d))

#define FF(f, a, b, c, d, e, w, k) 	ROTL(a, 5) + (f(b, c, d)) + e + w + k
#define FF1(a, b, c, d, e, w, k) 	FF(F1, a, b, c, d, e, w, k)
#define FF2(a, b, c, d, e, w, k) 	FF(F2, a, b, c, d, e, w, k)
#define FF3(a, b, c, d, e, w, k) 	FF(F3, a, b, c, d, e, w, k)
#define FF4(a, b, c, d, e, w, k) 	FF(F4, a, b, c, d, e, w, k)

#define ROUND(a, b, c, d, e, t)	\
	(e) = (d); \
	(d) = (c); \
	(c) = ROTL(b, 30); \
	(b) = (a); \
	(a) = t

	int i=0;
	u32 W[80];
	u32 A = *(lpState);
	u32 B = *(lpState+1);
	u32 C = *(lpState+2);
	u32 D = *(lpState+3);
	u32 E = *(lpState+4);
	u32 tmp;

	for( ; i < 16; ++i )
	{
		W[i] = IRIS_ByteArray2DWordBE(lpBuffer+i*4);
	}
	for( ; i < 80; ++i )
	{
		W[i] = ROTL(W[i-3] ^ W[i-8] ^ W[i-14] ^ W[i-16], 1);
	}

	for( i=0; i < 20; ++i )
	{
		tmp = FF1(A, B, C, D, E, W[i], K1);
		ROUND(A, B, C, D, E, tmp);
	}

	for( ; i < 40; ++i )
	{
		tmp = FF2(A, B, C, D, E, W[i], K2);
		ROUND(A, B, C, D, E, tmp);
	}

	for( ; i < 60; ++i )
	{
		tmp = FF3(A, B, C, D, E, W[i], K3);
		ROUND(A, B, C, D, E, tmp);
	}

	for( ; i < 80; ++i )
	{
		tmp = FF4(A, B, C, D, E, W[i], K4);
		ROUND(A, B, C, D, E, tmp);
	}

	*(lpState)   += A;
	*(lpState+1) += B;
	*(lpState+2) += C;
	*(lpState+3) += D;
	*(lpState+4) += E;
}

}	// end of namespace math
}	// end of namespace iris

#if	(defined(_IRIS_SUPPORT_GOOGLETEST) || defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/gt/gt_inchead.h"
#include "../../iris_using.h"

TEST(CMathSHA1Test, Function)
{
	char comm[256] = "";
	u8 out[SHA1_HASHSIZE];
	SHA1CONTEXT ctx;

	// TEST1
	strcpy_s(comm, 256, "abc");
	SHA1Encode(out, comm, strlen(comm));
	SHA1ToStringA(comm, 256, out);
	ASSERT_STREQ( "a9993e364706816aba3e25717850c26c9cd0d89d"
		, comm );

	// TEST2
	strcpy_s(comm, 256, "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq");
	SHA1Encode(out, comm, strlen(comm));
	SHA1ToStringA(comm, 256, out);
	ASSERT_STREQ( "84983e441c3bd26ebaae4aa1f95129e5e54670f1"
		, comm );

	// TEST3
	strcpy_s(comm, 256, "a");
	SHA1InitContext(&ctx);
	for( int i=0; i < 1000000; ++i )
		SHA1Update(&ctx, comm, strlen(comm));
	SHA1Final(&ctx);
	SHA1Output(&ctx, out);
	SHA1ToStringA(comm, 256, out);
	ASSERT_STREQ( "34aa973cd4c4daa4f61eeb2bdbad27316534016f"
		, comm );

	// TEST4
	strcpy_s(comm, 256, "01234567012345670123456701234567""01234567012345670123456701234567");
	SHA1InitContext(&ctx);
	for( int i=0; i < 10; ++i )
		SHA1Update(&ctx, comm, strlen(comm));
	SHA1Final(&ctx);
	SHA1Output(&ctx, out);
	SHA1ToStringA(comm, 256, out);
	ASSERT_STREQ( "dea356a2cddd90c7a7ecedc5ebb563934f460452"
		, comm );
}

#endif	// #if	defined(_IRIS_SUPPORT_GOOGLETEST)

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

#define TEXT_TEST	0
//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_using.h"
#include "../../iris_iostream.h"
#if	!TEXT_TEST
#include "../../fnd/io/FndFile.h"
#endif

//======================================================================
// test
IRIS_UNITTEST(CMathSHA1UnitTest, Func)
{
	char comm[256] = "";
	u8 out[SHA1_HASHSIZE];

#if TEXT_TEST
	std::clog << "SHA1GR[h܂B" << std::endl;
#else
	std::clog << "SHA1GR[h܂Bt@CpX͂ĂB" << std::endl;
#endif
	std::safe_cin >> comm;

#if	TEXT_TEST
	SHA1Encode(out, comm, strlen(comm));
#else
	fnd::CFile file;
	if( !file.OpenA(comm, fnd::FOPEN_READ) ) return;
	IrisU32	size = file.GetSize();
	char* buf = new char [size+1];
	file.Read(buf, size);
	SHA1Encode(out, buf, size);
#endif

	for( int i=0; i < SHA1_HASHSIZE; ++i )
		printf("%.2x", out[i]);
	std::cout << std::endl;
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
