//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTQuaternion.h
 * @brief		NH[^jI֐t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTQuaternion_H_
#define INCG_IRIS_MathTQuaternion_H_

//======================================================================
// include
#include "MathTVector4.h"
#include "../../c++0x/type_traits/cpp0x_value_traits.hpp"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionUnit(IrisTQuaternion<_TN>* pq0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionCopy(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionAdd(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSub(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionMul(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2);
template<typename _TN>_TN					TFpuQuaternionInnerProduct(const IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionConj(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionTransform(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSlerp(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2, _TN t);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSquad(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2
													   , const IrisTQuaternion<_TN>* pq3, const IrisTQuaternion<_TN>* pq4, _TN t);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionNormalize(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionInverse(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionLn(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionExp(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuQuaternionToMatrix(IrisTMtx44<_TN>* pm0, const IrisTQuaternion<_TN>* pq0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromMatrix(IrisTQuaternion<_TN>* pq0, const IrisTMtx44<_TN>* pm0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotZYX(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotXYZ(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotYXZ(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotate(IrisTQuaternion<_TN>* pq0, _TN angle, const IrisTVec4<_TN>* pvAxis);
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionToRotZYX(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0);
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionToRotXYZ(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0);
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionToRotYXZ(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0);

template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionIdentity(IrisTQuaternion<_TN>* pq0);
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionApply(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv1);
template<typename _TN>_TN					TFpuQuaternionDot(const IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1);

//======================================================================
// function
/**
 * @brief	PʃNH[^jI𐶐
 * @param [out]	pq0	= o̓NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionUnit(IrisTQuaternion<_TN>* pq0)
{
	MATH_FPU_NULLASSERT( pq0 );
	pq0->x = pq0->y = pq0->z = cpp0x::zero_traits<_TN>::value;
	pq0->w = cpp0x::one_traits<_TN>::value;
	return pq0;
}

/**
 * @brief	NH[^jIRs[
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= Rs[NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionCopy(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	memcpy(pq0, pq1, sizeof(IrisTQuaternion<_TN>));
	return pq0;
}

/**
 * @brief	NH[^jI̘avZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionAdd(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	pq0->x = pq1->x + pq2->x;
	pq0->y = pq1->y + pq2->y;
	pq0->z = pq1->z + pq2->x;
	pq0->w = pq1->w + pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̍vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 팸NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓xNg
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSub(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	pq0->x = pq1->x - pq2->x;
	pq0->y = pq1->y - pq2->y;
	pq0->z = pq1->z - pq2->x;
	pq0->w = pq1->w - pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̐ςvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 搔NH[^jI
 * @param [in]	pq2	= 搔NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionMul(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	_TN x1 = pq1->x, y1 = pq1->y, z1 = pq1->z, w1 = pq1->w;
	_TN x2 = pq2->x, y2 = pq2->y, z2 = pq2->z, w2 = pq2->w;
	pq0->x = w1 * x2 + w2 * x1 + y1 * z2 - z1 * y2;
	pq0->y = w1 * y2 + w2 * y1 + z1 * x2 - x1 * z2;
	pq0->z = w1 * z2 + w2 * z1 + x1 * y2 - y1 * x2;
	pq0->w = w1 * w2 - x1 * x2 - y1 * y2 - z1 * z2;
	return pq0;
}

/**
 * @brief	NH[^jI̓ςvZ
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return 
*/
template<typename _TN>_TN				TFpuQuaternionInnerProduct(const IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	return ( pq0->x * pq1->x + pq0->y * pq1->y + pq0->z * pq0->z + pq0->w * pq0->w );
}

/**
 * @brief	NH[^jI̋vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionConj(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	pq0->x = -pq1->x;
	pq0->y = -pq1->y;
	pq0->z = -pq1->z;
	pq0->w =  pq1->w;
	return pq0;
}

/**
 * @brief	NH[^jIgăxNg]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pv1	= xNg
 * @return o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*			TFpuQuaternionTransform(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv1 );
	IrisTQuaternion<_TN> q1, q2;
	TFpuQuaternionConj(&q2, pq0);
	TFpuQuaternionMul(&q1, pq0, (const IrisTQuaternion<_TN>*)pv1);
	TFpuQuaternionMul((IrisTQuaternion<_TN>*)pv0, &q1, &q2);
	return pv0;
}

/**
 * @brief	NH[^jI̋ʐ`⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSlerp(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2, _TN t)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	MATH_FPU_NULLASSERT( pq2 );
	static const _TN	ONE	= cpp0x::one_traits<_TN>::value;
	_TN c = pq1->x * pq2->x + pq1->y * pq2->y + pq1->z * pq2->z + pq1->w * pq2->w;
	_TN angle;
	_TN s;
	_TN r1, r2;

	int reverse = 0;
	if( c < 0 ) { c = - c; reverse = 1; }

	angle = acos( c < -ONE ? -ONE : ( c > ONE ? ONE : c ) );
	s = F32_Sin( angle );
	if( s < 0.00005f )
	{
		memcpy(pq0, pq2, sizeof(IrisTQuaternion<_TN>));
		return (pq0);
	}
	r1 =  F32_Sin( ( 1.0f - t ) * angle ) / s;
	r2 =  F32_Sin( t * angle ) / s;
	if( reverse ) r2 = - r2;

	//@@
	//  : ٓ_ߕӂł̐xmۂɂ quat_normalizeg
	pq0->x = pq1->x * r1 + pq2->x * r2;
	pq0->y = pq1->y * r1 + pq2->y * r2;
	pq0->z = pq1->z * r1 + pq2->z * r2;
	pq0->w = pq1->w * r1 + pq2->w * r2;
	return pq0;
}

/**
 * @brief	NH[^jI̋3⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	pq3	= NH[^jI
 * @param [in]	pq4	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionSquad(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1, const IrisTQuaternion<_TN>* pq2
													   , const IrisTQuaternion<_TN>* pq3, const IrisTQuaternion<_TN>* pq4, _TN t)
{
	IrisTQuaternion<_TN> qa, qb;
	TFpuQuaternionSlerp(&qa, pq1, pq2, t);
	TFpuQuaternionSlerp(&qb, pq3, pq4, t);
	TFpuQuaternionSlerp(pq0, &qa, &qb, 2*t * (1.0f-t));
	return pq0;
}

/**
 * @brief	NH[^jI̐K
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionNormalize(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	_TN x = pq1->x, y = pq1->y, z = pq1->z, w = pq1->w;
	_TN q = 1 / F32_Sqrt( x * x + y * y + z * z + w * w );
	pq0->x = x * q;
	pq0->y = y * q;
	pq0->z = z * q;
	pq0->w = w * q;
	return pq0;
}

/**
 * @brief	tNH[^jǏvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionInverse(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	_TN n = pq1->x * pq1->x + pq1->y * pq1->y + pq1->z * pq1->z + pq1->w * pq1->w;
	pq0->x = -pq1->x / n;
	pq0->y = -pq1->y / n;
	pq0->z = -pq1->z / n;
	pq0->w =  pq1->w / n;
	return pq0;
}

/**
 * @brief	NH[^jI̎RΐvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionLn(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	_TN n, theta, m;
	n = F32_Sqrt( pq1->x * pq1->x + pq1->y * pq1->y + pq1->z * pq1->z );
	if(n > cpp0x::zero_traits<_TN>::value)
	{
		theta = F32_Acos( pq1->w / n );
		m     = theta / n;
		pq0->x = pq1->x * m;
		pq0->y = pq1->y * m;
		pq0->z = pq1->z * m;
	}
	else
	{
		pq0->x = pq1->x;
		pq0->y = pq1->y;
		pq0->z = pq1->z;
	}
	pq0->w = cpp0x::zero_traits<_TN>::value;
	return pq0;
}

/**
 * @brief	NH[^jI̎wvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionExp(IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pq1 );
	_TN theta = F32_Sqrt( pq1->x * pq1->x + pq1->y * pq1->y + pq1->z * pq1->z );
	_TN s = F32_Sin(theta) / theta;
	pq0->x = pq1->x * s;
	pq0->y = pq1->y * s;
	pq0->z = pq1->z * s;
	pq0->w = F32_Cos(theta);
	return pq0;
}

/**
 * @brief	NH[^jI}gbNXɕϊ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pq0	= NH[^jI
 * @return o̓}gbNX
*/
template<typename _TN>IrisFMtx44*		TFpuQuaternionToMatrix(IrisFMtx44* pm0, const IrisTQuaternion<_TN>* pq0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pq0 );
	static const _TN ONE	= cpp0x::one_traits<_TN>::value;
	_TN x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;
	pm0->x.x = ONE -	2 * y * y - 2 * z * z;
	pm0->x.y =			2 * x * y + 2 * w * z;
	pm0->x.z =			2 * z * x - 2 * w * y;
	pm0->x.w = ONE;
	pm0->y.x =			2 * x * y - 2 * w * z;
	pm0->y.y = ONE -	2 * x * x - 2 * z * z;
	pm0->y.z =			2 * y * z + 2 * w * x;
	pm0->y.w = cpp0x::zero_traits<_TN>::value;
	pm0->z.x =			2 * z * x + 2 * w * y;
	pm0->z.y =			2 * y * z - 2 * w * x;
	pm0->z.z = ONE -	2 * x * x - 2 * y * y;
	pm0->z.w = cpp0x::zero_traits<_TN>::value;
	pm0->w.x = cpp0x::zero_traits<_TN>::value;
	pm0->w.y = cpp0x::zero_traits<_TN>::value;
	pm0->w.z = cpp0x::zero_traits<_TN>::value;
	pm0->w.w = ONE;
	return pm0;
}

/**
 * @brief	}gbNXNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pm0	= }gbNX
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromMatrix(IrisTQuaternion<_TN>* pq0, const IrisFMtx44* pm0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pm0 );
	_TN x, y, z, w;
	_TN r, t;
	const IrisTVec4<_TN> *pv = pm0->v;

	t = pv[0].x + pv[1].y + pv[2].z + 1.0f;

	if( t > 0.01f )
	{
		w = F32_Sqrt( t ) * 0.5f;
		r = 0.25f / w;
		x = ( pv[1].z - pv[2].y ) * r;
		y = ( pv[2].x - pv[0].z ) * r;
		z = ( pv[0].y - pv[1].x ) * r;
	}
	else if( pv[0].x > pv[1].y )
	{
		if( pv[0].x > pv[2].z )
		{
			x = F32_Sqrt( 1.0f + pv[0].x - pv[1].y - pv[2].z ) * 0.5f;
			r = 0.25f / x;
			y = ( pv[1].x + pv[0].y ) * r;
			z = ( pv[2].x + pv[0].z ) * r;
			w = ( pv[1].z - pv[2].y ) * r;
		}
		else
		{
			z = F32_Sqrt( 1.0f + pv[2].z - pv[0].x - pv[1].y ) * 0.5f;
			r = 0.25f / z;
			x = ( pv[2].x + pv[0].z ) * r;
			y = ( pv[2].y + pv[1].z ) * r;
			w = ( pv[0].y - pv[1].x ) * r;
		}
	}
	else
	{
		if( pv[1].y > pv[2].z )
		{
			y = F32_Sqrt( 1.0f + pv[1].y - pv[2].z - pv[0].x ) * 0.5f;
			r = 0.25f / y;
			x = ( pv[1].x + pv[0].y ) * r;
			z = ( pv[2].y + pv[1].z ) * r;
			w = ( pv[2].x - pv[0].z ) * r;
		}
		else
		{
			z = F32_Sqrt( 1.0f + pv[2].z - pv[0].x - pv[1].y ) * 0.5f;
			r = 0.25f / z;
			x = ( pv[2].x + pv[0].z ) * r;
			y = ( pv[2].y + pv[1].z ) * r;
			w = ( pv[0].y - pv[1].x ) * r;
		}
	}
	pq0->x = x;
	pq0->y = y;
	pq0->z = z;
	pq0->w = w;
	return (pq0);
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotZYX(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	_TN x = pv0->x * 0.5f;
	_TN y = pv0->y * 0.5f;
	_TN z = pv0->z * 0.5f;
	_TN cx = F32_Cos( x );	_TN sx = F32_Sin( x );
	_TN cy = F32_Cos( y );	_TN sy = F32_Sin( y );
	_TN cz = F32_Cos( z );	_TN sz = F32_Sin( z );
	pq0->x = cz * cy * sx - cx * sz * sy;
	pq0->y = cx * cz * sy + cy * sz * sx;
	pq0->z = cx * cy * sz - cz * sy * sx;
	pq0->w = cz * cy * cx + sz * sy * sx;
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotXYZ(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	_TN x = pv0->x * 0.5f;
	_TN y = pv0->y * 0.5f;
	_TN z = pv0->z * 0.5f;
	_TN cx = F32_Cos( x );	_TN sx = F32_Sin( x );
	_TN cy = F32_Cos( y );	_TN sy = F32_Sin( y );
	_TN cz = F32_Cos( z );	_TN sz = F32_Sin( z );
	pq0->x = sx * cy * cz + cx * sy * sz;
	pq0->y = cx * sy * cz - sx * cy * sz;
	pq0->z = cx * cy * sz + sx * sy * cz;
	pq0->w = cx * cy * cz - sx * sy * sz;
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotYXZ(IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pv0 );
	_TN x = pv0->x * 0.5f;
	_TN y = pv0->y * 0.5f;
	_TN z = pv0->z * 0.5f;
	_TN cx = F32_Cos( x );	_TN sx = F32_Sin( x );
	_TN cy = F32_Cos( y );	_TN sy = F32_Sin( y );
	_TN cz = F32_Cos( z );	_TN sz = F32_Sin( z );
	pq0->x = cy * cz * sx + sy * cx * sz;
	pq0->y = sy * cx * cz - cy * sx * sz;
	pq0->z = cy * cx * sz - sy * cz * sx;
	pq0->w = cy * cx * cz + sy * sx * sz;
	return pq0;
}

/**
 * @brief	]xNgƉ]ʂANH[^jI𐶐
 * @param [out]	pq0		= o̓NH[^jI
 * @param [in]	angle	= ]
 * @param [in]	pvAxis	= ]xNg
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionFromRotate(IrisTQuaternion<_TN>* pq0, _TN angle, const IrisTVec4<_TN>* pvAxis)
{
	MATH_FPU_NULLASSERT( pq0 );
	MATH_FPU_NULLASSERT( pvAxis );
	FpuVec4NormalizeXYZ((IrisTVec4<_TN>*)pq0, pvAxis);

	angle = angle * 0.5f;
	FpuVec4Scale((IrisTVec4<_TN>*)pq0, (IrisTVec4<_TN>*)pq0, F32_Sin(angle));
	pq0->w = F32_Cos( angle );
	return pq0;
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*			TFpuQuaternionToRotZYX(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	_TN cx, sx, sy, cz, sz;
	_TN rx, ry, rz;
	_TN x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = -( 2.0f * z * x - 2.0f * w * y );
	if( sy <= 0.99995f && sy >= -0.99995f )
	{
		sx =		2.0f * y * z + 2.0f * w * x;
		cx = 1.0f - 2.0f * x * x - 2.0f * y * y;
		sz =		2.0f * x * y + 2.0f * w * z;
		cz = 1.0f - 2.0f * y * y - 2.0f * z * z;
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
	}
	else
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sx =		-(2.0f * y * z - 2.0f * w * x);
		cx = 1.0f	- 2.0f * x * x - 2.0f * z * z;
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = cpp0x::zero_traits<_TN>::value;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵ZY]߂
		sz =		-(2.0f * x * y - 2.0f * w * z);
		cz = 1.0f	- 2.0f * x * x - 2.0f * z * z;
		rx = 0.0f;
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = cpp0x::zero_traits<_TN>::value;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*			TFpuQuaternionToRotXYZ(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	_TN cx, sx, sy, cz, sz;
	_TN rx, ry, rz;
	_TN x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = 2.0f * z * x + 2.0f * w * y;
	if( sy <= 0.99995f && sy >= -0.99995f )
	{
		sx =		-(2.0f * y * z - 2.0f * w * x);
		cx = 1.0f	- 2.0f * x * x - 2.0f * y * y;
		sz =		-(2.0f * x * y - 2.0f * w * z);
		cz = 1.0f	- 2.0f * y * y - 2.0f * z * z;
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
	}
	else
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵XY]߂
		sx =		2.0f * y * z + 2.0f * w * x;
		cx = 1.0f - 2.0f * x * x - 2.0f * z * z;
		rx = F32_Atan2( sx, cx );
		ry = F32_Asin( sy );
		rz = cpp0x::zero_traits<_TN>::value;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵YZ]߂
		sz =		2.0f * x * y + 2.0f * w * z;
		cz = 1.0f - 2.0f * x * x - 2.0f * z * z;
		rx = cpp0x::zero_traits<_TN>::value;
		ry = F32_Asin( sy );
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = cpp0x::zero_traits<_TN>::value;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*			TFpuQuaternionToRotYXZ(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pq0 );
	_TN sx, sy, cy, sz, cz;
	_TN rx, ry, rz;
	_TN x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sx = -(2.0f * y * z - 2.0f * w * x);
	if( sx <= 0.99995f && sx >= -0.99995f )
	{
		sy =		2.0f * z * x + 2.0f * w * y;
		cy = 1.0f - 2.0f * x * x - 2.0f * y * y;
		sz =		2.0f * x * y + 2.0f * w * z;
		cz = 1.0f - 2.0f * x * x - 2.0f * z * z;
		rx = F32_Asin( sx );
		ry = F32_Atan2( sy, cy );
		rz = F32_Atan2( sz, cz );
	}
	else
	{
#if 1
		//J |cosX|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sy =		-(2.0f * z * x - 2.0f * w * y);
		cy = 1.0f	- 2.0f * y * y - 2.0f * z * z;
		rx = F32_Asin( sx );
		ry = F32_Atan2( sy, cy );
		rz = cpp0x::zero_traits<_TN>::value;
#else
		//J |cosX|0ɋ߂ƂAY]0Ɖ肵XZ]߂
		sz =		-(2.0f * x * y - 2.0f * w * z);
		cz = 1.0f	- 2.0f * y * y - 2.0f * z * z;
		rx = F32_Asin( sx );
		ry = cpp0x::zero_traits<_TN>::value;
		rz = F32_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = cpp0x::zero_traits<_TN>::value;
	return (pv0);
}

/**
 * @brief	PʃNH[^jI𐶐
 * @param [out]	pq0	= o̓NH[^jI
 * @return o̓NH[^jI
*/
template<typename _TN>IrisTQuaternion<_TN>*	TFpuQuaternionIdentity(IrisTQuaternion<_TN>* pq0)
{
	return TFpuQuaternionUnit(pq0);
}

/**
 * @brief	NH[^jIgăxNg]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pv1	= xNg
 * @return o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*		TFpuQuaternionApply(IrisTVec4<_TN>* pv0, const IrisTQuaternion<_TN>* pq0, const IrisTVec4<_TN>* pv1)
{
	return TFpuQuaternionTransform(pv0, pq0, pv1);
}

/**
 * @brief	NH[^jI̓ςvZ
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return 
*/
template<typename _TN>_TN					TFpuQuaternionDot(const IrisTQuaternion<_TN>* pq0, const IrisTQuaternion<_TN>* pq1)
{
	return TFpuQuaternionInnerProduct(pq0, pq1);
}

}	// end of namespace math
}	// end of namespace iris

#endif
