//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTVector4.h
 * @brief		4DxNgt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTVector4_H_
#define INCG_IRIS_MathTVector4_H_

//======================================================================
// include
#include "../fpu/MathFpuDef.h"
#include "../../c++0x/type_traits/cpp0x_value_traits.hpp"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Set(IrisTVec4<_TN>* pv0, _TN x, _TN y, _TN z, _TN w);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SetXYZ(IrisTVec4<_TN>* pv0, _TN x, _TN y, _TN z);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Copy(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4PositiveZero(IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NegativeZero(IrisTVec4<_TN>* pv0);
template<typename _TN>IrisIVec4*		TFpuVec4Ceil(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisIVec4*		TFpuVec4Trunc(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisIVec4*		TFpuVec4Round(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisIVec4*		TFpuVec4Floor(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4FromIVec4(IrisTVec4<_TN>* pv0, const IrisIVec4* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Add(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4AddXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Sub(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SubXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Mul(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4MulXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Div(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4DivXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Neg(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Abs(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Lerp(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4LerpXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Scale(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN s);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN s);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleAdd(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN s);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleAddXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN s);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Hermite(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pt1
									  , const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pt2, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4HermiteXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pt1
									  , const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pt2, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Bezier(IrisTVec4<_TN>* pv0, s32 n, const IrisTVec4<_TN>* pva, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4BezierXYZ(IrisTVec4<_TN>* pv0, s32 n, const IrisTVec4<_TN>* pva, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Clamp(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN min, _TN max);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ClampXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN min, _TN max);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Max(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Min(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>_TN				TFpuVec4InnerProduct(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>_TN				TFpuVec4InnerProductXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4OuterProductXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>_TN				TFpuVec4Funnel(const IrisTVec4<_TN>* pv0);
template<typename _TN>_TN				TFpuVec4Average(const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisBool			TFpuVec4IsEqual(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisBool			TFpuVec4IsZero(const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisBool			TFpuVec4IsZeroXYZ(const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SignFloat(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisIVec4*		TFpuVec4SignInt(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Normalize(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NormalizeXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NormalizePhase(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>_TN				TFpuVec4LengthXYZ(const IrisTVec4<_TN>* pv0);
template<typename _TN>_TN				TFpuVec4DistanceXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4FaceForwardXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pv3);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ReflectXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4RefractXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN eta);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4TruncatePrecision24(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);

template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Null(IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Zero(IrisTVec4<_TN>* pv0);
template<typename _TN>_TN				TFpuVec4Dot(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>_TN				TFpuVec4DotXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Cross(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Subtract(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Multiply(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Divide(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Inter(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SubtractXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4MultiplyXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4DivideXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2);
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4InterXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t);

//======================================================================
// inline
/**
 * @brief	vfݒ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @param [in]	z	= zl
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Set(IrisTVec4<_TN>* pv0, _TN x, _TN y, _TN z, _TN w)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y; pv0->z = z; pv0->w = w;
	return pv0;
}

/**
 * @brief	vfݒ(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @param [in]	z	= zl
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SetXYZ(IrisTVec4<_TN>* pv0, _TN x, _TN y, _TN z)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y; pv0->z = z;
	return pv0;
}

/**
 * @brief	vfRs[
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Copy(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x; pv0->y = pv1->y; pv0->z = pv1->z; pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	+0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4PositiveZero(IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = pv0->z = pv0->w = 0;
	return pv0;
}

/**
 * @brief	-0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NegativeZero(IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = pv0->z = pv0->w = -0;
	return pv0;
}

/**
 * @brief	؂グ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec4*	TFpuVec4Ceil(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Ceil(pv1->x) );
	pv0->y = (s32)( F64_Ceil(pv1->y) );
	pv0->z = (s32)( F64_Ceil(pv1->z) );
	pv0->w = (s32)( F64_Ceil(pv1->w) );
	return pv0;
}

/**
 * @brief	0ۂ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec4*	TFpuVec4Trunc(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Trunc(pv1->x) );
	pv0->y = (s32)( F64_Trunc(pv1->y) );
	pv0->z = (s32)( F64_Trunc(pv1->z) );
	pv0->w = (s32)( F64_Trunc(pv1->w) );
	return pv0;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec4*	TFpuVec4Round(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Round(pv1->x) );
	pv0->y = (s32)( F64_Round(pv1->y) );
	pv0->z = (s32)( F64_Round(pv1->z) );
	pv0->w = (s32)( F64_Round(pv1->w) );
	return pv0;
}

/**
 * @brief	؂̂
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec4*	TFpuVec4Floor(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Floor(pv1->x) );
	pv0->y = (s32)( F64_Floor(pv1->y) );
	pv0->z = (s32)( F64_Floor(pv1->z) );
	pv0->w = (s32)( F64_Floor(pv1->w) );
	return pv0;
}

/**
 * @brief	xNgϊ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4FromIVec4(IrisTVec4<_TN>* pv0, const IrisIVec4* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (_TN)( pv1->x );
	pv0->y = (_TN)( pv1->y );
	pv0->z = (_TN)( pv1->z );
	pv0->w = (_TN)( pv1->w );
	return pv0;
}

/**
 * @brief	xNg̉Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Add(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	pv0->z = pv1->z + pv2->z;
	pv0->w = pv1->w + pv2->w;
	return pv0;
}

/**
 * @brief	xNg̉Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4AddXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	pv0->z = pv1->z + pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Sub(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	pv0->z = pv1->z - pv2->z;
	pv0->w = pv1->w - pv2->w;
	return pv0;
}

/**
 * @brief	xNǧZ(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SubXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	pv0->z = pv1->z - pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Mul(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * pv2->x;
	pv0->y = pv1->y * pv2->y;
	pv0->z = pv1->z * pv2->z;
	pv0->w = pv1->w * pv2->w;
	return pv0;
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4MulXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * pv2->x;
	pv0->y = pv1->y * pv2->y;
	pv0->z = pv1->z * pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Div(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x / pv2->x;
	pv0->y = pv1->y / pv2->y;
	pv0->z = pv1->z / pv2->z;
	pv0->w = pv1->w / pv2->w;
	return pv0;
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4DivXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x / pv2->x;
	pv0->y = pv1->y / pv2->y;
	pv0->z = pv1->z / pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̕]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Neg(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = -pv1->x;
	pv0->y = -pv1->y;
	pv0->z = -pv1->z;
	pv0->w = -pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̐Βl擾
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Abs(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = F32_ABS(pv1->x);
	pv0->y = F32_ABS(pv1->y);
	pv0->z = F32_ABS(pv1->z);
	pv0->w = F32_ABS(pv1->w);
	return pv0;
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Lerp(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + t * (pv2->x - pv1->x);
	pv0->y = pv1->y + t * (pv2->y - pv1->y);
	pv0->z = pv1->z + t * (pv2->z - pv1->z);
	pv0->w = pv1->w + t * (pv2->w - pv1->w);
	return pv0;
}

/**
* @brief	xNg̊Ԃ̓}(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4LerpXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + t * (pv2->x - pv1->x);
	pv0->y = pv1->y + t * (pv2->y - pv1->y);
	pv0->z = pv1->z + t * (pv2->z - pv1->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Scale(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x * s;
	pv0->y = pv1->y * s;
	pv0->z = pv1->z * s;
	pv0->w = pv1->w * s;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = pv1->x * s;
	pv0->y = pv1->y * s;
	pv0->z = pv1->z * s;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleAdd(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * s + pv2->x;
	pv0->y = pv1->y * s + pv2->y;
	pv0->z = pv1->z * s + pv2->z;
	pv0->w = pv1->w * s + pv2->w;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ScaleAddXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN s)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->x * s + pv2->x;
	pv0->y = pv1->y * s + pv2->y;
	pv0->z = pv1->z * s + pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Hermite(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pt1
									  , const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pt2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pt1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pt2 );
	_TN t2 = t * t;
	_TN t3 = t * t2;
	IrisTVec4<_TN> v[4];
	TFpuVec4Scale(&v[0], pv1, t);
	TFpuVec4Scale(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1));
	TFpuVec4Scale(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	TFpuVec4Scale(&v[2], pt2, ((     t3) - (    t2)           ));
	TFpuVec4Scale(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	TFpuVec4Add(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	TFpuVec4Add(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	TFpuVec4Add(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4HermiteXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pt1
									  , const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pt2, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pt1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pt2 );
	_TN t2 = t * t;
	_TN t3 = t * t2;
	IrisTVec4<_TN> v[4];
	TFpuVec4ScaleXYZ(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1));
	TFpuVec4ScaleXYZ(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	TFpuVec4ScaleXYZ(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	TFpuVec4ScaleXYZ(&v[2], pt2, ((     t3) - (    t2)           ));
	TFpuVec4AddXYZ(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	TFpuVec4AddXYZ(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	TFpuVec4AddXYZ(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Bezier(IrisTVec4<_TN>* pv0, s32 n, const IrisTVec4<_TN>* pva, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0 ) { TFpuVec4Copy(pv0, &pva[0]); return pv0; }
	if( t >= 1 ) { TFpuVec4Copy(pv0, &pva[n-1]); return pv0; }
	TFpuVec4PositiveZero(pv0);
	_TN tt = 1;			// t^ip
	_TN it = 1 - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisTVec4<_TN>* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		_TN J = Combination(nn, i) * tt * F64_Pow(it, (_TN)(nn-i));
		pv0->x += pv->x * J;
		pv0->y += pv->y * J;
		pv0->z += pv->z * J;
		pv0->w += pv->w * J;
		tt = tt * t;
	}
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4BezierXYZ(IrisTVec4<_TN>* pv0, s32 n, const IrisTVec4<_TN>* pva, _TN t)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0 ) { TFpuVec4Copy(pv0, &pva[0]); return pv0; }
	if( t >= 1 ) { TFpuVec4Copy(pv0, &pva[n-1]); return pv0; }
	TFpuVec4PositiveZero(pv0);
	pv0->w = pva[0].w;
	_TN tt = 1;			// t^ip
	_TN it = 1 - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisTVec4<_TN>* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		_TN J = Combination(nn, i) * F32_Mul(tt, F32_Pow(it, (_TN)(nn-i)));
		pv0->x += pv->x * J;
		pv0->y += pv->y * J;
		pv0->z += pv->z * J;
		tt = tt * t;
	}
	return pv0;
}

/**
 * @brief	xNg̃Nv
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Clamp(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN min, _TN max)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	pv0->z = (pv1->z < min) ? min : ((pv1->z > max) ? max : pv1->z);
	pv0->w = (pv1->w < min) ? min : ((pv1->w > max) ? max : pv1->w);
	return pv0;
}

/**
 * @brief	xNg̃Nv(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ClampXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, _TN min, _TN max)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	pv0->z = (pv1->z < min) ? min : ((pv1->z > max) ? max : pv1->z);
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̑傫Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Max(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x > pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y > pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z > pv2->z) ? pv1->z : pv2->z;
	pv0->w = (pv1->w > pv2->w) ? pv1->w : pv2->w;
	return pv0;
}

/**
 * @brief	xNg̊evf̏Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Min(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y < pv2->y) ? pv1->y : pv2->y;
	pv0->z = (pv1->z < pv2->z) ? pv1->z : pv2->z;
	pv0->w = (pv1->w < pv2->w) ? pv1->w : pv2->w;
	return pv0;
}

/**
 * @brief	
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
template<typename _TN>_TN		TFpuVec4InnerProduct(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	return pv0->x * pv1->x + pv0->y * pv1->y + pv0->z * pv1->z + pv0->w * pv1->w;
}

/**
 * @brief	(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
template<typename _TN>_TN		TFpuVec4InnerProductXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	return pv0->x * pv1->x + pv0->y * pv1->y + pv0->z * pv1->z;
}

/**
 * @brief	O(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4OuterProductXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	pv0->x = pv1->y * pv2->z - pv2->y * pv1->z;
	pv0->y = pv1->z * pv2->x - pv2->z * pv1->x;
	pv0->z = pv1->x * pv2->y - pv2->x * pv1->y;
	pv0->w = 0;
	return pv0;
}

/**
 * @brief	xNg̊evf̑a擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̑a
*/
template<typename _TN>_TN		TFpuVec4Funnel(const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z + pv0->w);
}

/**
 * @brief	xNg̊evf̕ς擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̕
*/
template<typename _TN>_TN		TFpuVec4Average(const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y + pv0->z + pv0->w) / 4;
}

/**
 * @brief	xNgǂԂ
 * @param [in]	pv0	= rxNg
 * @param [in]	pv1	= rxNg
 * @return	^Ul
*/
template<typename _TN>IrisBool	TFpuVec4IsEqual(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	if(    (pv0->x != pv1->x)
		|| (pv0->y != pv1->y)
		|| (pv0->z != pv1->z)
		|| (pv0->w != pv1->w) ) return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	[xNgǂԂ
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
template<typename _TN>IrisBool	TFpuVec4IsZero(const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	if( (pv0->x == 0) && (pv0->y == 0) && (pv0->z == 0) && (pv0->w == 0) )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	[xNgǂԂ(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
template<typename _TN>IrisBool	TFpuVec4IsZeroXYZ(const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	if( (pv0->x == 0) && (pv0->y == 0) && (pv0->z == 0) )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	,̕IrisTVec4<_TN>ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SignFloat(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	pv0->z = (pv1->z < 0) ? -1 : (pv1->z > 0) ? 1 : 0;
	pv0->w = (pv1->w < 0) ? -1 : (pv1->w > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	,̕IrisIVec4ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisIVec4*	TFpuVec4SignInt(IrisIVec4* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	pv0->z = (pv1->z < 0) ? -1 : (pv1->z > 0) ? 1 : 0;
	pv0->w = (pv1->w < 0) ? -1 : (pv1->w > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	xNg̐K
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Normalize(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x, y = pv1->y, z = pv1->z, w = pv1->w;
	_TN q = F64_Sqrt( x * x + y * y + z * z + w * w );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0 );
	q = 1 / q;
#else
	if( q != 0 ) q = 1 / q;
#endif
	pv0->x = x * q;
	pv0->y = y * q;
	pv0->z = z * q;
	pv0->w = w * q;
	return pv0;
}

/**
 * @brief	xNg̐K(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NormalizeXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x, y = pv1->y, z = pv1->z;
	_TN q = F64_Sqrt( x * x + y * y + z * z );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0 );
	q = 1 / q;
#else
	if( q != 0 ) q = 1 / q;
#endif
	pv0->x = x * q;
	pv0->y = y * q;
	pv0->z = z * q;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̊el-΁`+΂Ɏ߂
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4NormalizePhase(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x, y = pv1->y, z = pv1->z;
	_TN q = F64_Sqrt( x * x + y * y + z * z );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0 );
	q = 1 / q;
#else
	if( q != 0 ) q = 1 / q;
#endif
	pv0->x = x * q;
	pv0->y = y * q;
	pv0->z = z * q;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	xNg̒Ԃ(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>_TN	TFpuVec4LengthXYZ(const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	return (F64_Sqrt( pv0->x * pv0->x + pv0->y * pv0->y + pv0->z * pv0->z ));
}

/**
 * @brief	xNg̊Ԃ̋(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>_TN	TFpuVec4DistanceXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pv0 );
	_TN x = pv0->x - pv1->x;
	_TN y = pv0->y - pv1->y;
	_TN z = pv0->z - pv1->z;
	return (F64_Sqrt( x * x + y * y + z * z ));
}

/**
 * @brief	IuWFNg\ʂJɌ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pv3	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4FaceForwardXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, const IrisTVec4<_TN>* pv3)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	MATH_FPU_NULLASSERT( pv3 );
	_TN d = pv2->x * pv3->x + pv2->y * pv3->y + pv2->z * pv3->z;
	if(d < 0)
	{
		pv0->x = pv1->x;
		pv0->y = pv1->y;
		pv0->z = pv1->z;
		pv0->w = pv1->w;
	}
	else
	{
		pv0->x = -pv1->x;
		pv0->y = -pv1->y;
		pv0->z = -pv1->z;
		pv0->w = pv1->w;
	}
	return pv0;
}

/**
 * @brief	˃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4ReflectXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	_TN d = pv1->x * pv2->x + pv1->y * pv2->y + pv1->z * pv2->z;
	pv0->x = pv1->x - 2 * d * pv2->x;
	pv0->y = pv1->y - 2 * d * pv2->y;
	pv0->z = pv1->z - 2 * d * pv2->z;
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	܃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	eta	= ܗ
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4RefractXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN eta)
{
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	MATH_FPU_NULLASSERT( pv2 );
	_TN d, e, f;
	d = pv1->x * pv2->x + pv1->y * pv2->y + pv1->z * pv2->z;
	e = 1 - eta * eta * (1 - d * d);
	if(e < 0)
	{
		pv0->x = 0;
		pv0->y = 0;
		pv0->z = 0;
	}
	else
	{
		f = eta * d - F64_Sqrt(e);
		pv0->x = eta * pv1->x - f * pv2->x;
		pv0->y = eta * pv1->y - f * pv2->y;
		pv0->z = eta * pv1->z - f * pv2->z;
	}
	pv0->w = pv1->w;
	return pv0;
}

/**
 * @brief	+0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Null(IrisTVec4<_TN>* pv0)
{
	return TFpuVec4PositiveZero(pv0);
}

/**
 * @brief	+0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Zero(IrisTVec4<_TN>* pv0)
{
	return TFpuVec4PositiveZero(pv0);
}

/**
 * @brief	
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
template<typename _TN>_TN				TFpuVec4Dot(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	return TFpuVec4InnerProduct(pv0, pv1);
}

/**
 * @brief	(XYZ)
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	ϒl
*/
template<typename _TN>_TN				TFpuVec4DotXYZ(const IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1)
{
	return TFpuVec4InnerProductXYZ(pv0, pv1);
}

/**
 * @brief	O(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Cross(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4OuterProductXYZ(pv0, pv1, pv2);
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Subtract(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4Sub(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Multiply(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4Mul(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Divide(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4Div(pv0, pv1, pv2);
}

/**
* @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4Inter(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t)
{
	return TFpuVec4Lerp(pv0, pv1, pv2, t);
}

/**
 * @brief	xNǧZ(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4SubtractXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4SubXYZ(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4MultiplyXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4MulXYZ(pv0, pv1, pv2);
}

/**
 * @brief	xNg̏Z(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4DivideXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2)
{
	return TFpuVec4DivXYZ(pv0, pv1, pv2);
}

/**
* @brief	xNg̊Ԃ̓}(XYZ)
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*	TFpuVec4InterXYZ(IrisTVec4<_TN>* pv0, const IrisTVec4<_TN>* pv1, const IrisTVec4<_TN>* pv2, _TN t)
{
	return TFpuVec4LerpXYZ(pv0, pv1, pv2, t);
}

}	// end of namespace math
}	// end of namespace iris

#endif
