//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		WXObjLeakCheck.cpp
 * @brief		IuWFNgnh[N`FbN t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_WXObjLeakCheck_CPP_

//======================================================================
// include
#include "WXObjLeakCheck.h"
#include "WXProcHook.h"

#ifdef _IRIS_DEBUG
#include "WXDebugHelp.h"
#include "../../../../../fnd/container/FndScopedLock.h"
#include "WXStackTrace.h"
#include "WXSymbol.h"
#include "../../rtl/WXRtlBackTrace.h"
#include "../WXDebugLeakCheckMacro.h"
#include "../../os/WXCriticalSection.h"
#include "../../base/WXError.h"
#include "../../../../../iris_xchar.hpp"
#include "../../../../../iris_debug.h"
#include <tchar.h>

namespace iris {
namespace wx {
namespace dbg
{

namespace
{

//======================================================================
// struct
typedef struct tagObjectList
{
	void*	object;
	PROC	close;
	TCHAR	file[MAX_PATH];
	TCHAR	symbol[MAX_PATH];
	DWORD	line;
	int		count;
	tagObjectList*	next;
} ObjectList;

//======================================================================
// define
#define APIHOOK_BEGIN()							\
	HMODULE hModule = GetModuleHandle(nullptr);	\
	if( hModule == nullptr ) return false

#define APIHOOK(dll, OriginName)							\
	if( m_##OriginName.Open(dll, #OriginName, hModule) )	\
		m_##OriginName.Replace((PROC)Hook_##OriginName)

#define APIHOOK_END()

#define HOOK_FUNC(ret, OriginName, args)			\
	typedef ret (WINAPI *PFN_##OriginName)##args;	\
	CProcHook	m_##OriginName;						\
	static ret	WINAPI Hook_##OriginName##args

#define HOOK_VARIABLE_NAME(OriginName)		m_##OriginName
#define HOOK_FUNC_NAME(OriginName)			Hook_##OriginName

#define GET_ORIGIN_FUNC(OriginName)	\
	CObjLeakCheckImpl::GetInstance().m_##OriginName.GetOriginProc()

IRIS_MSC_PRAGMA_WARNING_BEGIN()
IRIS_MSC_PRAGMA_WARNING_DISABLE(4191)	// 'hoge'   'PROC' ւ̕ϊ͕ۏ؂܂B

// ɃX^bNǂꍇ 1
#define CHECK_STRICT	0
// RunTime Library gꍇ 1
#define USE_RTL			1

//======================================================================
// class
// NX
class CObjLeakCheckImpl : public IIrisObject
{
	typedef fnd::CSingleLock<CCriticalSection> CCSScopedLock;

	static const int LIST_CAPACITY	= 256;

#if !USE_RTL
	// CObjLeakCheck p StackTrace
	class CObjLeakCheckStackTrace : public CStackTrace
	{
#if CHECK_STRICT
		bool	m_find_this;
	public:
		CObjLeakCheckStackTrace(void)
			: m_find_this(false)
		{
		}
	protected:
		virtual bool	IsSkip(const TRACEINFO& ti)
		{
			if( _tcsstr(ti.filename, IRIS_TEXT("wxobjleakcheck.cpp")) != nullptr )
			{
				m_find_this = true;
				return true;
			}
			return !m_find_this;
		}
#else
	protected:
		virtual bool	IsSkip(const TRACEINFO& ti)
		{
			if( _tcsstr(ti.filename, IRIS_TEXT("wxobjleakcheck.cpp")) != nullptr )
			{
				return true;
			}
			return false;
		}
#endif
	};
#endif

protected:
	HANDLE		m_Heap;
	ObjectList*	m_pTop;
	ObjectList*	m_pLast;
	int			m_AllocCount;
	CCriticalSection	m_CS;
	CSymbol		m_Symbol;

public:
	static CObjLeakCheckImpl&	GetInstance(void)	{ static CObjLeakCheckImpl impl; return impl; }
private:
	// RXgN^
	CObjLeakCheckImpl(void)
		: m_pTop(nullptr)
		, m_Heap(nullptr)
		, m_pLast(nullptr)
		, m_AllocCount(0)
	{
	}

	// fXgN^
	~CObjLeakCheckImpl(void)
	{
		Terminate();
	}

public:
	// 
	bool	Initialize(int uFlags)
	{
		m_AllocCount = 0;

		// Xgpq[v쐬
		m_Heap = HeapCreate(HEAP_GENERATE_EXCEPTIONS, sizeof(ObjectList)*LIST_CAPACITY, 0);
		if( m_Heap == INVALID_HANDLE_VALUE )
		{
			m_Heap = nullptr;
			return false;
		}

		DWORD option = CSymbol::GetOptions();
		option |= SYMOPT_LOAD_LINES;
		option &= ~SYMOPT_UNDNAME;
		option = CSymbol::SetOptions(option);

#if 0
		LPSTR p = strrchr(modulepath, '\\');
		if( p != nullptr ) *p = '\0';
		// V{GW̏
		if( !m_Symbol.Initialize(::GetCurrentProcess(), modulepath, TRUE) ) return false;
#else
		if( !m_Symbol.Initialize(::GetCurrentProcess(), nullptr, TRUE) ) return false;
#endif

		if( !BeginHook(uFlags) )
		{
			return false;
		}
		return true;
	}

	// I
	void	Terminate(void)
	{
		ShowLeakList();

		// V{GW̒~
		m_Symbol.Cleanup();

		Clear();
		if( m_Heap != nullptr )
		{
			HeapDestroy(m_Heap);
			m_Heap = nullptr;
		}
	}

public:
	// Xgɒǉ
	void	Add(void* lpObject, PROC pfnClose)
	{
#if !USE_RTL
		CObjLeakCheckStackTrace st;
#endif

		CScopedLastError e;
		IRIS_ASSERT(m_Heap != nullptr);
		ObjectList* p = (ObjectList*)(HeapAlloc(m_Heap, 0, sizeof(ObjectList)));
		if( p == nullptr )
		{
			IRIS_WARNING("memory alloc failed.");
			return;
		}

		CCSScopedLock lock(m_CS);

#if USE_RTL	// use RunTime Library
		{
			const int kBackTraceCount = 10;
			CRtlBackTrace<kBackTraceCount> backtrace;
			bool find = false;
			for( WORD i=0; i < backtrace.GetCount(); ++i )
			{
				DWORD dwAddr = (DWORD)backtrace[i];
				IMAGEHLP_LINE line = { sizeof(IMAGEHLP_LINE) };
				XDWORD disp=0;
				if( m_Symbol.GetLineFromAddr(dwAddr, &disp, &line) )
				{
					if( strstr(line.FileName, "wxobjleakcheck.cpp") != nullptr )
					{
						find = true;
					}
					else
					{
						if( find )
						{
							// t@CŔ
							DWORD dwAttr = ::GetFileAttributesA(line.FileName);
							if( dwAttr != (DWORD)-1 )
							{
								xcstoxcs_s(p->file, MAX_PATH, line.FileName);
								p->line = line.LineNumber;
								goto find;
							}
						}
					}
				}
			}
			p->file[0] = '\0';
			p->line = 0;
		}
find:
#else
		{
			bool origin = HOOK_VARIABLE_NAME(CreateThread).IsOrigin();
			if( !origin )
			{
				HOOK_VARIABLE_NAME(CreateThread).Reset();	// ޔ
			}
#if CHECK_STRICT
			if( st.Trace(1, CStackTrace::SKIP_UNKOWN_SOURCE|CStackTrace::EXAMINE_EACH) )
#else
			if( st.Trace(7, CStackTrace::SKIP_UNKOWN_SOURCE) )
#endif
			{
				_tcscpy_s(p->file, MAX_PATH, st.GetFileName());
				p->line = st.GetLine();
			}
			else
			{
				p->file[0] = '\0';
				p->line = 0;
			}
			if( !origin )
			{
				HOOK_VARIABLE_NAME(CreateThread).Revert();	// ߂
			}
		}
#endif
		p->close = pfnClose;
		p->object = lpObject;
		p->count = m_AllocCount++;
		p->next = nullptr;

		if( m_pTop != nullptr )
		{
			IRIS_ASSERT( m_pLast != nullptr );
			m_pLast->next = p;
		}
		else
		{
			IRIS_ASSERT( m_pLast == nullptr );
			m_pTop = p;
		}
		m_pLast = p;
	}
	// Xg폜
	void	Remove(void* lpObject, PROC pfnClose)
	{
		CScopedLastError e;
		CCSScopedLock lock(m_CS);

		ObjectList* pre = nullptr;
		ObjectList* p = m_pTop;
		while(p != nullptr)
		{
			if( p->object == lpObject && p->close == pfnClose )
			{
				break;
			}
			pre = p;
			p = p->next;
		}
		if( p == nullptr )
		{
			return;
		}
		if( pre == nullptr )
		{
			IRIS_ASSERT( m_pTop == p );
			m_pTop = p->next;
			if( m_pLast == p )
			{
				m_pLast = p->next;
				IRIS_ASSERT( m_pLast == nullptr );
			}
		}
		else
		{
			pre->next = p->next;
			if( m_pLast == p )
			{
				m_pLast = pre;
				IRIS_ASSERT( pre->next == nullptr );
			}
		}
		HeapFree(m_Heap, 0, p);
	}


private:
	// XgNA
	void	Clear(void)
	{
		CCSScopedLock lock(m_CS);
		ObjectList* pre = nullptr;
		ObjectList* p = m_pTop;

		while(p != nullptr)
		{
			pre = p;
			p = p->next;
			HeapFree(m_Heap, 0, pre);
		}
		m_pTop = nullptr;
	}

	// [NXg̏o
	void	ShowLeakList(void)
	{
		CCSScopedLock lock(m_CS);
		ObjectList* p = m_pTop;
		if( m_pTop != nullptr )
		{
			dbgoutputA("---------- OBJECT Leak!! ----------\r\n");
			while( p != nullptr )
			{
				TCHAR buffer[1024];
				wsprintf(buffer, IRIS_TEXT("%s(%d) : [%d] Object=0x%p\r\n"), p->file, p->line, p->count, p->object);
				dbgoutput(buffer);
				p = p->next;
			}
			dbgoutputA("-----------------------------------\r\n");
		}
	}


private:
	// tbNJn
	bool	BeginHook(int uFlags)
	{
		APIHOOK_BEGIN();

		if( uFlags & CObjLeakCheck::HOOKTARGET_HANDLE )
		{
			APIHOOK("kernel32.dll", CloseHandle);

			APIHOOK("kernel32.dll", CreateThread);
			APIHOOK("kernel32.dll", OpenEventA);
			APIHOOK("kernel32.dll", CreateEventA);
			APIHOOK("kernel32.dll", CreateProcessA);
			APIHOOK("kernel32.dll", CreateFileA);
			APIHOOK("kernel32.dll", CreateFileMappingA);
			APIHOOK("kernel32.dll", OpenFileMappingA);
			APIHOOK("kernel32.dll", CreateMutexA);
			APIHOOK("kernel32.dll", OpenMutexA);
			APIHOOK("kernel32.dll", CreateSemaphoreA);
			APIHOOK("kernel32.dll", OpenSemaphoreA);
			APIHOOK("kernel32.dll", CreateNamedPipeA);
			APIHOOK("kernel32.dll", CreatePipe);

			APIHOOK("kernel32.dll", OpenEventW);
			APIHOOK("kernel32.dll", CreateEventW);
			APIHOOK("kernel32.dll", CreateProcessW);
			APIHOOK("kernel32.dll", CreateFileW);
			APIHOOK("kernel32.dll", CreateFileMappingW);
			APIHOOK("kernel32.dll", OpenFileMappingW);
			APIHOOK("kernel32.dll", CreateMutexW);
			APIHOOK("kernel32.dll", OpenMutexW);
			APIHOOK("kernel32.dll", CreateSemaphoreW);
			APIHOOK("kernel32.dll", OpenSemaphoreW);
			APIHOOK("kernel32.dll", CreateNamedPipeW);
		}

#if 0
		// [N`FbNIuWFNgɉꂽ猟ołȂ̂ŁBBB
		if( uFlags & CObjLeakCheck::HOOKTARGET_HEAP )
		{
			APIHOOK("kernel32.dll", HeapDestroy);
			APIHOOK("kernel32.dll", HeapCreate);

			APIHOOK("kernel32.dll", LocalFree);
			APIHOOK("kernel32.dll", LocalReAlloc);
			APIHOOK("kernel32.dll", LocalAlloc);

			APIHOOK("kernel32.dll", GlobalFree);
			APIHOOK("kernel32.dll", GlobalReAlloc);
			APIHOOK("kernel32.dll", GlobalAlloc);
		}
#endif

		if( uFlags & CObjLeakCheck::HOOKTARGET_FINDFILE )
		{
			APIHOOK("kernel32.dll", FindClose);
			APIHOOK("kernel32.dll", FindFirstFileA);
			APIHOOK("kernel32.dll", FindFirstFileW);
		}

		if( uFlags & CObjLeakCheck::HOOKTARGET_REG )
		{
			APIHOOK("kernel32.dll", RegCloseKey);
			APIHOOK("kernel32.dll", RegOpenKeyExA);
			APIHOOK("kernel32.dll", RegOpenKeyA);
			APIHOOK("kernel32.dll", RegConnectRegistryA);
			APIHOOK("kernel32.dll", RegCreateKeyExA);
			APIHOOK("kernel32.dll", RegCreateKeyA);

			APIHOOK("kernel32.dll", RegOpenKeyExW);
			APIHOOK("kernel32.dll", RegOpenKeyW);
			APIHOOK("kernel32.dll", RegConnectRegistryW);
			APIHOOK("kernel32.dll", RegCreateKeyExW);
			APIHOOK("kernel32.dll", RegCreateKeyW);
		}

		//APIHOOK("kernel32.dll", CreateToolhelp32Snapshot);
		//APIHOOK("kernel32.dll", CreateToolhelp32Snapshot);

		if( uFlags & CObjLeakCheck::HOOKTARGET_GDI )
		{
			APIHOOK("gdi32.dll", DeleteObject);
			APIHOOK("gdi32.dll", CreatePen);
			APIHOOK("gdi32.dll", CreatePenIndirect);
			APIHOOK("gdi32.dll", ExtCreatePen);
			APIHOOK("gdi32.dll", CreateBrushIndirect);
			APIHOOK("gdi32.dll", CreateDIBPatternBrush);
			APIHOOK("gdi32.dll", CreateDIBPatternBrushPt);
			APIHOOK("gdi32.dll", CreateHatchBrush);
			APIHOOK("gdi32.dll", CreatePatternBrush);
			APIHOOK("gdi32.dll", CreateSolidBrush);

			APIHOOK("gdi32.dll", CreateFontA);
			APIHOOK("gdi32.dll", CreateFontW);
			APIHOOK("gdi32.dll", CreateFontIndirectA);
			APIHOOK("gdi32.dll", CreateFontIndirectW);

			APIHOOK("gdi32.dll", CreateBitmap);
			APIHOOK("gdi32.dll", CreateBitmapIndirect);
			APIHOOK("gdi32.dll", CreateCompatibleBitmap);
			APIHOOK("gdi32.dll", CreateDIBitmap);
			APIHOOK("gdi32.dll", CreateDIBSection);

			APIHOOK("gdi32.dll", CreateEllipticRgn);
			APIHOOK("gdi32.dll", CreateEllipticRgnIndirect);
			APIHOOK("gdi32.dll", CreatePolyPolygonRgn);
			APIHOOK("gdi32.dll", CreatePolygonRgn);
			APIHOOK("gdi32.dll", CreateRectRgn);
			APIHOOK("gdi32.dll", CreateRectRgnIndirect);
			APIHOOK("gdi32.dll", CreateRoundRectRgn);
			APIHOOK("gdi32.dll", ExtCreateRegion);
		}

		if( uFlags & CObjLeakCheck::HOOKTARGET_METAFILE )
		{
			APIHOOK("gdi32.dll", CloseEnhMetaFile);
			APIHOOK("gdi32.dll", CloseMetaFile);
			APIHOOK("gdi32.dll", DeleteEnhMetaFile);
			APIHOOK("gdi32.dll", DeleteMetaFile);
			APIHOOK("gdi32.dll", CopyEnhMetaFileA);
			APIHOOK("gdi32.dll", CopyEnhMetaFileW);
			APIHOOK("gdi32.dll", CopyMetaFileA);
			APIHOOK("gdi32.dll", CopyMetaFileW);
		}

		if( uFlags & CObjLeakCheck::HOOKTARGET_DC )
		{
			APIHOOK("gdi32.dll", DeleteDC);
			APIHOOK("gdi32.dll", CreateDCA);
			APIHOOK("gdi32.dll", CreateDCW);
			APIHOOK("gdi32.dll", CreateCompatibleDC);
			APIHOOK("user32.dll", GetDC);
			APIHOOK("user32.dll", ReleaseDC);
		}

		APIHOOK_END();

		return true;
	}

protected:
	// tbN֐`
	HOOK_FUNC(BOOL	, CloseHandle	, (HANDLE hObject))
	{
		BOOL ret = ((PFN_CloseHandle)GET_ORIGIN_FUNC(CloseHandle))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(CloseHandle));
		}
		return ret;
	}

	// OpenEvent
	HOOK_FUNC(HANDLE, OpenEventA	, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCSTR lpName))
	{
		HANDLE hdle = ((PFN_OpenEventA)GET_ORIGIN_FUNC(OpenEventA))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, OpenEventW	, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCWSTR lpName))
	{
		HANDLE hdle = ((PFN_OpenEventW)GET_ORIGIN_FUNC(OpenEventW))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateEvent
	HOOK_FUNC(HANDLE, CreateEventA	, (LPSECURITY_ATTRIBUTES lpEventAttributes, BOOL bManualReset, BOOL bInitialState, LPCSTR lpName))
	{
		HANDLE hdle = ((PFN_CreateEventA)GET_ORIGIN_FUNC(CreateEventA))(lpEventAttributes, bManualReset, bInitialState, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateEventW	, (LPSECURITY_ATTRIBUTES lpEventAttributes, BOOL bManualReset, BOOL bInitialState, LPCWSTR lpName))
	{
		HANDLE hdle = ((PFN_CreateEventW)GET_ORIGIN_FUNC(CreateEventW))(lpEventAttributes, bManualReset, bInitialState, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateProcess
	HOOK_FUNC(BOOL, CreateProcessA , (LPCSTR lpApplicationName, LPSTR lpCommandLine, LPSECURITY_ATTRIBUTES lpProcessAttributes
		, LPSECURITY_ATTRIBUTES lpThreadAttributes, BOOL bInheritHandles, DWORD dwCreationFlags, LPVOID lpEnvironment
		, LPCSTR lpCurrentDirectory, LPSTARTUPINFOA lpStartupInfo, LPPROCESS_INFORMATION lpProcessInformation))
	{
		BOOL ret = ((PFN_CreateProcessA)GET_ORIGIN_FUNC(CreateProcessA))(lpApplicationName, lpCommandLine, lpProcessAttributes
			, lpThreadAttributes, bInheritHandles, dwCreationFlags, lpEnvironment
			, lpCurrentDirectory, lpStartupInfo, lpProcessInformation);
		if( ret && lpProcessInformation != nullptr )
		{
			if( lpProcessInformation->hProcess != nullptr )
				GetInstance().Add(lpProcessInformation->hProcess, GET_ORIGIN_FUNC(CloseHandle));
			if( lpProcessInformation->hThread != nullptr )
				GetInstance().Add(lpProcessInformation->hThread, GET_ORIGIN_FUNC(CloseHandle));
		}
		return ret;
	}

	HOOK_FUNC(BOOL, CreateProcessW , (LPCWSTR lpApplicationName, LPWSTR lpCommandLine, LPSECURITY_ATTRIBUTES lpProcessAttributes
		, LPSECURITY_ATTRIBUTES lpThreadAttributes, BOOL bInheritHandles, DWORD dwCreationFlags, LPVOID lpEnvironment
		, LPCSTR lpCurrentDirectory, LPSTARTUPINFOA lpStartupInfo, LPPROCESS_INFORMATION lpProcessInformation))
	{
		BOOL ret = ((PFN_CreateProcessW)GET_ORIGIN_FUNC(CreateProcessW))(lpApplicationName, lpCommandLine, lpProcessAttributes
			, lpThreadAttributes, bInheritHandles, dwCreationFlags, lpEnvironment
			, lpCurrentDirectory, lpStartupInfo, lpProcessInformation);
		if( ret && lpProcessInformation != nullptr )
		{
			if( lpProcessInformation->hProcess != nullptr )
				GetInstance().Add(lpProcessInformation->hProcess, GET_ORIGIN_FUNC(CloseHandle));
			if( lpProcessInformation->hThread != nullptr )
				GetInstance().Add(lpProcessInformation->hThread, GET_ORIGIN_FUNC(CloseHandle));
		}
		return ret;
	}

	// CreateFile
	HOOK_FUNC(HANDLE, CreateFileA , (LPCSTR lpFileName, DWORD dwDesiredAccess, DWORD dwShareMode, LPSECURITY_ATTRIBUTES lpSecurityAttributes
		, DWORD dwCreationDisposition, DWORD dwFlagsAndAttributes, HANDLE hTemplateFile) )
	{
		HANDLE hdle = ((PFN_CreateFileA)GET_ORIGIN_FUNC(CreateFileA))(lpFileName, dwDesiredAccess, dwShareMode, lpSecurityAttributes
			, dwCreationDisposition, dwFlagsAndAttributes, hTemplateFile);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateFileW , (LPCWSTR lpFileName, DWORD dwDesiredAccess, DWORD dwShareMode, LPSECURITY_ATTRIBUTES lpSecurityAttributes
		, DWORD dwCreationDisposition, DWORD dwFlagsAndAttributes, HANDLE hTemplateFile) )
	{
		HANDLE hdle = ((PFN_CreateFileW)GET_ORIGIN_FUNC(CreateFileW))(lpFileName, dwDesiredAccess, dwShareMode, lpSecurityAttributes
			, dwCreationDisposition, dwFlagsAndAttributes, hTemplateFile);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateFileMappingA
	HOOK_FUNC(HANDLE, CreateFileMappingA , (HANDLE hFile, LPSECURITY_ATTRIBUTES lpFileMappingAttributes, DWORD flProtect
		, DWORD dwMaximumSizeHigh, DWORD dwMaximumSizeLow, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateFileMappingA)GET_ORIGIN_FUNC(CreateFileMappingA))(hFile, lpFileMappingAttributes, flProtect
			, dwMaximumSizeHigh, dwMaximumSizeLow, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateFileMappingW , (HANDLE hFile, LPSECURITY_ATTRIBUTES lpFileMappingAttributes, DWORD flProtect
		, DWORD dwMaximumSizeHigh, DWORD dwMaximumSizeLow, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateFileMappingW)GET_ORIGIN_FUNC(CreateFileMappingW))(hFile, lpFileMappingAttributes, flProtect
			, dwMaximumSizeHigh, dwMaximumSizeLow, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// OpenFileMappingA
	HOOK_FUNC(HANDLE, OpenFileMappingA , (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenFileMappingA)GET_ORIGIN_FUNC(OpenFileMappingA))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, OpenFileMappingW , (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenFileMappingW)GET_ORIGIN_FUNC(OpenFileMappingW))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateThread
	HOOK_FUNC(HANDLE, CreateThread, (LPSECURITY_ATTRIBUTES lpThreadAttributes, SIZE_T dwStackSize, LPTHREAD_START_ROUTINE lpStartAddress
		, LPVOID lpParameter, DWORD dwCreationFlags, LPDWORD lpThreadId) )
	{
		HANDLE hdle = ((PFN_CreateThread)GET_ORIGIN_FUNC(CreateThread))(lpThreadAttributes, dwStackSize, lpStartAddress
			, lpParameter, dwCreationFlags, lpThreadId);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateMutexA
	HOOK_FUNC(HANDLE, CreateMutexA, (LPSECURITY_ATTRIBUTES lpMutexAttributes, BOOL bInitialOwner, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateMutexA)GET_ORIGIN_FUNC(CreateMutexA))(lpMutexAttributes, bInitialOwner, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateMutexW, (LPSECURITY_ATTRIBUTES lpMutexAttributes, BOOL bInitialOwner, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateMutexW)GET_ORIGIN_FUNC(CreateMutexW))(lpMutexAttributes, bInitialOwner, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// OpenMutexA
	HOOK_FUNC(HANDLE, OpenMutexA, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenMutexA)GET_ORIGIN_FUNC(OpenMutexA))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, OpenMutexW, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenMutexW)GET_ORIGIN_FUNC(OpenMutexW))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateSemaphoreA
	HOOK_FUNC(HANDLE, CreateSemaphoreA, (LPSECURITY_ATTRIBUTES lpSemaphoreAttributes, LONG lInitialCount, LONG lMaximumCount, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateSemaphoreA)GET_ORIGIN_FUNC(CreateSemaphoreA))(lpSemaphoreAttributes, lInitialCount, lMaximumCount, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateSemaphoreW, (LPSECURITY_ATTRIBUTES lpSemaphoreAttributes, LONG lInitialCount, LONG lMaximumCount, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_CreateSemaphoreW)GET_ORIGIN_FUNC(CreateSemaphoreW))(lpSemaphoreAttributes, lInitialCount, lMaximumCount, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// OpenSemaphoreA
	HOOK_FUNC(HANDLE, OpenSemaphoreA, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenSemaphoreA)GET_ORIGIN_FUNC(OpenSemaphoreA))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, OpenSemaphoreW, (DWORD dwDesiredAccess, BOOL bInheritHandle, LPCWSTR lpName) )
	{
		HANDLE hdle = ((PFN_OpenSemaphoreW)GET_ORIGIN_FUNC(OpenSemaphoreW))(dwDesiredAccess, bInheritHandle, lpName);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreateNamedPipeA
	HOOK_FUNC(HANDLE, CreateNamedPipeA, (LPCSTR lpName, DWORD dwOpenMode, DWORD dwPipeMode, DWORD nMaxInstances
		, DWORD nOutBufferSize, DWORD nInBufferSize, DWORD nDefaultTimeOut, LPSECURITY_ATTRIBUTES lpSecurityAttributes) )
	{
		HANDLE hdle = ((PFN_CreateNamedPipeA)GET_ORIGIN_FUNC(CreateNamedPipeA))(lpName, dwOpenMode, dwPipeMode, nMaxInstances
			, nOutBufferSize, nInBufferSize, nDefaultTimeOut, lpSecurityAttributes);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, CreateNamedPipeW, (LPCWSTR lpName, DWORD dwOpenMode, DWORD dwPipeMode, DWORD nMaxInstances
		, DWORD nOutBufferSize, DWORD nInBufferSize, DWORD nDefaultTimeOut, LPSECURITY_ATTRIBUTES lpSecurityAttributes) )
	{
		HANDLE hdle = ((PFN_CreateNamedPipeW)GET_ORIGIN_FUNC(CreateNamedPipeW))(lpName, dwOpenMode, dwPipeMode, nMaxInstances
			, nOutBufferSize, nInBufferSize, nDefaultTimeOut, lpSecurityAttributes);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(CloseHandle));
		}
		return hdle;
	}

	// CreatePipe
	HOOK_FUNC(BOOL, CreatePipe, (PHANDLE hReadPipe, PHANDLE hWritePipe, LPSECURITY_ATTRIBUTES lpPipeAttributes, DWORD nSize) )
	{
		BOOL ret = ((PFN_CreatePipe)GET_ORIGIN_FUNC(CreatePipe))(hReadPipe, hWritePipe, lpPipeAttributes, nSize);
		if( ret )
		{
			if( hReadPipe != nullptr )
				GetInstance().Add(*hReadPipe, GET_ORIGIN_FUNC(CloseHandle));
			if( hWritePipe != nullptr )
				GetInstance().Add(*hWritePipe, GET_ORIGIN_FUNC(CloseHandle));
		}
		return ret;
	}



	HOOK_FUNC(BOOL	, FindClose	, (HANDLE hFindFile))
	{
		BOOL ret = ((PFN_FindClose)GET_ORIGIN_FUNC(FindClose))(hFindFile);
		if( ret )
		{
			GetInstance().Remove(hFindFile, GET_ORIGIN_FUNC(FindClose));
		}
		return ret;
	}
	// FindFirstFileA
	HOOK_FUNC(HANDLE, FindFirstFileA, (LPCSTR lpFileName, LPWIN32_FIND_DATAA lpFindFileData) )
	{
		HANDLE hdle = ((PFN_FindFirstFileA)GET_ORIGIN_FUNC(FindFirstFileA))(lpFileName, lpFindFileData);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(FindClose));
		}
		return hdle;
	}
	HOOK_FUNC(HANDLE, FindFirstFileW, (LPCWSTR lpFileName, LPWIN32_FIND_DATAW lpFindFileData) )
	{
		HANDLE hdle = ((PFN_FindFirstFileW)GET_ORIGIN_FUNC(FindFirstFileW))(lpFileName, lpFindFileData);
		if( hdle != nullptr && hdle != INVALID_HANDLE_VALUE )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(FindClose));
		}
		return hdle;
	}


	HOOK_FUNC(LSTATUS	, RegCloseKey	, (HKEY hKey))
	{
		LSTATUS ret = ((PFN_RegCloseKey)GET_ORIGIN_FUNC(RegCloseKey))(hKey);
		if( ret == ERROR_SUCCESS )
		{
			GetInstance().Remove(hKey, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	// RegOpenKeyExA
	HOOK_FUNC(LSTATUS, RegOpenKeyExA, (HKEY hKey, LPCSTR lpSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegOpenKeyExA)GET_ORIGIN_FUNC(RegOpenKeyExA))(hKey, lpSubKey, ulOptions, samDesired, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	HOOK_FUNC(LSTATUS, RegOpenKeyExW, (HKEY hKey, LPCWSTR lpSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegOpenKeyExW)GET_ORIGIN_FUNC(RegOpenKeyExW))(hKey, lpSubKey, ulOptions, samDesired, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}

	// RegOpenKeyA
	HOOK_FUNC(LSTATUS, RegOpenKeyA, (HKEY hKey, LPCSTR lpSubKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegOpenKeyA)GET_ORIGIN_FUNC(RegOpenKeyA))(hKey, lpSubKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	HOOK_FUNC(LSTATUS, RegOpenKeyW, (HKEY hKey, LPCWSTR lpSubKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegOpenKeyW)GET_ORIGIN_FUNC(RegOpenKeyW))(hKey, lpSubKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}

	// RegCreateKeyExA
	HOOK_FUNC(LSTATUS, RegCreateKeyExA, (HKEY hKey, LPCSTR lpSubKey, DWORD Reserved, LPSTR lpClass
		, DWORD dwOptions, REGSAM samDesired, CONST LPSECURITY_ATTRIBUTES lpSecurityAttributes, PHKEY phkResult, LPDWORD lpdwDisposition) )
	{
		LSTATUS ret = ((PFN_RegCreateKeyExA)GET_ORIGIN_FUNC(RegCreateKeyExA))(hKey, lpSubKey, Reserved, lpClass
			, dwOptions, samDesired, lpSecurityAttributes, phkResult, lpdwDisposition);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	HOOK_FUNC(LSTATUS, RegCreateKeyExW, (HKEY hKey, LPCWSTR lpSubKey, DWORD Reserved, LPSTR lpClass
		, DWORD dwOptions, REGSAM samDesired, CONST LPSECURITY_ATTRIBUTES lpSecurityAttributes, PHKEY phkResult, LPDWORD lpdwDisposition) )
	{
		LSTATUS ret = ((PFN_RegCreateKeyExW)GET_ORIGIN_FUNC(RegCreateKeyExW))(hKey, lpSubKey, Reserved, lpClass
			, dwOptions, samDesired, lpSecurityAttributes, phkResult, lpdwDisposition);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}

	// RegCreateKeyA
	HOOK_FUNC(LSTATUS, RegCreateKeyA, (HKEY hKey, LPCSTR lpSubKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegCreateKeyA)GET_ORIGIN_FUNC(RegCreateKeyA))(hKey, lpSubKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	HOOK_FUNC(LSTATUS, RegCreateKeyW, (HKEY hKey, LPCWSTR lpSubKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegCreateKeyW)GET_ORIGIN_FUNC(RegCreateKeyW))(hKey, lpSubKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}

	// RegConnectRegistryA
	HOOK_FUNC(LSTATUS, RegConnectRegistryA, (LPCSTR lpMachineName, HKEY hKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegConnectRegistryA)GET_ORIGIN_FUNC(RegConnectRegistryA))(lpMachineName, hKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}
	HOOK_FUNC(LSTATUS, RegConnectRegistryW, (LPCWSTR lpMachineName, HKEY hKey, PHKEY phkResult) )
	{
		LSTATUS ret = ((PFN_RegConnectRegistryW)GET_ORIGIN_FUNC(RegConnectRegistryW))(lpMachineName, hKey, phkResult);
		if( ret == ERROR_SUCCESS && phkResult != nullptr )
		{
			GetInstance().Add(*phkResult, GET_ORIGIN_FUNC(RegCloseKey));
		}
		return ret;
	}

	// GDI
	HOOK_FUNC(BOOL	, DeleteObject	, (HGDIOBJ hObject))
	{
		BOOL ret = ((PFN_DeleteObject)GET_ORIGIN_FUNC(DeleteObject))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(DeleteObject));
		}
		return ret;
	}
	HOOK_FUNC(HPEN	, CreatePen	, (int iStyle, int cWidth, COLORREF color))
	{
		HPEN hdle = ((PFN_CreatePen)GET_ORIGIN_FUNC(CreatePen))(iStyle, cWidth, color);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HPEN	, CreatePenIndirect	, (CONST LOGPEN *plpen))
	{
		HPEN hdle = ((PFN_CreatePenIndirect)GET_ORIGIN_FUNC(CreatePenIndirect))(plpen);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HPEN	, ExtCreatePen	, (DWORD iPenStyle, DWORD cWidth, CONST LOGBRUSH *plbrush, DWORD cStyle, CONST DWORD *pstyle))
	{
		HPEN hdle = ((PFN_ExtCreatePen)GET_ORIGIN_FUNC(ExtCreatePen))(iPenStyle, cWidth, plbrush, cStyle, pstyle);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}

	HOOK_FUNC(HBRUSH, CreateBrushIndirect	, (CONST LOGBRUSH *plbrush))
	{
		HBRUSH hdle = ((PFN_CreateBrushIndirect)GET_ORIGIN_FUNC(CreateBrushIndirect))(plbrush);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBRUSH, CreateDIBPatternBrush		, (HGLOBAL h, UINT iUsage))
	{
		HBRUSH hdle = ((PFN_CreateDIBPatternBrush)GET_ORIGIN_FUNC(CreateDIBPatternBrush))(h, iUsage);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBRUSH, CreateDIBPatternBrushPt	, (CONST VOID *lpPackedDIB, UINT iUsage))
	{
		HBRUSH hdle = ((PFN_CreateDIBPatternBrushPt)GET_ORIGIN_FUNC(CreateDIBPatternBrushPt))(lpPackedDIB, iUsage);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBRUSH, CreateHatchBrush	, (int iHatch, COLORREF color))
	{
		HBRUSH hdle = ((PFN_CreateHatchBrush)GET_ORIGIN_FUNC(CreateHatchBrush))(iHatch, color);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBRUSH, CreatePatternBrush	, (HBITMAP hbm))
	{
		HBRUSH hdle = ((PFN_CreatePatternBrush)GET_ORIGIN_FUNC(CreatePatternBrush))(hbm);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBRUSH, CreateSolidBrush	, (COLORREF color))
	{
		HBRUSH hdle = ((PFN_CreateSolidBrush)GET_ORIGIN_FUNC(CreateSolidBrush))(color);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}

	HOOK_FUNC(HFONT	, CreateFontA	, (int cHeight, int cWidth, int cEscapement, int cOrientation, int cWeight, DWORD bItalic
		, DWORD bUnderline, DWORD bStrikeOut, DWORD iCharSet, DWORD iOutPrecision, DWORD iClipPrecision
		, DWORD iQuality, DWORD iPitchAndFamily, LPCSTR pszFaceName))
	{
		HFONT hdle = ((PFN_CreateFontA)GET_ORIGIN_FUNC(CreateFontA))(cHeight, cWidth, cEscapement, cOrientation, cWeight, bItalic
			, bUnderline, bStrikeOut, iCharSet, iOutPrecision, iClipPrecision, iQuality, iPitchAndFamily, pszFaceName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HFONT	, CreateFontW	, (int cHeight, int cWidth, int cEscapement, int cOrientation, int cWeight, DWORD bItalic
		, DWORD bUnderline, DWORD bStrikeOut, DWORD iCharSet, DWORD iOutPrecision, DWORD iClipPrecision
		, DWORD iQuality, DWORD iPitchAndFamily, LPCWSTR pszFaceName))
	{
		HFONT hdle = ((PFN_CreateFontW)GET_ORIGIN_FUNC(CreateFontW))(cHeight, cWidth, cEscapement, cOrientation, cWeight, bItalic
			, bUnderline, bStrikeOut, iCharSet, iOutPrecision, iClipPrecision, iQuality, iPitchAndFamily, pszFaceName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HFONT, CreateFontIndirectA	, (CONST LOGFONTA *lplf))
	{
		HFONT hdle = ((PFN_CreateFontIndirectA)GET_ORIGIN_FUNC(CreateFontIndirectA))(lplf);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HFONT, CreateFontIndirectW	, (CONST LOGFONTW *lplf))
	{
		HFONT hdle = ((PFN_CreateFontIndirectW)GET_ORIGIN_FUNC(CreateFontIndirectW))(lplf);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}

	HOOK_FUNC(HBITMAP, CreateBitmap	, (int nWidth, int nHeight, UINT nPlanes, UINT nBitCount, CONST VOID *lpBits))
	{
		HBITMAP hdle = ((PFN_CreateBitmap)GET_ORIGIN_FUNC(CreateBitmap))(nWidth, nHeight, nPlanes, nBitCount, lpBits);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBITMAP, CreateBitmapIndirect	, (CONST BITMAP *pbm))
	{
		HBITMAP hdle = ((PFN_CreateBitmapIndirect)GET_ORIGIN_FUNC(CreateBitmapIndirect))(pbm);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBITMAP, CreateCompatibleBitmap	, (HDC hdc, int cx, int cy))
	{
		HBITMAP hdle = ((PFN_CreateCompatibleBitmap)GET_ORIGIN_FUNC(CreateCompatibleBitmap))(hdc, cx, cy);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBITMAP, CreateDiscardableBitmap	, (HDC hdc, int cx, int cy))
	{
		HBITMAP hdle = ((PFN_CreateDiscardableBitmap)GET_ORIGIN_FUNC(CreateDiscardableBitmap))(hdc, cx, cy);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBITMAP, CreateDIBitmap	, (HDC hdc, CONST BITMAPINFOHEADER *pbmih, DWORD flInit, CONST VOID *pjBits, CONST BITMAPINFO *pbmi, UINT iUsage))
	{
		HBITMAP hdle = ((PFN_CreateDIBitmap)GET_ORIGIN_FUNC(CreateDIBitmap))(hdc, pbmih, flInit, pjBits, pbmi, iUsage);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HBITMAP, CreateDIBSection	, (HDC hdc, CONST BITMAPINFO *lpbmi, UINT usage, VOID **ppvBits, HANDLE hSection, DWORD offset))
	{
		HBITMAP hdle = ((PFN_CreateDIBSection)GET_ORIGIN_FUNC(CreateDIBSection))(hdc, lpbmi, usage, ppvBits, hSection, offset);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	//HOOK_FUNC(BOOL	, SetWindowRgn	, (HWND hWnd, HRGN hRgn, BOOL bRedraw))
	//{
	//}
	HOOK_FUNC(HRGN, CreateEllipticRgn	, (int x1, int y1, int x2, int y2))
	{
		HRGN hdle = ((PFN_CreateEllipticRgn)GET_ORIGIN_FUNC(CreateEllipticRgn))(x1, y1, x2, y2);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreateEllipticRgnIndirect	, (CONST RECT *lprect))
	{
		HRGN hdle = ((PFN_CreateEllipticRgnIndirect)GET_ORIGIN_FUNC(CreateEllipticRgnIndirect))(lprect);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreatePolyPolygonRgn	, (CONST POINT *pptl, CONST INT *pc, int cPoly, int iMode))
	{
		HRGN hdle = ((PFN_CreatePolyPolygonRgn)GET_ORIGIN_FUNC(CreatePolyPolygonRgn))(pptl, pc, cPoly, iMode);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreatePolygonRgn	, (CONST POINT *pptl, int cPoly, int iMode))
	{
		HRGN hdle = ((PFN_CreatePolygonRgn)GET_ORIGIN_FUNC(CreatePolygonRgn))(pptl, cPoly, iMode);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreateRectRgn	, (int x1, int y1, int x2, int y2))
	{
		HRGN hdle = ((PFN_CreateRectRgn)GET_ORIGIN_FUNC(CreateRectRgn))(x1, y1, x2, y2);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreateRectRgnIndirect	, (CONST RECT *lprect))
	{
		HRGN hdle = ((PFN_CreateRectRgnIndirect)GET_ORIGIN_FUNC(CreateRectRgnIndirect))(lprect);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, CreateRoundRectRgn	, (int x1, int y1, int x2, int y2, int w, int h))
	{
		HRGN hdle = ((PFN_CreateRoundRectRgn)GET_ORIGIN_FUNC(CreateRoundRectRgn))(x1, y1, x2, y2, w, h);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}
	HOOK_FUNC(HRGN, ExtCreateRegion	, (CONST XFORM * lpx, DWORD nCount, CONST RGNDATA * lpData))
	{
		HRGN hdle = ((PFN_ExtCreateRegion)GET_ORIGIN_FUNC(ExtCreateRegion))(lpx, nCount, lpData);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteObject));
		}
		return hdle;
	}


	// meta file
	HOOK_FUNC(BOOL	, DeleteEnhMetaFile	, (HENHMETAFILE hObject))
	{
		BOOL ret = ((PFN_DeleteEnhMetaFile)GET_ORIGIN_FUNC(DeleteEnhMetaFile))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return ret;
	}
	HOOK_FUNC(BOOL	, DeleteMetaFile	, (HENHMETAFILE hObject))
	{
		BOOL ret = ((PFN_DeleteMetaFile)GET_ORIGIN_FUNC(DeleteMetaFile))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return ret;
	}
	HOOK_FUNC(HENHMETAFILE, CloseEnhMetaFile	, (HDC hdc))
	{
		HENHMETAFILE hdle = ((PFN_CloseEnhMetaFile)GET_ORIGIN_FUNC(CloseEnhMetaFile))(hdc);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}
	HOOK_FUNC(HENHMETAFILE, CloseMetaFile	, (HDC hdc))
	{
		HENHMETAFILE hdle = ((PFN_CloseMetaFile)GET_ORIGIN_FUNC(CloseMetaFile))(hdc);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}
	HOOK_FUNC(HENHMETAFILE, CopyEnhMetaFileA	, (HENHMETAFILE hEnh, LPCSTR lpFileName))
	{
		HENHMETAFILE hdle = ((PFN_CopyEnhMetaFileA)GET_ORIGIN_FUNC(CopyEnhMetaFileA))(hEnh, lpFileName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}
	HOOK_FUNC(HENHMETAFILE, CopyEnhMetaFileW	, (HENHMETAFILE hEnh, LPCWSTR lpFileName))
	{
		HENHMETAFILE hdle = ((PFN_CopyEnhMetaFileW)GET_ORIGIN_FUNC(CopyEnhMetaFileW))(hEnh, lpFileName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}
	HOOK_FUNC(HENHMETAFILE, CopyMetaFileA	, (HENHMETAFILE hEnh, LPCSTR lpFileName))
	{
		HENHMETAFILE hdle = ((PFN_CopyMetaFileA)GET_ORIGIN_FUNC(CopyMetaFileA))(hEnh, lpFileName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}
	HOOK_FUNC(HENHMETAFILE, CopyMetaFileW	, (HENHMETAFILE hEnh, LPCWSTR lpFileName))
	{
		HENHMETAFILE hdle = ((PFN_CopyMetaFileW)GET_ORIGIN_FUNC(CopyMetaFileW))(hEnh, lpFileName);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(DeleteEnhMetaFile));
		}
		return hdle;
	}

	// DC

	HOOK_FUNC(BOOL	, DeleteDC	, (HDC hObject))
	{
		BOOL ret = ((PFN_DeleteDC)GET_ORIGIN_FUNC(DeleteDC))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(DeleteDC));
		}
		return ret;
	}
	HOOK_FUNC(HDC	, CreateDCA	, (LPCSTR pwszDriver, LPCSTR pwszDevice, LPCSTR pszPort, CONST DEVMODEA * pdm))
	{
		HDC hdc = ((PFN_CreateDCA)GET_ORIGIN_FUNC(CreateDCA))(pwszDriver, pwszDevice, pszPort, pdm);
		if( hdc != nullptr )
		{
			GetInstance().Add(hdc, GET_ORIGIN_FUNC(DeleteDC));
		}
		return hdc;
	}
	HOOK_FUNC(HDC	, CreateDCW	, (LPCWSTR pwszDriver, LPCWSTR pwszDevice, LPCSTR pszPort, CONST DEVMODEW * pdm))
	{
		HDC hdc = ((PFN_CreateDCW)GET_ORIGIN_FUNC(CreateDCW))(pwszDriver, pwszDevice, pszPort, pdm);
		if( hdc != nullptr )
		{
			GetInstance().Add(hdc, GET_ORIGIN_FUNC(DeleteDC));
		}
		return hdc;
	}
	HOOK_FUNC(HDC	, CreateCompatibleDC	, (HDC hDC))
	{
		HDC hdc = ((PFN_CreateCompatibleDC)GET_ORIGIN_FUNC(CreateCompatibleDC))(hDC);
		if( hdc != nullptr )
		{
			GetInstance().Add(hdc, GET_ORIGIN_FUNC(DeleteDC));
		}
		return hdc;
	}
	HOOK_FUNC(HDC	, GetDC	, (HWND hWnd))
	{
		HDC hdc = ((PFN_GetDC)GET_ORIGIN_FUNC(GetDC))(hWnd);
		if( hdc != nullptr )
		{
			GetInstance().Add(hdc, GET_ORIGIN_FUNC(ReleaseDC));
		}
		return hdc;
	}
	HOOK_FUNC(BOOL	, ReleaseDC	, (HWND hWnd, HDC hObject))
	{
		BOOL ret = ((PFN_ReleaseDC)GET_ORIGIN_FUNC(ReleaseDC))(hWnd, hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(ReleaseDC));
		}
		return ret;
	}


	// memory

	HOOK_FUNC(BOOL	, HeapDestroy	, (HANDLE hObject))
	{
		BOOL ret = ((PFN_HeapDestroy)GET_ORIGIN_FUNC(HeapDestroy))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(HeapDestroy));
		}
		return ret;
	}
	HOOK_FUNC(HANDLE, HeapCreate	, (DWORD dwOptions, SIZE_T InitialSize, SIZE_T MaximumSize))
	{
		HANDLE hdle = ((PFN_HeapCreate)GET_ORIGIN_FUNC(HeapCreate))(dwOptions, InitialSize, MaximumSize);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(HeapDestroy));
		}
		return hdle;
	}

	HOOK_FUNC(BOOL	, LocalFree		, (HLOCAL hObject))
	{
		BOOL ret = ((PFN_LocalFree)GET_ORIGIN_FUNC(LocalFree))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(LocalFree));
		}
		return ret;
	}
	HOOK_FUNC(HLOCAL, LocalReAlloc		, (HLOCAL hObject, SIZE_T uBytes, UINT uFlags))
	{
		HLOCAL hdle = ((PFN_LocalReAlloc)GET_ORIGIN_FUNC(LocalReAlloc))(hObject, uBytes, uFlags);
		GetInstance().Remove(hObject, GET_ORIGIN_FUNC(LocalFree));
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(LocalFree));
		}
		return hdle;
	}
	HOOK_FUNC(HLOCAL, LocalAlloc	, (UINT uFlags, SIZE_T uBytes))
	{
		HLOCAL hdle = ((PFN_LocalAlloc)GET_ORIGIN_FUNC(LocalAlloc))(uFlags, uBytes);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(LocalFree));
		}
		return hdle;
	}

	HOOK_FUNC(BOOL	, GlobalFree		, (HGLOBAL hObject))
	{
		BOOL ret = ((PFN_GlobalFree)GET_ORIGIN_FUNC(GlobalFree))(hObject);
		if( ret )
		{
			GetInstance().Remove(hObject, GET_ORIGIN_FUNC(GlobalFree));
		}
		return ret;
	}
	HOOK_FUNC(HGLOBAL, GlobalReAlloc		, (HLOCAL hObject, SIZE_T uBytes, UINT uFlags))
	{
		HLOCAL hdle = ((PFN_GlobalReAlloc)GET_ORIGIN_FUNC(GlobalReAlloc))(hObject, uBytes, uFlags);
		GetInstance().Remove(hObject, GET_ORIGIN_FUNC(GlobalFree));
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(GlobalFree));
		}
		return hdle;
	}
	HOOK_FUNC(HGLOBAL, GlobalAlloc	, (UINT uFlags, SIZE_T uBytes))
	{
		HGLOBAL hdle = ((PFN_GlobalAlloc)GET_ORIGIN_FUNC(GlobalAlloc))(uFlags, uBytes);
		if( hdle != nullptr )
		{
			GetInstance().Add(hdle, GET_ORIGIN_FUNC(GlobalFree));
		}
		return hdle;
	}

};

IRIS_MSC_PRAGMA_WARNING_END()

}

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CObjLeakCheck::CObjLeakCheck(void)
{
	CObjLeakCheckImpl::GetInstance().Initialize(HOOKTARGET_ALL);
}

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CObjLeakCheck::CObjLeakCheck(int enableTarget)
{
	CObjLeakCheckImpl::GetInstance().Initialize(enableTarget);
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CObjLeakCheck::~CObjLeakCheck(void)
{
	CObjLeakCheckImpl::GetInstance().Terminate();
}

}	// end of namespace dbg
}	// end of namespace wx
}	// end of namespace iris

#endif
