//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		WXRvlRemote.cpp
 * @brief		WXRvlRemote t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 *
 * @par			copyright
 * Copyright (C) 2010 t.sirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
 *
 * protocol infomation : http://wiibrew.org/wiki/Wii_Remote
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_WXRvlRemote_CPP_

//======================================================================
// include
#include "WXRvlRemote.h"
#include "../WXHIDPreparsedData.h"
#include "iris_debug.h"

#if	defined(_IRIS_SUPPORT_WDK)

namespace iris {
namespace wx {
namespace rvl
{

//======================================================================
// R}ht@X
/*
IO	ID(s)		Size	Function
O	0x10		1		Unknown
O	0x11		1		Player LEDs
O	0x12		2		Data Reporting mode
O	0x13		1		IR Camera Enable
O	0x14		1		Speaker Enable
O	0x15		1		Status Information Request
O	0x16		21		Write Memory and Registers
O	0x17		6		Read Memory and Registers
O	0x18		21		Speaker Data
O	0x19		1		Speaker Mute
O	0x1a		1		IR Camera Enable 2
I	0x20		6		Status Information
I	0x21		21		Read Memory and Registers Data
I	0x22		4		Acknowledge output report, return function result
I	0x30-0x3f	2-21	Data reports
*/

//======================================================================
// define
// R}hID
#define RR_O_CMD_UNKNOWN			0x10	// unknown (Size:1)
#define RR_O_CMD_LED				0x11	// LED (Size:1)
#define RR_O_CMD_DATA_REPORT_MODE	0x12	// Data Reporting mode (Size:2)
#define RR_O_CMD_IR_CAMERA			0x13	// IR Camera Enable (Size:1)
#define RR_O_CMD_SPEAKER			0x14	// Speaker Enable (Size:1)
#define RR_O_CMD_REQ_STATUS			0x15	// Status Request (Size:1)
#define RR_O_CMD_WRITE_MEM			0x16	// Write Memory and Registers (Size:21)
#define RR_O_CMD_READ_MEM			0x17	// Read Memory and Registers (Size:6)
#define RR_O_CMD_SPEAKER_DATA		0x18	// Speaker Data (Size:21)
#define RR_O_CMD_SPEAKER_MUTE		0x19	// Speaker Mute (Size:1)
#define RR_O_CMD_IR_CAMERA2			0x1a	// IR Camera Enable 2 (Size:1)
#define RR_I_CMD_STATUS				0x20	// Status (Size:6)
#define RR_I_CMD_READ_MEM			0x21	// Read Memory and Registers Data (Size:21)
#define RR_I_CMD_REPORT				0x22	// Acknowledge output report, return function result (Size:4)
#define RR_I_CMD_CORE_BTN			0x30	// Read Core Button (0x30 BB BB)
#define RR_I_CMD_BTN_AND_ACC		0x31	// Read Core Button And Accelerometer (0x31 BB BB AA AA AA)
#define RR_I_CMD_BTN_WITH_8EX		0x32	// Read Core Button Adn 8 Extension Bytes (0x32 BB BB EE EE EE EE EE EE EE EE)
#define RR_I_CMD_BTN_ACC_12IR		0x33	// Read Core Buttons and Accelerometer with 12 IR bytes (0x33 BB BB AA AA AA II * 12)
#define RR_I_CMD_BTN_WITH_19EX		0x34	// Read Core Button Adn 19 Extension Bytes (0x34 BB BB EE * 19)
#define RR_I_CMD_BTN_ACC_16EX		0x35	// Read Core Buttons and Accelerometer with 16 Extension Bytes (0x35 BB BB AA AA AA E * 16)
#define RR_I_CMD_BTN_10IR_9EX		0x36	// Read Core Buttons with 10 IR bytes and 9 Extension Bytes (0x36 BB BB II * 10 EE * 9)
#define RR_I_CMD_BTN_ACC_10IR_6EX	0x37	// Read Core Buttons and Accelerometer with 10 IR bytes and 6 Extension Bytes (0x37 BB BB AA AA AA II * 10 EE * 6)
#define RR_I_CMD_EXTENSION			0x3d	// Read 21 Extension Bytes
#define RR_I_CMD_BTN_ACC_36IR_LOW	0x3e	// Read Interleaved Core Buttons and Accelerometer with 36 IR bytes (0x3e BB BB AA II * 18)
#define RR_I_CMD_BTN_ACC_36IR_HI	0x3f	// Read Interleaved Core Buttons and Accelerometer with 36 IR bytes (0x3f BB BB AA II * 18)

#define RR_STATUS_BTN1_INDEX		1
#define RR_STATUS_BTN2_INDEX		2
#define RR_STATUS_FLAG_INDEX		3
#define RR_STATUS_BATTERY_INDEX		6

#define RR_CMD_ON					0x04
#define RR_CMD_OFF					0x00

#define RR_READWRITE_REG			0x04
#define RR_READWRITE_MEM			0x00

#define RR_IR_CAMERA_SENSBLOCK1_OFFSET			0xB00000
#define RR_IR_CAMERA_SENSBLOCK1_SIZE			9
#define RR_IR_CAMERA_SENSBLOCK1_MARCAN(_buf)	(_buf[0] = 0x00,  _buf[1] = 0x00, _buf[2] = 0x00, _buf[3] = 0x00, _buf[4] = 0x00, _buf[5] = 0x00, _buf[6] = 0x90, _buf[7] = 0x00, _buf[8] = 0xC0)

#define RR_IR_CAMERA_SENSBLOCK2_OFFSET			0xB0001A
#define RR_IR_CAMERA_SENSBLOCK2_SIZE			2
#define RR_IR_CAMERA_SENSBLOCK2_MARCAN(_buf)	(_buf[0] = 0x40,  _buf[1] = 0x00)

#define RR_IR_CAMERA_MODENUM_OFFSET				0xB00033
#define RR_IR_CAMERA_MODENUM_SIZE				1
#define RR_IR_CAMERA_MODENUM(_buf)				_buf[0] = 0x33

//======================================================================
// class

/**********************************************************************//**
 *
 * ΏۃfoCXǂf
 *
 -----------------------------------------------------------------------
 * @param [in]	lpDevicePath	= foCX̃pX
 * @return	
*//***********************************************************************/
bool CRvlRemote::IsSpecifiedDevice(LPCTSTR lpDevicePath)
{
	return CHID::IsSpecifiedDevice(lpDevicePath, CRvlRemote::VENDOR_ID, CRvlRemote::PRODUCT_ID);
}

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CRvlRemote::CRvlRemote(void)
: m_BatteryLevel(0)
, m_Flags(0)
, m_Status(0)
{
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CRvlRemote::~CRvlRemote(void)
{
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [in]	lpDevicePath	= foCXpX
 * @return	
*//***********************************************************************/
bool CRvlRemote::Initialize(LPCTSTR lpDevicePath)
{
	if( !Open(lpDevicePath, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE
		, nullptr, 0, OPEN_EXISTING, nullptr) ) return false;

	// ĊmF
	do
	{
		HIDD_ATTRIBUTES attr;
		if( !GetAttributes(&attr) ) break;
		if( (attr.VendorID != VENDOR_ID) ||
			(attr.ProductID != PRODUCT_ID) ) break;

		// \͎擾
		CHIDPreparsedData pd;
		if( !GetPreparsedData(&pd()) ) break;
		if( !pd.GetCaps(&m_Caps) ) break;

		// o̓obt@m
		m_Input.reserve(m_Caps.InputReportByteLength);
		m_Output.reserve(m_Caps.OutputReportByteLength);

		return true;
	} while(0);

	Close();
	return false;
}

/**********************************************************************//**
 *
 * XV
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::Update(void)
{
	if( !IsValid() ) return false;

	ReadCommand();
	return true;
}

/**********************************************************************//**
 *
 * Ԃ̍Ď擾
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::Reflesh(void)
{
	if( !IsValid() ) return false;
	if( !ReadStatus() ) return false;
	return true;
}

/**********************************************************************//**
 *
 * Xe[^X̎擾
 *
 -----------------------------------------------------------------------
 * @return	Xe[^X
*//***********************************************************************/
u8 CRvlRemote::GetStatus(void) const
{
	return m_Status;
}

/**********************************************************************//**
 *
 * LED ̖
 *
 -----------------------------------------------------------------------
 * @param [in]	LedMask	= LED ̐ݒ
 * @return	
*//***********************************************************************/
bool CRvlRemote::SetLED(u8 LedMask)
{
	u8 led = (LedMask & 0xF) << 4;
	m_Output[0] = RR_O_CMD_LED;
	m_Output[1] = led;
	if( m_Flags.IsFlag(FLAG_RUMBLE) )
	{
		m_Output[1] |= 0x01;
	}
	m_Status.EnableMaskBits(STATUS_LED, led);
	return Write();
}

/**********************************************************************//**
 *
 * LED ̏Ԏ擾
 *
 -----------------------------------------------------------------------
 * @return	LED ̐ݒ}XN
*//***********************************************************************/
u8 CRvlRemote::GetLED(void) const
{
	return m_Status >> 4;
}

/**********************************************************************//**
 *
 * obe[cʂ킸ǂ
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CRvlRemote::IsBatteryLittle(void) const
{
	return m_Status.IsBits(STATUS_BATTERY_LITTLE);
}

/**********************************************************************//**
 *
 * U̐ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	rumble	= ON/OFF
 * @return	
*//***********************************************************************/
bool CRvlRemote::SetRumble(bool rumble)
{
	m_Flags.SetFlag(FLAG_RUMBLE, rumble);
	return SetLED(GetLED());
}

/**********************************************************************//**
 *
 * Uǂ
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CRvlRemote::IsRumble(void) const
{
	return m_Flags.IsFlag(FLAG_RUMBLE);
}

/**********************************************************************//**
 *
 * Xs[J[̐ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	enable	= ON/OFF
 * @return	
*//***********************************************************************/
bool CRvlRemote::SetSpeaker(bool enable)
{
	m_Output[0] = RR_O_CMD_SPEAKER;
	m_Output[1] = enable ? RR_CMD_ON : RR_CMD_OFF;
	return Write();
}

/**********************************************************************//**
 *
 * Xs[J[̃~[g
 *
 -----------------------------------------------------------------------
 * @param [in]	enable	= ON/OFF
 * @return	
*//***********************************************************************/
bool CRvlRemote::MuteSpeaker(bool enable)
{
	m_Output[0] = RR_O_CMD_SPEAKER_MUTE;
	m_Output[1] = enable ? RR_CMD_ON : RR_CMD_OFF;
	return Write();
}

/**********************************************************************//**
 *
 * Xs[J[̃f[^ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= obt@
 * @param [in]	size	= TCY[1, 20]
 * @return	
*//***********************************************************************/
bool CRvlRemote::SetSpeakerData(const u8* src, u8 size)
{
	IRIS_ASSERT( size <= 20 );
	m_Output[0] = RR_O_CMD_SPEAKER_DATA;
	m_Output[1] = size;
	for( u8 i=0; i < size; ++i )
	{
		m_Output[i+2] = src[i];
	}
	return Write();
}

/**********************************************************************//**
 *
 * Xs[J[̐ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	fmt				= tH[}bg(FMT_4BIT_ADPCM or FMT_8BIT_PCM)
 * @param [in]	sampling_rate	= TvO[g
 * @param [in]	volume			= {[[0,1]
 * @return	
*//***********************************************************************/
bool CRvlRemote::SetSpeakerConfig(u8 fmt, u32 sampling_rate, f32 volume)
{
	IRIS_ASSERT(fmt == FMT_4BIT_ADPCM || fmt == FMT_8BIT_PCM);
	if( !IsValid() ) return false;
	if( volume < 0.0f ) volume = 0.0f;
	if( volume > 1.0f ) volume = 1.0f;
	u8 reg1[1] = { 0x01 };
	u8 reg2[1] = { 0x08 };
	u16 rate = 0;
	u8 vol = 0;
	if( fmt == FMT_4BIT_ADPCM )
	{
		rate = (u16)( 6000000/sampling_rate);
		vol = (u8)(0x40 * volume);
	}
	else
	{
		rate = (u16)(12000000/sampling_rate);
		vol = (u8)(0xFF * volume);
	}
	u8 conf[7] = { 0x00, fmt, rate&0xFF, (rate>>8)&0xFF, vol, 0x00 };
	if( !WriteRegister(reg1, 0xa20009, 1) ) return false;
	if( !WriteRegister(reg2, 0xa20001, 1) ) return false;

	if( !WriteRegister(conf, 0xa20001, 7) ) return false;
	if( !WriteRegister(reg1, 0xa20008, 1) ) return false;
	return true;
}

/**********************************************************************//**
 *
 * f[^tH[}bg̐ݒ
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CRvlRemote::SetDataFormat(u8 format)
{
	IRIS_ASSERT( format >= RR_I_CMD_CORE_BTN && format < 0x40);
	m_Output[0] = RR_O_CMD_DATA_REPORT_MODE;
	m_Output[1] = 0;
	m_Output[2] = format;
	if( !Write() ) return false;
	m_DataFormat = format;
	return true;
}

/**********************************************************************//**
 *
 * J̋N
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::EnableIRCamera(void)
{
	if( !IsValid() ) return false;
	m_Output[0] = RR_O_CMD_IR_CAMERA;
	m_Output[1] = RR_CMD_ON;
	if( !Write() ) return false;

	m_Output[0] = RR_O_CMD_IR_CAMERA2;
	m_Output[1] = RR_CMD_ON;
	if( !Write() ) return false;

	u8 mem[16];
	mem[0] = 0x08;
	if( !WriteRegister(mem, 0xB00030, 1) ) return false;

	RR_IR_CAMERA_SENSBLOCK1_MARCAN(mem);
	if( !WriteRegister(mem, RR_IR_CAMERA_SENSBLOCK1_OFFSET, RR_IR_CAMERA_SENSBLOCK1_SIZE) ) return false;

	RR_IR_CAMERA_SENSBLOCK2_MARCAN(mem);
	if( !WriteRegister(mem, RR_IR_CAMERA_SENSBLOCK2_OFFSET, RR_IR_CAMERA_SENSBLOCK2_SIZE) ) return false;

	RR_IR_CAMERA_MODENUM(mem);
	if( !WriteRegister(mem, RR_IR_CAMERA_MODENUM_OFFSET, RR_IR_CAMERA_MODENUM_SIZE) ) return false;

	mem[0] = 0x08;
	if( !WriteRegister(mem, 0xB00030, 1) ) return false;
	return true;
}

/**********************************************************************//**
 *
 * ւ̏
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ݃TCY[1,16]
 * @return	
*//***********************************************************************/
bool CRvlRemote::WriteMemory(const u8* src, u32 offset, u8 size)
{
	return WriteMemoryRegister(RR_READWRITE_MEM, src, offset, size);
}

/**********************************************************************//**
 *
 * ̓ǂݍ
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= ǂݍ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ǂݍ݃TCY
 * @return	
*//***********************************************************************/
bool CRvlRemote::ReadMemory(u8* dst, u32 offset, u16 size)
{
	return ReadMemoryRegister(RR_READWRITE_MEM, dst, offset, size);
}

/**********************************************************************//**
 *
 * WX^ւ̏
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ݃TCY[1,16]
 * @return	
*//***********************************************************************/
bool CRvlRemote::WriteRegister(const u8* src, u32 offset, u8 size)
{
	return WriteMemoryRegister(RR_READWRITE_REG, src, offset, size);
}

/**********************************************************************//**
 *
 * WX^̓ǂݍ
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= ǂݍ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ǂݍ݃TCY
 * @return	
*//***********************************************************************/
bool CRvlRemote::ReadRegister(u8* dst, u32 offset, u16 size)
{
	return ReadMemoryRegister(RR_READWRITE_REG, dst, offset, size);
}

/**********************************************************************//**
 *
 *  or WX^ւ̏
 *
 -----------------------------------------------------------------------
 * @param [in]	memreg	= WX^
 * @param [in]	src		= ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ݃TCY[1,16]
 * @return	
*//***********************************************************************/
bool CRvlRemote::WriteMemoryRegister(u8 memreg, const u8* src, u32 offset, u8 size)
{
	IRIS_ASSERT( src != nullptr );
	IRIS_ASSERT( size <= 16 );
	m_Output[0] = RR_O_CMD_WRITE_MEM;
	m_Output[1] = memreg;
	m_Output[2] = (offset>>16) & 0xFF;
	m_Output[3] = (offset>>8) & 0xFF;
	m_Output[4] = (offset>>0) & 0xFF;
	m_Output[5] = size;
	for( u8 i=0; i < size; ++i )
	{
		m_Output[i+5] = src[i];
	}
	return Write();
}

/**********************************************************************//**
 *
 *  or WX^̓ǂݍ
 *
 -----------------------------------------------------------------------
 * @param [in]	memreg	= WX^
 * @param [out]	dst		= ǂݍ݃obt@
 * @param [in]	offset	= ItZbg
 * @param [in]	size	= ǂݍ݃TCY
 * @return	
*//***********************************************************************/
bool CRvlRemote::ReadMemoryRegister(u8 memreg, u8* dst, u32 offset, u16 size)
{
	IRIS_ASSERT( dst != nullptr );
	if( !IsValid() ) return false;
	m_Output[0] = RR_O_CMD_READ_MEM;
	m_Output[1] = memreg;
	m_Output[2] = (offset>>16) & 0xFFu;
	m_Output[3] = (offset>>8) & 0xFFu;
	m_Output[4] = (offset>>0) & 0xFFu;
	m_Output[5] = (size>>8) & 0xFFu;
	m_Output[6] = (size>>0) & 0xFFu;
	if( !Write() ) return false;
	u16 total = 0;
	while(1)
	{
		if( ReadCommand() )
		{
			if( m_Input[0] == RR_I_CMD_READ_MEM )
			{
				if( m_Input[3] & 0xF ) return false;
				u8 read = ((m_Input[3] >> 4) & 0xF) + 1;
				for( u8 i=0; i < read; ++i, ++dst )
				{
					*dst = m_Input[i+4];
				}
				total += read;
				if( total >= size ) break;
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * Xe[^X̓ǂݍ
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::ReadStatus(void)
{
	if( !IsValid() ) return false;
	m_Output[0] = RR_O_CMD_REQ_STATUS;
	if( m_Flags.IsFlag(FLAG_RUMBLE) )
	{
		m_Output[1] = 0x01;
	}
	else
	{
		m_Output[1] = 0x00;
	}
	if( !Write() ) return false;
	while(1)
	{
		if( ReadCommand() )
		{
			if( m_Input[0] == RR_I_CMD_STATUS )
				break;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * ǂݍ & 
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::ReadCommand(void)
{
	if( !Read() ) return false;
	m_Button = static_cast<u16>((m_Input[RR_STATUS_BTN1_INDEX] << 8) | (m_Input[RR_STATUS_BTN2_INDEX]));

	switch( m_Input[0] )
	{
	case RR_I_CMD_STATUS:
		OnReadStatus();
		break;

	case RR_I_CMD_READ_MEM:
		break;

	case RR_I_CMD_REPORT:
		break;

	case RR_I_CMD_CORE_BTN:
	case RR_I_CMD_BTN_WITH_8EX:
	case RR_I_CMD_BTN_WITH_19EX:
		break;
	case RR_I_CMD_BTN_AND_ACC:
		for( int i=0; i < 3; ++i )
			m_Acc[i] = m_Input[i+3];
		break;

	case RR_I_CMD_BTN_ACC_12IR:
		for( int i=0; i < 3; ++i )
			m_Acc[i] = m_Input[i+3];
		break;

	case RR_I_CMD_BTN_ACC_16EX:
		for( int i=0; i < 3; ++i )
			m_Acc[i] = m_Input[i+3];
		break;

	case RR_I_CMD_BTN_10IR_9EX:
		break;

	case RR_I_CMD_BTN_ACC_10IR_6EX:
		for( int i=0; i < 3; ++i )
			m_Acc[i] = m_Input[i+3];
		break;

	case RR_I_CMD_EXTENSION:
	case RR_I_CMD_BTN_ACC_36IR_LOW:
	case RR_I_CMD_BTN_ACC_36IR_HI:
		break;
	}
	return true;
}

/**********************************************************************//**
 *
 * ǂݍ & 
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
void CRvlRemote::OnReadStatus(void)
{
	u8 prev = m_Status;
	m_Status = m_Input[RR_STATUS_FLAG_INDEX];
	m_BatteryLevel = m_Input[RR_STATUS_BATTERY_INDEX];

	prev ^= m_Status;
	if( prev & STATUS_CONNECT_EXCTRL )
	{
		// R[obN֐̃R[
	}
	if( prev & STATUS_BATTERY_LITTLE )
	{
	}
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::Write(void)
{
	if( !IsValid() ) return false;
	DWORD dwWritten = 0;
	if( !CHID::Write(m_Output, m_Output.size(), &dwWritten, nullptr) ) return false;
	return dwWritten != m_Output.size();
}

/**********************************************************************//**
 *
 * ǂݍ
 *
 -----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
bool CRvlRemote::Read(void)
{
	if( !IsValid() ) return false;
	DWORD dwRead = 0;
	if( !CHID::Read(m_Input, m_Input.size(), &dwRead, nullptr) ) return false;
	return ( dwRead != m_Input.size() );
}

}	// end of namespace rvl
}	// end of namespace wx
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "unit/UnitCore.h"
#include "iris_iostream.h"
#include "iris_using.h"

IRIS_UNITTEST(CRvlRemoteUnitTest, Func)
{
}

#endif

#endif
