//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		iris_types_endian.h
 * @brief		iris_types_endian t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_iris_types_endian_H_
#define INCG_IRIS_iris_types_endian_H_

//======================================================================
// include
#include "iris_types_alias.h"

//======================================================================
// declare
// union
union IrisEndian16;
union IrisEndian32;
union IrisEndian64;

//======================================================================
// union
//! endianΉϐ(16bit)
typedef union IrisEndian16
{
	IrisU16	uv;
	IrisS16 sv;
	struct
	{
		IrisU8 byte[2];
	};
	struct
	{
#ifdef __BIG_ENDIAN__
		IrisU8	c1;
		IrisU8	c2;
#else
		IrisU8	c2;
		IrisU8	c1;
#endif
	};
} IrisEndian16;

//! endianΉϐ(32bit)
typedef union IrisEndian32
{
	IrisU32	uv;
	IrisS32 sv;
	struct
	{
		IrisU8 byte[4];
	};
	struct
	{
#ifdef __BIG_ENDIAN__
		IrisU8	c1;
		IrisU8	c2;
		IrisU8	c3;
		IrisU8	c4;
#else
		IrisU8	c4;
		IrisU8	c3;
		IrisU8	c2;
		IrisU8	c1;
#endif
	};
	struct
	{
		IrisU16	s1;
		IrisU16	s2;
	};
} IrisEndian32;

//! endianΉϐ(64bit)
typedef union IrisEndian64
{
	IrisU64	uv;
	IrisS64 sv;
	struct
	{
		IrisU8 byte[8];
	};
	struct
	{
#ifdef __BIG_ENDIAN__
		IrisU8	c1;
		IrisU8	c2;
		IrisU8	c3;
		IrisU8	c4;
		IrisU8	c5;
		IrisU8	c6;
		IrisU8	c7;
		IrisU8	c8;
#else
		IrisU8	c8;
		IrisU8	c7;
		IrisU8	c6;
		IrisU8	c5;
		IrisU8	c4;
		IrisU8	c3;
		IrisU8	c2;
		IrisU8	c1;
#endif
	};
	struct
	{
		IrisU16	s1;
		IrisU16	s2;
		IrisU16	s3;
		IrisU16	s4;
	};
	struct
	{
		IrisU32	l1;
		IrisU32	l2;
	};
} IrisEndian64;

#if defined(__cplusplus)

namespace iris
{

//======================================================================
// typedef
typedef IrisEndian16 endian16;
typedef IrisEndian32 endian32;
typedef IrisEndian64 endian64;

//======================================================================
// struct
template<typename TN>
struct endian_base
{
	static TN reverse(TN v)
	{
		TN ret;
		const int size = sizeof(TN);
		const u8* src = pointer_cast<const u8*>(&v);
		u8* dst = pointer_cast<u8*>(&ret);
		dst += size-1;
		for( int i=0; i < size; ++i, ++src, --dst )
		{
			*dst = *src;
		}
		return ret;
	}
	struct type
	{
		TN	m_v;
		type(TN v=0) : m_v(v)		{}
		type& operator = (TN v)		{ m_v = reverse(v); return *this; }
		operator TN (void) const	{ return reverse(m_v); }
	};
};

#ifdef __BIG_ENDIAN__

typedef s16		bes16;
typedef u16		beu16;
typedef s32		bes32;
typedef u32		beu32;

typedef endian_base<s16>::type	les16;
typedef endian_base<u16>::type	leu16;
typedef endian_base<s32>::type	les32;
typedef endian_base<u32>::type	leu32;

#else

typedef endian_base<s16>::type		bes16;
typedef endian_base<u16>::type		beu16;
typedef endian_base<s32>::type		bes32;
typedef endian_base<u32>::type		beu32;

typedef s16		les16;
typedef u16		leu16;
typedef s32		les32;
typedef u32		leu32;

#endif

}	// end of namespace iris

#endif

#endif
