﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		NaclFile.h
 * @brief		NaclFile ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_NaclFile_H_
#define INCG_IRIS_NaclFile_H_

//======================================================================
// include
#include "../nacl_inchead.h"
#include "../../../fnd/io/FndIFile.h"

#if defined(IRIS_NACL)

//======================================================================
// include
#include "NaclFileIO.h"
#include "NaclFileRef.h"
#include "NaclFileSystem.h"

namespace iris {
namespace nacl
{

//======================================================================
// class
/**
 * @ingroup	NativeClient
 * @brief	ファイルクラス
*/
class CFile : public fnd::IFile
{
protected:
	pp::FileSystem&	m_rFileSystem;
	CFileRef	m_Ref;
	CFileIO		m_File;
	int64_t		m_Offset;
	PP_FileInfo	m_Info;
public:
	/**
	 * @brief	コンストラクタ
	*/
	CFile(pp::Instance* instance, pp::FileSystem& system)
		: m_rFileSystem(system)
		, m_Ref()
		, m_File(instance)
		, m_Offset(0)
	{
	}

	/**
	 * @brief	デストラクタ
	*/
	~CFile(void)
	{
		Close();
	}

public:
	virtual bool	OpenA(LPCSTR  lpFileName, u32 mode)
	{
		CFileRef	ref(m_rFileSystem, lpFileName);
		m_Ref = ref;
		uint32_t flags = 0;
		if( mode & fnd::FOPEN_READ )	flags |= PP_FILEOPENFLAG_READ;
		if( mode & fnd::FOPEN_WRITE )	flags |= PP_FILEOPENFLAG_WRITE;
		//if( mode & fnd::FOPEN_READLOCK )	flags &= ~FILE_SHARE_READ;
		//if( mode & fnd::FOPEN_WRITELOCK )	flags &= ~FILE_SHARE_WRITE;
		//if( mode & fnd::FOPEN_APPEND )	flags = OPEN_EXISTING;
		if( mode & fnd::FOPEN_CREATE )		flags |= PP_FILEOPENFLAG_CREATE;
		if( mode & fnd::FOPEN_TRUNC )		flags |= PP_FILEOPENFLAG_TRUNCATE;
		if( mode & fnd::FOPEN_CREATENEW )	flags |= PP_FILEOPENFLAG_EXCLUSIVE | PP_FILEOPENFLAG_CREATE;
		//if( mode & fnd::FOPEN_EXCL )		flags |= 0;
		//if( mode & fnd::FOPEN_POWERLOCK )	flags |= 0;

		int32_t hr = m_File.Open(m_Ref, (PP_FileOpenFlags)flags);
		if( NACL_FAILED(hr) ) return false;
		hr = m_File.Query(&m_Info);
		m_Offset = 0;
		return NACL_SUCCEEDED(hr);
	}
	virtual bool	OpenW(LPCWSTR lpFileName, u32 mode)
	{
		return false;
	}
	virtual bool	IsOpen(void)	const
	{
		return false;
	}
	/**
	 * 閉じる
	 * @return	成否
	*/
	virtual bool	Close(void)
	{
		m_File.Close();
		m_Ref.Close();
		return true;
	}
	/**
	 * seek
	 * @param [in]	offset	= オフセット
	 * @param [in]	origin	= 計算方法
	 * @return	成否
	*/
	virtual bool	Seek(long offset, int origin)
	{
		switch(origin)
		{
		case fnd::FILE_SEEK_SET:
			m_Offset = static_cast<int64_t>(offset);
			break;
		case fnd::FILE_SEEK_CUR:
			m_Offset += static_cast<int64_t>(offset);
			break;
		case fnd::FILE_SEEK_END:
			//m_Offset = static_cast<int64_t>(offset);
			return false;
		}
		return true;
	}
	/**
	 * tell
	 * @return	現在のファイル位置
	*/
	virtual s32		Tell(void)		const
	{
		return m_Offset;
	}
	/**
	 * ファイルサイズ取得
	 * @return ファイルサイズ（バイト単位）
	*/
	virtual u32		GetSize(void)	const
	{
		return m_Info.size;
	}

	/**
	 * 読み込み
	 * @param [out]	buf		= 出力バッファ
	 * @param [in]	size	= 出力バッファサイズ
	 * @return	読み込みサイズ
	*/
	virtual u32		Read(void* buf, u32 size)
	{
		return m_File.Read(m_Offset, static_cast<char*>(buf), size);
	}
	virtual u32		ReadAsync(void* buf, u32 size)
	{
		return m_File.Read(m_Offset, static_cast<char*>(buf), size);
	}
	/**
	 * 書き込み
	 * @param [in]	buf		= 入力バッファ
	 * @param [in]	size	= 入力バッファサイズ
	 * @return	書き込みサイズ
	*/
	virtual u32		Write(const void* buf, u32 size)
	{
		return m_File.Write(m_Offset, static_cast<const char*>(buf), size);
	}
	virtual u32		WriteAsync(const void* buf, u32 size)
	{
		return m_File.Write(m_Offset, static_cast<const char*>(buf), size);
	}
};

}	// end of namespace nacl
}	// end of namespace iris

#endif

#endif
