﻿//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		NaclFileIO.h
 * @brief		NaclFileIO ファイル
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_NaclFileIO_H_
#define INCG_IRIS_NaclFileIO_H_

//======================================================================
// include
#include "../nacl_inchead.h"
#include "../../../iris_object.h"

#if defined(IRIS_NACL)

//======================================================================
// include
#include <ppapi/cpp/file_io.h>
#include <ppapi/c/ppb_file_io.h>
#include <ppapi/cpp/completion_callback.h>

namespace iris {
namespace nacl
{

//======================================================================
// class
/**
 * @ingroup	NativeClient
 * @brief	File IO クラス
*/
class CFileIO : public IIrisObject
{
protected:
	typedef pp::CompletionCallbackFactory<CFileIO>	Factory;

protected:
	pp::FileIO	m_FileIO;
	Factory		m_Factory;


public:
	// コンストラクタ
	CFileIO(pp::Instance* instance) 
		: m_FileIO(instance)
		, m_Factory(this)
	{
	}

public:
	CFileIO&	operator = (const CFileIO& rhs)	{ m_FileIO = rhs.m_FileIO; return *this; }

public:
	/**
	 * @brief	FileIO の取得
	*/
			pp::FileIO&	GetFileIO(void)			{ return m_FileIO; }
	const	pp::FileIO&	GetFileIO(void)	const	{ return m_FileIO; }

	operator const pp::FileIO&	(void)	const	{ return m_FileIO; }

public:
	/**
	 * @brief	開く
	 * @param [in]	ref				= FileRef
	 * @param [in]	option_flags	= フラグ
	 * @return	結果
	*/
	int32_t	Open(const pp::FileRef& ref, int32_t option_flags)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnOpen);
		return m_FileIO.Open(ref, option_flags, cc);
	}

	/**
	 * @brief	読み込み
	 * @param [in]	offsert			= オフセット
	 * @param [out]	buffer			= 出力バッファ
	 * @param [in]	bytes_to_read	= 読み込みサイズ
	 * @return	結果
	*/
	int32_t	Read(int64_t offset, char* buffer, int32_t bytes_to_read)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnRead);
		return m_FileIO.Read(offset, buffer, bytes_to_read, cc);
	}

	/**
	 * @brief	書き込み
	 * @param [in]	offsert			= オフセット
	 * @param [in]	buffer			= 入力バッファ
	 * @param [in]	bytes_to_write	= 書き込みサイズ
	 * @return	結果
	*/
	int32_t	Write(int64_t offset, const char* buffer, int32_t bytes_to_write)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnWrite);
		return m_FileIO.Write(offset, buffer, bytes_to_write, cc);
	}

	/**
	 * @brief	Flush
	 * @return	結果
	*/
	int32_t	Flush(void)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnFlush);
		return m_FileIO.Flush(cc);
	}

	/**
	 * @brief	Query
	 * @param [out]	result_buf	= 出力
	 * @return	結果
	*/
	int32_t	Query(PP_FileInfo* result_buf)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnQuery);
		return m_FileIO.Query(result_buf, cc);
	}

	/**
	 * @brief	
	 * @return	結果
	*/
	int32_t	Touch(PP_Time last_access_time, PP_Time last_modified_time)
	{
		pp::CompletionCallback cc = m_Factory.NewRequiredCallback(&CFileIO::OnTouch);
		return m_FileIO.Touch(last_access_time, last_modified_time, cc);
	}

	/**
	 * @brief	閉じる
	*/
	void	Close(void)
	{
		m_FileIO.Close();
	}

protected:
	/**
	 * @brief	コールバック関数
	 * @param [in]	result	= 結果
	*/
	virtual void	OnOpen(int32_t result)
	{
		IRIS_UNUSED_VAR(result);
		OutputDebugVar( FormatMessage(result) );
	}

	/**
	 * @brief	コールバック関数
	 * @param [in]	bytes_read	= 読み込んだサイズ
	*/
	virtual void	OnRead(int32_t bytes_read)
	{
		IRIS_UNUSED_VAR(bytes_read);
		OutputDebugVar(bytes_read);
	}

	/**
	 * @brief	コールバック関数
	 * @param [in]	bytes_written	= 書き込んだサイズ
	*/
	virtual void	OnWrite(int32_t bytes_written)
	{
		IRIS_UNUSED_VAR(bytes_written);
		OutputDebugVar("OnWrite");
		OutputDebugVar(bytes_written);
	}

	/**
	 * @brief	コールバック関数
	 * @param [in]	result	= 結果
	*/
	virtual void	OnFlush(int32_t result)
	{
		IRIS_UNUSED_VAR(result);
		OutputDebugVar( FormatMessage(result) );
	}

	/**
	 * @brief	コールバック関数
	 * @param [in]	result	= 結果
	*/
	virtual void	OnQuery(int32_t result)
	{
		IRIS_UNUSED_VAR(result);
		OutputDebugVar( FormatMessage(result) );
	}

	/**
	 * @brief	コールバック関数
	 * @param [in]	result	= 結果
	*/
	virtual void	OnTouch(int32_t result)
	{
		IRIS_UNUSED_VAR(result);
	}
};

}	// end of namespace nacl
}	// end of namespace iris

#endif

#endif
