/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */
package jp.sourceforge.jpnvalidator.util;

import java.io.UnsupportedEncodingException;
import java.util.regex.Pattern;

import jp.sourceforge.jpnvalidator.constant.FixedValue;

/**
 * SimpleStringUtil<br>
 * <br>
 * Simple utility on String value.<br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */
public class SimpleStringUtil
{

	private static Pattern PATTERN_CARRIAGE_RETURN;
	private static Pattern PATTERN_LINE_FEED;
	static
	{
		PATTERN_CARRIAGE_RETURN = Pattern.compile(FixedValue.CARRIAGE_RETURN_STRING);
		PATTERN_LINE_FEED = Pattern.compile(FixedValue.LINE_FEED_STRING);
	}

	/**
	 * Convert LowerCamelCase to UndersocreLowerCase
	 * 
	 * @param lccStr
	 *            str(LowerCamelCase)
	 * @return str(UnderscoreLowerCase)
	 */
	public static String convertLCCase2ULCase(String lccStr)
	{
		char[] charArray = lccStr.toCharArray();
		String resultStr = FixedValue.EMPTY_STRING;
		for (int j = 0; j < charArray.length; j++)
		{
			if (Character.isUpperCase(charArray[j]))
			{
				resultStr = resultStr + FixedValue.UNDER_SCORE_STRING + charArray[j];
			} else
			{
				resultStr += Character.toUpperCase(charArray[j]);
			}
		}
		return resultStr;
	}

	/**
	 * Convert UnderscoreLowerCase to LowerCamelCase
	 * 
	 * @param lccStr
	 *            str(UnderscoreLowerCase)
	 * @return str(LowerCamelCase)
	 */
	public static String convertULCase2LCCase(String ulcStr)
	{
		char[] charArray = ulcStr.toCharArray();
		String resultStr = FixedValue.EMPTY_STRING;
		boolean upperCharFlag = true;
		for (int j = 0; j < charArray.length; j++)
		{
			if (Character.toString(charArray[j]).equals(FixedValue.UNDER_SCORE_STRING))
			{
				upperCharFlag = true;
				continue;
			} else if (upperCharFlag)
			{
				resultStr += Character.toUpperCase(charArray[j]);
				upperCharFlag = false;
			} else
			{
				resultStr += Character.toLowerCase(charArray[j]);
				upperCharFlag = false;
			}
		}
		return resultStr;
	}

	/**
	 * 
	 * Replace all line breaks to CRLF
	 * 
	 * @param str
	 *            target String value
	 * @return result String value
	 */
	public static String replaceAllLF2CRLF(String str)
	{
		String retStr = str;
		retStr = PATTERN_CARRIAGE_RETURN.matcher(str).replaceAll(FixedValue.EMPTY_STRING);
		retStr = PATTERN_LINE_FEED.matcher(retStr).replaceAll(
				FixedValue.CARRIAGE_RETURN_STRING + FixedValue.LINE_FEED_STRING);
		return retStr;
	}

	/**
	 * Remove all CR from arg String value
	 * 
	 * @param str
	 *            arg String value
	 * @return result String value
	 */
	public static String removeCarriageReturnAll(String str)
	{
		str = PATTERN_CARRIAGE_RETURN.matcher(str).replaceAll(FixedValue.EMPTY_STRING);
		return str;
	}

	/**
	 * Remove all LF from arg String value
	 * 
	 * @param str
	 *            arg String value
	 * @return result String value
	 */
	public static String removeLineFeedAll(String str)
	{
		str = PATTERN_LINE_FEED.matcher(str).replaceAll(FixedValue.EMPTY_STRING);
		return str;
	}

	/**
	 * 
	 * Convert arg byte array to String value made from encoded byte array.
	 * 
	 * @param byteAr
	 *            target byte array
	 * @param encoding
	 *            encoding to String value
	 * @return String value that is made from encoding byte array
	 */
	public static String convertByteArray2String(byte[] byteAr, String encoding)
	{
		String returnString = null;
		try
		{
			returnString = new String(byteAr, encoding);
		} catch (UnsupportedEncodingException ignore)
		{
		}
		// remove CR
		returnString = PATTERN_CARRIAGE_RETURN.matcher(returnString).replaceAll(
				FixedValue.EMPTY_STRING);
		return returnString;
	}

}