#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "jtk.h"

/* .bmp file */
typedef struct BMP_FileHeader BMP_FileHeader;
struct BMP_FileHeader
{
	char bf_type[2];
	unsigned long bf_size;
	unsigned short bf_reserved1;
	unsigned short bf_reserved2;
	unsigned long bf_offbits;
};

typedef union BMP_InfoHeader BMP_InfoHeader;
union BMP_InfoHeader
{
	int type;	/* type=0 : OS2, type=1 : Windows */
	
	struct{
		int type;
		
		unsigned long size;
		short width;
		short height;
		unsigned short planes;
		unsigned short bit_count;
	}info_os2;
	
	struct{
		int type;
		
		unsigned long size;
		long width;
		long height;
		unsigned short planes;
		unsigned short bit_count;
		unsigned long compression;
		unsigned long size_image;
		long x_pix_per_meter;
		long y_pix_per_meter;
		unsigned long clr_used;
		unsigned long clr_important;
	}info_win;
};

typedef struct BMP_ColorPalette BMP_ColorPalette;
struct BMP_ColorPalette
{
	unsigned char blue;
	unsigned char green;
	unsigned char red;
	unsigned char reserved;
};

static JtkError bmp_load_file_header(FILE *fp, BMP_FileHeader *bfh)
{
	if(fread(bfh->bf_type, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fread(&bfh->bf_size, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fread(&bfh->bf_reserved1, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fread(&bfh->bf_reserved2, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fread(&bfh->bf_offbits, 4, 1, fp) != 1)
		return JTK_ERROR;
	
	if((bfh->bf_type[0] != 'B') ||
		(bfh->bf_type[1] != 'M'))
		return JTK_ERROR;
	
	return JTK_NOERROR;
}

static JtkError bmp_load_info_header(FILE *fp, BMP_InfoHeader *bih)
{
	unsigned long ih_size;
	
	if(fread(&ih_size, 4, 1, fp) != 1)
		return JTK_ERROR;
	
	if(ih_size == 12){
		bih->type = 0;
		bih->info_os2.size = ih_size;
	}else if(ih_size == 40){
		bih->type = 1;
		bih->info_win.size = ih_size;
	}else{
		return JTK_ERROR;
	}
	
	if(bih->type == 0){
		if(fread(&bih->info_os2.width, 2, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_os2.height, 2, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_os2.planes, 2, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_os2.bit_count, 2, 1, fp) != 1)
			return JTK_ERROR;
	}
	
	if(bih->type == 1){
		if(fread(&bih->info_win.width, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.height, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.planes, 2, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.bit_count, 2, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.compression, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.size_image, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.x_pix_per_meter, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.y_pix_per_meter, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.clr_used, 4, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bih->info_win.clr_important, 4, 1, fp) != 1)
			return JTK_ERROR;
	}
	
	return JTK_NOERROR;
}

static int bmp_get_image_width(BMP_InfoHeader *bih)
{
	if(bih->type == 0)
		return bih->info_os2.width;
	if(bih->type == 1)
		return bih->info_win.width;
	
	return 0;
}

static int bmp_get_image_height(BMP_InfoHeader *bih)
{
	if(bih->type == 0)
		return bih->info_os2.height;
	if(bih->type == 1)
		return bih->info_win.height;
	
	return 0;
}

static int bmp_get_offbits(BMP_FileHeader *bfh)
{
	return bfh->bf_offbits;
}

static int bmp_get_bit_count(BMP_InfoHeader *bih)
{
	if(bih->type == 0)
		return bih->info_os2.bit_count;
	if(bih->type == 1)
		return bih->info_win.bit_count;
	
	return 0;
}

static JtkError bmp_get_palette_num(int *palette_num, BMP_InfoHeader *bih)
{
	if(bih->type == 0){
		switch(bih->info_os2.bit_count){
		case 1:
			*palette_num = 2;
			return JTK_NOERROR;
		case 4:
			*palette_num = 16;
			return JTK_NOERROR;
		case 8:
			*palette_num = 256;
			return JTK_NOERROR;
		case 16:
		case 24:
		case 32:
			*palette_num = 0;
			return JTK_NOERROR;
		}
		return JTK_ERROR;
	}
	
	if(bih->type == 1){
		switch(bih->info_win.bit_count){
		case 1:
			*palette_num = 2;
			return JTK_NOERROR;
		case 4:
			*palette_num = 16;
			return JTK_NOERROR;
		case 8:
			*palette_num = 256;
			return JTK_NOERROR;
		case 16:
		case 24:
		case 32:
			*palette_num =0;
			return JTK_NOERROR;
		}
		return JTK_ERROR;
	}
	
	return JTK_ERROR;
}

static BMP_ColorPalette* bmp_create_color_palette(int num)
{
	return malloc(sizeof(BMP_ColorPalette) * num);
}

static void bmp_destroy_color_palette(BMP_ColorPalette *bcp)
{
	free(bcp);
}

static JtkError bmp_load_color_palette_os2(FILE *fp, int palette_num, BMP_ColorPalette *bcp)
{
	int i;
	
	for(i=0; i<palette_num; i++){
		if(fread(&bcp[i].blue, 1, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bcp[i].green, 1, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bcp[i].red, 1, 1, fp) != 1)
			return JTK_ERROR;
	}
	
	return JTK_NOERROR;
}

static JtkError bmp_load_color_palette_win(FILE *fp, int palette_num, BMP_ColorPalette *bcp)
{
	int i;
	
	for(i=0; i<palette_num; i++){
		if(fread(&bcp[i].blue, 1, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bcp[i].green, 1, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bcp[i].red, 1, 1, fp) != 1)
			return JTK_ERROR;
		if(fread(&bcp[i].reserved, 1, 1, fp) != 1)
			return JTK_ERROR;
	}
	
	return JTK_NOERROR;
}

static void bmp_get_pixel(
	JtkPixel *pixel,
	char *line_data,
	int bit_count,
	int x,
	BMP_ColorPalette *bcp)
{
	char *pixel_data;

	switch(bit_count){
	case 1:
		break;
	case 4:
		break;
	case 8:
		break;
	case 16:
		break;
	case 24:
		pixel_data = line_data + 3 * x;
		pixel->b = pixel_data[0];
		pixel->g = pixel_data[1];
		pixel->r = pixel_data[2];
		break;
	case 32:
		pixel_data = line_data + 4 * x;
		pixel->b = pixel_data[0];
		pixel->g = pixel_data[1];
		pixel->r = pixel_data[2];
		break;
	}
}

static JtkImage* bmp_load_image_file(char *name)
{
	FILE *fp;
	BMP_FileHeader bfh;
	BMP_InfoHeader bih;
	int width;
	int height;
	int palette_num;
	BMP_ColorPalette *bcp = NULL;
	int offbits;
	int bit_count;
	int line_size;
	char *line_data;
	int fp_pos;
	JtkImage *image;
	JtkPixel pixel;
	int i, j;
	
	if((fp = fopen(name, "r")) == NULL)
		return NULL;
	
	if(bmp_load_file_header(fp, &bfh) == JTK_ERROR){
		fclose(fp);
		return NULL;
	}
	if(bmp_load_info_header(fp, &bih) == JTK_ERROR){
		fclose(fp);
		return NULL;
	}
	width = bmp_get_image_width(&bih);
	height = bmp_get_image_height(&bih);
	if((width == 0) || (height == 0)){
		fclose(fp);
		return NULL;
	}
	if(bmp_get_palette_num(&palette_num, &bih) == JTK_ERROR){
		fclose(fp);
		return NULL;
	}
	if(palette_num != 0){
		bcp = bmp_create_color_palette(palette_num);
		if(bcp == NULL){
			fclose(fp);
			return NULL;
		}
		if(bih.type == 0){
			if(bmp_load_color_palette_os2(fp, palette_num, bcp) == JTK_ERROR){
				fclose(fp);
				return NULL;
			}
		}
		if(bih.type == 1){
			if(bmp_load_color_palette_win(fp, palette_num, bcp) == JTK_ERROR){
				fclose(fp);
				return NULL;
			}
		}
	}
	
	offbits = bmp_get_offbits(&bfh);
	bit_count = bmp_get_bit_count(&bih);
	
	line_size = (width * bit_count) / 8;
	if((line_size % 4) != 0){
		line_size = ((line_size / 4) + 1) * 4;
	}
	
	if((line_data = malloc(line_size)) == NULL){
		fclose(fp);
		if(palette_num != 0)
			bmp_destroy_color_palette(bcp);
		return NULL;
	}
	
	if((image = jtkCreateImage(width, height)) == NULL){
		fclose(fp);
		if(palette_num != 0)
			bmp_destroy_color_palette(bcp);
		free(line_data);
		return NULL;
	}
	
	for(i=0; i<height; i++){
		fp_pos = offbits + line_size * (height - (i + 1));
		if(fseek(fp, fp_pos, SEEK_SET) != 0){
			fclose(fp);
			if(palette_num != 0)
				bmp_destroy_color_palette(bcp);
			free(line_data);
			jtkDestroyImage(image);
			return NULL;
		}
		if(fread(line_data, line_size, 1, fp) != 1){
			fclose(fp);
			if(palette_num != 0)
				bmp_destroy_color_palette(bcp);
			free(line_data);
			jtkDestroyImage(image);
			return NULL;
		}
		for(j=0; j<width; j++){
			bmp_get_pixel(&pixel, line_data, bit_count, j, bcp);
			jtkPutImagePixel(image, &pixel, j, i);
		}
	}
	
	fclose(fp);
	if(palette_num != 0)
		bmp_destroy_color_palette(bcp);
	free(line_data);
	return image;
}

static JtkError bmp_save_file_header(JtkImage *image, FILE *fp)
{
	/* Windows Bitmap / 24Bit Color Format */
	
	char bfType[2] = {'B', 'M'};
	unsigned long bfSize;
	unsigned short bfReserved1;
	unsigned short bfReserved2;
	unsigned long bfOffBits;
	
	unsigned long line_data_size;
	
	line_data_size = image->width * 3;
	if((line_data_size % 4) != 0){
		line_data_size = ((line_data_size / 4) + 1) * 4;
	}
	
	bfSize = 54 + (line_data_size * image->height);
	bfReserved1 = 0;
	bfReserved2 = 0;
	bfOffBits = 54;
	
	if(fwrite(bfType, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&bfSize, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&bfReserved1, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&bfReserved2, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&bfOffBits, 4, 1, fp) != 1)
		return JTK_ERROR;
	
	return JTK_NOERROR;
}

static JtkError bmp_save_info_header(JtkImage *image, FILE *fp)
{
	/* Windows Bitmap / 24Bit Color Format */
	
	unsigned long biSize = 40;
	long biWidth = image->width;
	long biHeight = image->height;
	unsigned short biPlanes = 1;
	unsigned short biBitCount = 24;
	unsigned long biCompression = 0;
	unsigned long biSizeImage = 0;
	long biXPixPerMeter = 0;
	long biYPixPerMeter = 0;
	unsigned long biClrUsed = 0;
	unsigned long biClrImportant = 0;
	
	if(fwrite(&biSize, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biWidth, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biHeight, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biPlanes, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biBitCount, 2, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biCompression, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biSizeImage, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biXPixPerMeter, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biYPixPerMeter, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biClrUsed, 4, 1, fp) != 1)
		return JTK_ERROR;
	if(fwrite(&biClrImportant, 4, 1, fp) != 1)
		return JTK_ERROR;
	
	return JTK_NOERROR;
}

static JtkError bmp_save_image(JtkImage *image, FILE *fp)
{
	/* Windows Bitmap / 24Bit Color Format */
	
	unsigned long line_data_size;
	unsigned long line_pad_size;
	int i;
	unsigned long position;
	int j;
	unsigned char d;
	unsigned char pad = 0;
	JtkPixel pixel;
	
	line_data_size = image->width * 3;
	if((line_data_size % 4) != 0){
		line_data_size = ((line_data_size / 4) + 1) * 4;
	}
	line_pad_size = line_data_size - (image->width * 3);
	
	for(i=0; i<image->height; i++){
		position = 54 + line_data_size * (image->height - (i + 1));
		if(fseek(fp, position, SEEK_SET) != 0)
			return JTK_ERROR;
		
		for(j=0; j<image->width; j++){
			jtkGetImagePixel(image, &pixel, j, i);
			d = pixel.b;
			if(fwrite(&d, 1, 1, fp) != 1)
				return JTK_ERROR;
			d = pixel.g;
			if(fwrite(&d, 1, 1, fp) != 1)
				return JTK_ERROR;
			d = pixel.r;
			if(fwrite(&d, 1, 1, fp) != 1)
				return JTK_ERROR;
		}
		for(j=0; j<line_pad_size; j++){
			if(fwrite(&pad, 1, 1, fp) != 1)
				return JTK_ERROR;
		}
	}
	
	return JTK_NOERROR;
}

static void bmp_save_image_file(JtkImage *image, char *name)
{
	/* Windows Bitmap / 24Bit Color Format */
	
	FILE *fp;
	
	if((fp = fopen(name, "w")) == NULL)
		return;
	if(bmp_save_file_header(image, fp) != JTK_NOERROR){
		fclose(fp);
		return;
	}
	if(bmp_save_info_header(image, fp) != JTK_NOERROR){
		fclose(fp);
		return;
	}
	if(bmp_save_image(image, fp) != JTK_NOERROR){
		fclose(fp);
		return;
	}
	
	fclose(fp);
}

/* JTK Image File API */
JtkImage* jtkLoadImageFile(char *name, char *type)
{
	JtkImage *image = NULL;
	
	if(name == NULL)
		return NULL;
	
	if(image == NULL)
		image = bmp_load_image_file(name);
	
	return image;
}

void jtkSaveImageFile(JtkImage *image, char *name, char *type)
{
	if(image == NULL)
		return;
	if(name == NULL)
		return;
	
	if((type == NULL) || (strcmp(type, "bmp") == 0))
		bmp_save_image_file(image, name);
}

