#include "jwk.h"

/* Button */
typedef struct Button Button;
struct Button
{
	JwkWindow *jw;
	JtkGC *gc;
	char *text;
};

static void callback_button(JwkEvent *event)
{
	JtkEvent je;
	JtkWidget *widget = event->any.widget;
	Button *button = widget->widget;

	switch(event->type){
	case JWK_EVENT_EXPOSE:
		{
			JtkSize sb, st;
			
			if(widget->callback != NULL){
				je.type = JTK_EVENT_EXPOSE;
				je.expose.widget = widget;
				je.expose.data = widget->data;
				je.expose.x = event->expose.x;
				je.expose.y = event->expose.y;
				je.expose.width = event->expose.width;
				je.expose.height = event->expose.height;
				je.expose.count = event->expose.count;
				widget->callback(&je);
			}
			if(button->text != NULL){
				jwkGetWindowSize(button->jw, &sb);
				jtkGetTextSize(button->gc, button->text, j_strlen(button->text), &st);
				jtkDrawText(button->gc,
							(sb.width - st.width)/2, 
							(sb.height - st.height)/2,
							button->text, j_strlen(button->text));
			}
		}
		break;
	case JWK_EVENT_BUTTONDOWN:
		{
			if(widget->callback != NULL){
				je.type = JTK_EVENT_CLICK;
				je.click.widget = widget;
				je.click.data = widget->data;
				je.click.button = event->buttondown.button;
				widget->callback(&je);
			}
		}
		break;
	case JWK_EVENT_ENTER:
		{
			if(widget->callback != NULL){
				je.type = JTK_EVENT_ENTER;
				je.enter.widget = widget;
				je.enter.data = widget->data;
				je.enter.px = event->enter.px;
				je.enter.py = event->enter.py;
				je.enter.screen_px = event->enter.screen_px;
				je.enter.screen_py = event->enter.screen_py;
				je.enter.keymask = event->enter.keymask;
				widget->callback(&je);
			}
		}
		break;
	case JWK_EVENT_LEAVE:
		{
			if(widget->callback != NULL){
				je.type = JTK_EVENT_LEAVE;
				je.leave.widget = widget;
				je.leave.data = widget->data;
				je.leave.px = event->leave.px;
				je.leave.py = event->leave.py;
				je.leave.screen_px = event->leave.screen_px;
				je.leave.screen_py = event->leave.screen_py;
				je.leave.keymask = event->leave.keymask;
				widget->callback(&je);
			}
		}
		break;
	default:
		break;
	}
}

static JtkGC* create_button_gc(JtkWidget *widget, JtkGCMode mode, JtkGCType type)
{
	Button *button = widget->widget;
	
	return jwkCreateGC(button->jw, mode, type);
}

static void destroy_button(JtkWidget *widget)
{
	Button *button = widget->widget;

	jtkDestroyGC(button->gc);
	jwkDestroyWindow(button->jw);
	if(button->text != NULL)
		j_free(button->text);
	j_free(button);
}

static void map_button(JtkWidget *widget)
{
	Button *button = widget->widget;
	
	jwkMapWindow(button->jw);
}

static void unmap_button(JtkWidget *widget)
{
	Button *button = widget->widget;

	jwkUnmapWindow(button->jw);
}

static void clear_button(JtkWidget *widget,
	int px, int py, int width, int height, JtkBool exposures)
{
	Button *button = widget->widget;
	
	jwkClearWindow(button->jw,
		px, py, width, height, exposures);
}

static void set_button_pos(JtkWidget *widget, int px, int py)
{
	Button *button = widget->widget;

	jwkSetWindowPos(button->jw, px, py);
}

static void set_button_size(JtkWidget *widget, int width, int height)
{
	Button *button = widget->widget;

	jwkSetWindowSize(button->jw, width, height);
}

static void get_button_pos(JtkWidget *widget, JtkPoint *point)
{
	Button *button = widget->widget;
	
	jwkGetWindowPos(button->jw, point);
}

static void get_button_size(JtkWidget *widget, JtkSize *size)
{
	Button *button = widget->widget;
	
	jwkGetWindowSize(button->jw, size);
}

static void set_button_color(JtkWidget *widget, JtkColor color)
{
	Button *button = widget->widget;
	
	jwkSetWindowColor(button->jw, color);
}

static void set_button_border_size(JtkWidget *widget, int size)
{
	Button *button = widget->widget;
	
	jwkSetWindowBorderSize(button->jw, size);
}

static void set_button_border_color(JtkWidget *widget, JtkColor color)
{
	Button *button = widget->widget;
	
	jwkSetWindowBorderColor(button->jw, color);
}

static void set_button_text_color(JtkWidget *widget, JtkColor color)
{
	Button *button = widget->widget;
	
	jtkSetColor(button->gc, color);
}

static void set_button_text(JtkWidget *widget, char *text)
{
	Button *button = widget->widget;

	if(button->text != NULL){
		j_free(button->text);
		button->text = NULL;
	}
	
	if(text == NULL){
		jwkClearWindow(button->jw, 0, 0, 0, 0, JTK_TRUE);
		return;
	}
	
	button->text = j_malloc(j_strlen(text) + 1);
	if(button->text == NULL)
		return;
	j_zero(button->text, j_strlen(text) + 1);
	j_strcpy(button->text, text);
	jwkClearWindow(button->jw, 0, 0, 0, 0, JTK_TRUE);
}

static void clear_button_text(JtkWidget *widget)
{
	set_button_text(widget, NULL);
}

static JwkWidgetFunc button_func =
{
	.create_gc = create_button_gc,
	.destroy = destroy_button,
	.map = map_button,
	.unmap = unmap_button,
	.clear = clear_button,
	.set_pos = set_button_pos,
	.set_size = set_button_size,
	.get_pos = get_button_pos,
	.get_size = get_button_size,
	.set_color = set_button_color,
	.set_border_size = set_button_border_size,
	.set_border_color = set_button_border_color,
	.set_text_color = set_button_text_color,
	.set_text = set_button_text,
	.clear_text = clear_button_text,
};

JtkWidget* jtkCreateButton(JtkWidget *parent)
{
	JtkWidget *widget;
	Button *button;
	
	if(parent == NULL)
		return NULL;
	
	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	button = j_malloc(sizeof(Button));
	if(button == NULL){
		j_free(widget);
		return NULL;
	}
	
	j_zero(widget, sizeof(JtkWidget));
	j_zero(button, sizeof(Button));
	
	button->jw = jwkCreateWindow(jwkGetContainer(parent));
	button->gc = jwkCreateGC(button->jw, JTK_GCMODE_SCREEN, JTK_GCTYPE_NORMAL);
	
	widget->widget = button;
	widget->func = &button_func;
	
	jwkSetWindowEventMask(button->jw,
		JWK_EVENTMASK_EXPOSE |
		JWK_EVENTMASK_BUTTONDOWN |
		JWK_EVENTMASK_ENTER |
		JWK_EVENTMASK_LEAVE);
	jwkSetWidget(button->jw, callback_button, widget);
	jwkSetWindowColor(button->jw, jtkRGB(192,192,192));
	jwkSetWindowBorderSize(button->jw, 1);
	jwkSetWindowBorderColor(button->jw, jtkRGB(128,128,128));
	
	return widget;
}
