#include "jwk.h"

typedef struct RemovableBox RemovableBox;
struct RemovableBox
{
	JwkWindow *frame;
	JwkWindow *container;
	JwkWindow *title_bar;
	char *title_text;
	JtkGC *title_gc;
	int title_bar_height;
	
	JtkPoint pos;
	JtkBool moving;
};

static void callback_rb_frame(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	RemovableBox *rb = widget->widget;

	switch(event->type){
	case JWK_EVENT_MOVE:
		{
			JtkEvent je;
			
			je.type = JTK_EVENT_MOVE;
			je.move.widget = widget;
			je.move.data = widget->data;
			je.move.px = event->move.px;
			je.move.py = event->move.py;
			if(widget->callback != NULL)
				widget->callback(&je);
		}
		break;
	case JWK_EVENT_RESIZE:
		jwkSetWindowSize(rb->title_bar,
			event->resize.width,
			rb->title_bar_height);
		jwkSetWindowSize(rb->container,
			event->resize.width,
			event->resize.height - rb->title_bar_height);
		break;
	case JWK_EVENT_DESTROY:
		{
			JtkEvent je;
			
			je.type = JTK_EVENT_DESTROY;
			je.destroy.widget = widget;
			je.destroy.data = widget->data;
			if(widget->callback != NULL)
				widget->callback(&je);
		}
		break;
	default:
		break;
	}
}

static void callback_rb_container(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	JtkEvent je;
	
	switch(event->type){
		case JWK_EVENT_KEYDOWN:
		je.type = JTK_EVENT_KEYDOWN;
		je.keydown.widget = widget;
		je.keydown.data = widget->data;
		je.keydown.px = event->keydown.px;
		je.keydown.py = event->keydown.py;
		je.keydown.screen_px = event->keydown.screen_px;
		je.keydown.screen_py = event->keydown.screen_py;
		je.keydown.keymask = event->keydown.keymask;
		je.keydown.keysym = event->keydown.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_KEYUP:
		je.type = JTK_EVENT_KEYUP;
		je.keyup.widget = widget;
		je.keyup.data = widget->data;
		je.keyup.px = event->keyup.px;
		je.keyup.py = event->keyup.py;
		je.keyup.screen_px = event->keyup.screen_px;
		je.keyup.screen_py = event->keyup.screen_py;
		je.keyup.keymask = event->keyup.keymask;
		je.keyup.keysym = event->keyup.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_BUTTONDOWN:
		je.type = JTK_EVENT_BUTTONDOWN;
		je.buttondown.widget = widget;
		je.buttondown.data = widget->data;
		je.buttondown.px = event->buttondown.px;
		je.buttondown.py = event->buttondown.py;
		je.buttondown.screen_px = event->buttondown.screen_px;
		je.buttondown.screen_py = event->buttondown.screen_py;
		je.buttondown.button = event->buttondown.button;
		je.buttondown.keymask = event->buttondown.keymask;
		je.buttondown.keysym = event->buttondown.keysym;
		jtkRaiseWidget(widget);
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_BUTTONUP:
		je.type = JTK_EVENT_BUTTONUP;
		je.buttonup.widget = widget;
		je.buttonup.data = widget->data;
		je.buttonup.px = event->buttonup.px;
		je.buttonup.py = event->buttonup.py;
		je.buttonup.screen_px = event->buttonup.screen_px;
		je.buttonup.screen_py = event->buttonup.screen_py;
		je.buttonup.button = event->buttonup.button;
		je.buttonup.keymask = event->buttonup.keymask;
		je.buttonup.keysym = event->buttonup.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_MOTION:
		je.type = JTK_EVENT_MOTION;
		je.motion.widget = widget;
		je.motion.data = widget->data;
		je.motion.px = event->motion.px;
		je.motion.py = event->motion.py;
		je.motion.screen_px = event->motion.screen_px;
		je.motion.screen_py = event->motion.screen_py;
		je.motion.keymask = event->motion.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_ENTER:
		je.type = JTK_EVENT_ENTER;
		je.enter.widget = widget;
		je.enter.data = widget->data;
		je.enter.px = event->enter.px;
		je.enter.py = event->enter.py;
		je.enter.screen_px = event->enter.screen_px;
		je.enter.screen_py = event->enter.screen_py;
		je.enter.keymask = event->enter.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_LEAVE:
		je.type = JTK_EVENT_LEAVE;
		je.leave.widget = widget;
		je.leave.data = widget->data;
		je.leave.px = event->leave.px;
		je.leave.py = event->leave.py;
		je.leave.screen_px = event->leave.screen_px;
		je.leave.screen_py = event->leave.screen_py;
		je.leave.keymask = event->leave.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_MAP:
		je.type = JTK_EVENT_MAP;
		je.map.widget = widget;
		je.map.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_UNMAP:
		je.type = JTK_EVENT_UNMAP;
		je.unmap.widget = widget;
		je.unmap.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_RESIZE:
		je.type = JTK_EVENT_RESIZE;
		je.resize.widget = widget;
		je.resize.data = widget->data;
		je.resize.width = event->resize.width;
		je.resize.height = event->resize.height;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_EXPOSE:
		je.type = JTK_EVENT_EXPOSE;
		je.expose.widget = widget;
		je.expose.data = widget->data;
		je.expose.x = event->expose.x;
		je.expose.y = event->expose.y;
		je.expose.width = event->expose.width;
		je.expose.height = event->expose.height;
		je.expose.count = event->expose.count;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	default:
		break;
	}
}

static void callback_rb_titlebar(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	RemovableBox *rb = widget->widget;

	switch(event->type){
	case JWK_EVENT_BUTTONDOWN:
		if(event->buttondown.button == JTK_MOUSE_BUTTON_LEFT){
			rb->pos.px = event->buttondown.px;
			rb->pos.py = event->buttondown.py;
			rb->moving = JTK_TRUE;
			jwkRaiseWindow(rb->frame);
		}
		break;
	case JWK_EVENT_BUTTONUP:
		if(event->buttonup.button == JTK_MOUSE_BUTTON_LEFT)
			rb->moving = JTK_FALSE;
		break;
	case JWK_EVENT_MOTION:
		if(rb->moving == JTK_TRUE){
			JtkPoint pos;
			
			jwkGetWindowPos(rb->frame, &pos);
			pos.px += event->motion.px - rb->pos.px;
			pos.py += event->motion.py - rb->pos.py;
			jtkSetWidgetPos(widget, pos.px, pos.py);
		}
		break;
	case JWK_EVENT_EXPOSE:
		{
			JtkSize window_size;
			JtkSize text_size;
			
			if(rb->title_text == NULL)
				break;
			
			jwkGetWindowSize(rb->title_bar, &window_size);
			jtkGetTextSize(rb->title_gc, rb->title_text,
							j_strlen(rb->title_text), &text_size);
			jtkDrawText(rb->title_gc, 2, (window_size.height - text_size.height)/2,
							rb->title_text, j_strlen(rb->title_text));
		}
		break;
	default:
		break;
	}
}

static JwkWindow* get_rb_container(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;
	
	return rb->container;
}

static JtkGC* create_rb_gc(JtkWidget *widget, JtkGCMode mode, JtkGCType type)
{
	RemovableBox *rb = widget->widget;
	
	return jwkCreateGC(rb->container, mode, type);
}

static void destroy_rb(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;

	jtkDestroyGC(rb->title_gc);
	jwkDestroyWindow(rb->frame);
	j_free(rb);
}

static void map_rb(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;

	jwkRaiseWindow(rb->frame);
	jwkMapWindow(rb->frame);
}

static void unmap_rb(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;

	rb->moving = JTK_FALSE;
	jwkUnmapWindow(rb->frame);
}

static void clear_rb(JtkWidget *widget,
	int px, int py, int width, int height, JtkBool exposures)
{
	RemovableBox *rb = widget->widget;
	
	jwkClearWindow(rb->container,
		px, py, width, height, exposures);
}

static void raise_rb(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;
	
	jwkRaiseWindow(rb->frame);
}

static void lower_rb(JtkWidget *widget)
{
	RemovableBox *rb = widget->widget;
	
	jwkLowerWindow(rb->frame);
}

static void set_rb_pos(JtkWidget *widget, int px, int py)
{
	RemovableBox *rb = widget->widget;

	jwkSetWindowPos(rb->frame, px, py);
}

static void set_rb_size(JtkWidget *widget, int width, int height)
{
	RemovableBox *rb = widget->widget;

	if(width < 32)
		width = 32;
	if(height < 32)
		height = 32;
	jwkSetWindowSize(rb->frame, width, height);
}

static void get_rb_pos(JtkWidget *widget, JtkPoint *pos)
{
	RemovableBox *rb = widget->widget;
	
	jwkGetWindowPos(rb->frame, pos);
}

static void get_rb_size(JtkWidget *widget, JtkSize *size)
{
	RemovableBox *rb = widget->widget;
	
	jwkGetWindowSize(rb->container, size);
}

static void set_rb_color(JtkWidget *widget, JtkColor color)
{
	RemovableBox *rb = widget->widget;

	jwkSetWindowColor(rb->container, color);
}

static void set_rb_border_size(JtkWidget *widget, int size)
{
	RemovableBox *rb = widget->widget;
	
	jwkSetWindowBorderSize(rb->frame, size);
}

static void set_rb_border_color(JtkWidget *widget, JtkColor color)
{
	RemovableBox *rb = widget->widget;
	
	jwkSetWindowBorderColor(rb->frame, color);
}

static void set_rb_text_color(JtkWidget *widget, JtkColor color)
{
	RemovableBox *rb = widget->widget;
	
	jtkSetColor(rb->title_gc, color);
}

static void set_rb_text(JtkWidget *widget, char *text)
{
	RemovableBox *rb = widget->widget;

	if(rb->title_text != NULL){
		j_free(rb->title_text);
		rb->title_text = NULL;
	}
	
	if(text == NULL)
		return;
	
	rb->title_text = j_malloc(j_strlen(text) + 1);
	if(rb->title_text == NULL)
		return;
	j_zero(rb->title_text, j_strlen(text) + 1);
	j_strcpy(rb->title_text, text);
	
	jwkClearWindow(rb->title_bar, 0, 0, 0, 0, JTK_TRUE);
}

static JwkWidgetFunc rb_func =
{
	.get_container = get_rb_container,
	.create_gc = create_rb_gc,
	.destroy = destroy_rb,
	.map = map_rb,
	.unmap = unmap_rb,
	.clear = clear_rb,
	.raise = raise_rb,
	.lower = lower_rb,
	.set_pos = set_rb_pos,
	.set_size = set_rb_size,
	.get_pos = get_rb_pos,
	.get_size = get_rb_size,
	.set_color = set_rb_color,
	.set_border_size = set_rb_border_size,
	.set_border_color = set_rb_border_color,
	.set_text_color = set_rb_text_color,
	.set_text = set_rb_text,
};

JtkWidget* jtkCreateRemovableBox(JtkWidget *parent)
{
	JtkWidget *widget;
	RemovableBox *rb;
	JtkSize size;
	char *test_text = "abcあいう**＄＠";
	
	if(parent == NULL)
		return NULL;
	
	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	rb = j_malloc(sizeof(RemovableBox));
	if(rb == NULL){
		j_free(widget);
		return NULL;
	}
	
	j_zero(widget, sizeof(JtkWidget));
	j_zero(rb, sizeof(RemovableBox));
	
	rb->frame = jwkCreateWindow(jwkGetFrame(parent));
	jwkSetWindowSize(rb->frame, 32, 32);
	jwkSetWindowColor(rb->frame, jtkRGB(255,255,255));
	jwkSetWindowBorderSize(rb->frame, 1);
	jwkSetWindowBorderColor(rb->frame, jtkRGB(64,64,64));
	jwkSetWindowEventMask(rb->frame,
		JWK_EVENTMASK_DESTROY |
		JWK_EVENTMASK_MOVE | JWK_EVENTMASK_RESIZE);
	jwkSetWidget(rb->frame, callback_rb_frame, widget);
	
	rb->title_bar = jwkCreateWindow(rb->frame);
	jwkSetWindowPos(rb->title_bar, 0, 0);
	jwkSetWindowColor(rb->title_bar, jtkRGB(0,0,128));
	jwkSetWindowEventMask(rb->title_bar,
		JWK_EVENTMASK_BUTTONDOWN | JWK_EVENTMASK_BUTTONUP |
		JWK_EVENTMASK_MOTION |
		JWK_EVENTMASK_EXPOSE);
	jwkSetWidget(rb->title_bar, callback_rb_titlebar, widget);
	rb->title_gc = jwkCreateGC(rb->title_bar, JTK_GCMODE_SCREEN, JTK_GCTYPE_NORMAL);
	jtkSetColor(rb->title_gc, jtkRGB(255,255,255));
	jtkGetTextSize(rb->title_gc, test_text, j_strlen(test_text), &size);
	rb->title_bar_height = size.height + 2;
	jwkSetWindowSize(rb->title_bar, 32, rb->title_bar_height);
	jwkMapWindow(rb->title_bar);
	
	rb->container = jwkCreateWindow(rb->frame);
	jwkSetWindowPos(rb->container, 0, rb->title_bar_height);
	jwkSetWindowSize(rb->container, 32, 32 - rb->title_bar_height);
	jwkSetWindowColor(rb->container, jtkRGB(255,255,255));
	jwkSetWindowEventMask(rb->container,
		JWK_EVENTMASK_KEYDOWN | JWK_EVENTMASK_KEYUP |
		JWK_EVENTMASK_BUTTONDOWN | JWK_EVENTMASK_BUTTONUP |
		JWK_EVENTMASK_MOTION | JWK_EVENTMASK_ENTER | JWK_EVENTMASK_LEAVE |
		JWK_EVENTMASK_FOCUSIN | JWK_EVENTMASK_FOCUSOUT |
		JWK_EVENTMASK_MAP | JWK_EVENTMASK_UNMAP |
		JWK_EVENTMASK_RESIZE |
		JWK_EVENTMASK_EXPOSE);
	jwkSetWidget(rb->container, callback_rb_container, widget);
	jwkMapWindow(rb->container);
	
	widget->widget = rb;
	widget->func = &rb_func;
	
	return widget;
}
