#include "x11-core.h"

static char *def_font_set = "-*-*-medium-r-normal--14-*";

static JtkError create_pixmap(JwkWindow *jw, JtkGC *gc, int width, int height)
{
	gc->buffer = XCreatePixmap(display, jw->window_id,
		width, height, DefaultDepth(display, 0));
	gc->buffer_width = width;
	gc->buffer_height = height;

	return JTK_NOERROR;
}

static void destroy_pixmap(JtkGC *gc)
{
	XFreePixmap(display, gc->buffer);
}

static JtkError change_size_pixmap(JtkGC *gc, int width, int height)
{
	destroy_pixmap(gc);
	if(create_pixmap(gc->jw, gc, width, height) != JTK_NOERROR)
		return JTK_ERROR;
	
	return JTK_NOERROR;
}

static JtkError create_shm_pixmap(JwkWindow *jw, JtkGC *gc, int width, int height)
{
	XImage *img;
	int format;
	int depth;
	
	format = XShmPixmapFormat(display);
	depth = DefaultDepth(display, 0);

	img = XShmCreateImage(display, DefaultVisual(display, 0), depth, format, NULL,
		&gc->shminfo, width, height);

	gc->shminfo.shmid = shmget(IPC_PRIVATE, img->bytes_per_line * img->height, IPC_CREAT | 0777);
	gc->shminfo.readOnly = False;
	gc->shminfo.shmaddr = gc->data_addr = shmat(gc->shminfo.shmid, 0, 0);
	gc->bytes_per_line = img->bytes_per_line;
	XDestroyImage(img);

	XShmAttach(display, &gc->shminfo);
	XSync(display, False);
	
	gc->buffer = XShmCreatePixmap(display, jw->window_id,
						gc->data_addr,
						&gc->shminfo, width, height, depth);
	gc->buffer_width = width;
	gc->buffer_height = height;
	
	if(depth == 16)
		gc->bytes_per_pixel = 2;
	else if((depth == 24) || (depth == 32))
		gc->bytes_per_pixel = 4;
	else
		gc->bytes_per_pixel = 0;
	
	return JTK_NOERROR;
}

static void destroy_shm_pixmap(JtkGC *gc)
{
	XShmDetach(display, &gc->shminfo);
	XFreePixmap(display, gc->buffer);
	shmdt(gc->shminfo.shmaddr);
	shmctl(gc->shminfo.shmid, IPC_RMID, 0);
}

static JtkError change_size_shm_pixmap(JtkGC *gc, int width, int height)
{
	destroy_shm_pixmap(gc);
	if(create_shm_pixmap(gc->jw, gc, width, height) != JTK_NOERROR)
		return JTK_ERROR;
	
	return JTK_NOERROR;
}

static void destroy_gl(JtkGC *gc)
{
	glXMakeCurrent(display, 0, NULL);
	glXDestroyContext(display, gc->glx_context);
}

static JtkError create_gc_normal_screen(JwkWindow *jw, JtkGC *gc)
{
	gc->screen = jw->window_id;
	gc->drawable = &gc->screen;
	
	return JTK_NOERROR;
}

static JtkError create_gc_normal_buffer(JwkWindow *jw, JtkGC *gc)
{
	JtkSize size;
	
	jwkGetWindowSize(jw, &size);

	if(create_pixmap(jw, gc, size.width, size.height) != JTK_NOERROR)
		return JTK_ERROR;
	gc->drawable = &gc->buffer;
	gc->destroy_gc = destroy_pixmap;
	
	return JTK_NOERROR;
}

static JtkError create_gc_normal_double(JwkWindow *jw, JtkGC *gc)
{
	JtkSize size;
	
	jwkGetWindowSize(jw, &size);

	gc->screen = jw->window_id;
	if(create_pixmap(jw, gc, size.width, size.height) != JTK_NOERROR)
		return JTK_ERROR;
	gc->drawable = &gc->buffer;
	gc->destroy_gc = destroy_pixmap;
	
	return JTK_NOERROR;
}

static JtkError create_gc_surface_screen(JwkWindow *jw, JtkGC *gc)
{
	return JTK_ERROR;
}

static JtkError create_gc_surface_buffer(JwkWindow *jw, JtkGC *gc)
{
	JtkSize size;
	
	jwkGetWindowSize(jw, &size);

	if(create_shm_pixmap(jw, gc, size.width, size.height) != JTK_NOERROR)
		return JTK_ERROR;
	gc->drawable = &gc->buffer;
	gc->destroy_gc = destroy_shm_pixmap;
	
	return JTK_NOERROR;
}

static JtkError create_gc_surface_double(JwkWindow *jw, JtkGC *gc)
{
	JtkSize size;
	
	jwkGetWindowSize(jw, &size);

	gc->screen = jw->window_id;
	if(create_shm_pixmap(jw, gc, size.width, size.height) != JTK_NOERROR)
		return JTK_ERROR;
	gc->drawable = &gc->buffer;
	gc->destroy_gc = destroy_pixmap;
	
	return JTK_NOERROR;
}

static JtkError create_gc_gl_screen(JwkWindow *jw, JtkGC *gc)
{
	return JTK_ERROR;
}

static JtkError create_gc_gl_buffer(JwkWindow *jw, JtkGC *gc)
{
	return JTK_ERROR;
}

static JtkError create_gc_gl_double(JwkWindow *jw, JtkGC *gc)
{
	int gl_attr[] = {
		GLX_RGBA,
		GLX_DOUBLEBUFFER,
		GLX_RED_SIZE, 1,
		GLX_GREEN_SIZE, 1,
		GLX_BLUE_SIZE, 1,
		None
	};
	XVisualInfo *vi;
		
	vi = glXChooseVisual(display, DefaultScreen(display), gl_attr);
	gc->glx_context = glXCreateContext(display, vi, 0, GL_TRUE);
	XFree(vi);
	
	gc->screen = jw->window_id;
	gc->drawable = &gc->screen;
	gc->destroy_gc = destroy_gl;
	
	return JTK_NOERROR;
}


/* graphics */
JtkGC* jwkCreateGC(JwkWindow *jw, JtkGCMode mode, JtkGCType type)
{
	JtkGC *gc = NULL;
	char **miss, *def;
	int n_miss;
	
	if(jw == NULL)
		return NULL;
	
	gc = j_malloc(sizeof(JtkGC));
	if(gc == NULL)
		return NULL;
	j_zero(gc, sizeof(JtkGC));

	if(type == JTK_GCTYPE_NORMAL){
		if(mode == JTK_GCMODE_SCREEN){
			if(create_gc_normal_screen(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_BUFFER){
			if(create_gc_normal_buffer(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_DOUBLE){
			if(create_gc_normal_double(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else{
			j_free(gc);
			return NULL;
		}
	}else if(type == JTK_GCTYPE_SURFACE){
		if(mode == JTK_GCMODE_SCREEN){
			if(create_gc_surface_screen(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_BUFFER){
			if(create_gc_surface_buffer(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_DOUBLE){
			if(create_gc_surface_double(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else{
			j_free(gc);
			return NULL;
		}
	}else if(type == JTK_GCTYPE_GL){
		if(mode == JTK_GCMODE_SCREEN){
			if(create_gc_gl_screen(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_BUFFER){
			if(create_gc_gl_buffer(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else if(mode == JTK_GCMODE_DOUBLE){
			if(create_gc_gl_double(jw, gc) != JTK_NOERROR){
				j_free(gc);
				return NULL;
			}
		}else{
			j_free(gc);
			return NULL;
		}
	}else{
		j_free(gc);
		return NULL;
	}
	
	gc->mode = mode;
	gc->type = type;
	gc->jw = jw;
	gc->gc = XCreateGC(display, *gc->drawable, 0, 0);
	gc->fs = XCreateFontSet(display, def_font_set, &miss, &n_miss, &def);
	gc->line_size = 1;
	gc->line_style = LineSolid;
	gc->cap_style = CapRound;
	gc->join_style = JoinMiter;
	
	return gc;
}

void jtkDestroyGC(JtkGC *gc)
{
	if(gc->destroy_gc != NULL)
		gc->destroy_gc(gc);
	XFreeFontSet(display, gc->fs);
	XFreeGC(display, gc->gc);
	j_free(gc);
}

void jtkBitBlt(JtkGC *src, JtkGC *dest, int src_x, int src_y,
					int width, int height, int dest_x, int dest_y)
{
	XCopyArea(display, *src->drawable, *dest->drawable, src->gc,
				src_x, src_y, width, height,
				dest_x, dest_y);
	XFlush(display);
}

void jtkSetColor(JtkGC *gc, JtkColor color)
{
	XSetForeground(display, gc->gc, x11_make_rgb(color));
	XFlush(display);
}

void jtkSetBackColor(JtkGC *gc, JtkColor color)
{
	XSetBackground(display, gc->gc, x11_make_rgb(color));
	XFlush(display);
}

void jtkSetLineSize(JtkGC *gc, unsigned int line_size)
{
	gc->line_size = line_size;
	
	XSetLineAttributes(display, gc->gc,
		gc->line_size,
		gc->line_style,
		gc->cap_style,
		gc->join_style);
}

void jtkSetLineStyle(JtkGC *gc, JtkLineStyle line_style)
{
	switch(line_style){
	case JTK_LINESTYLE_SOLID:
		gc->line_style = LineSolid;
		break;
	case JTK_LINESTYLE_ONOFFDASH:
		gc->line_style = LineOnOffDash;
		break;
	case JTK_LINESTYLE_DOUBLEDASH:
		gc->line_style = LineDoubleDash;
		break;
	}
	
	XSetLineAttributes(display, gc->gc,
		gc->line_size,
		gc->line_style,
		gc->cap_style,
		gc->join_style);
}

void jtkSetLineCapStyle(JtkGC *gc, JtkCapStyle cap_style)
{
	switch(cap_style){
	case JTK_CAPSTYLE_BUTT:
		gc->cap_style = CapButt;
		break;
	case JTK_CAPSTYLE_ROUND:
		gc->cap_style = CapRound;
		break;
	case JTK_CAPSTYLE_PROJECTING:
		gc->cap_style = CapProjecting;
		break;
	}
	
	XSetLineAttributes(display, gc->gc,
		gc->line_size,
		gc->line_style,
		gc->cap_style,
		gc->join_style);
}

void jtkSetLineJoinStyle(JtkGC *gc, JtkJoinStyle join_style)
{
	switch(join_style){
	case JTK_JOINSTYLE_MITER:
		gc->join_style = JoinMiter;
		break;
	case JTK_JOINSTYLE_ROUND:
		gc->join_style = JoinRound;
		break;
	case JTK_JOINSTYLE_BEVEL:
		gc->join_style = JoinBevel;
		break;
	}
	
	XSetLineAttributes(display, gc->gc,
		gc->line_size,
		gc->line_style,
		gc->cap_style,
		gc->join_style);
}

void jtkSetDashes(JtkGC *gc, int offset, char dash_list[], int n)
{
	XSetDashes(display, gc->gc, offset, dash_list, n);
}

void jtkDrawLine(JtkGC *gc, int x1, int y1, int x2, int y2)
{
	XDrawLine(display, *gc->drawable, gc->gc, x1, y1, x2, y2);
	XFlush(display);
}

void jtkDrawPoint(JtkGC *gc, int x, int y)
{
	XDrawPoint(display, *gc->drawable, gc->gc, x, y);
	XFlush(display);
}

void jtkDrawRect(JtkGC *gc, int x, int y, int width, int height)
{
	XDrawRectangle(display, *gc->drawable, gc->gc, x, y, width, height);
	XFlush(display);
}

void jtkFillRect(JtkGC *gc, int x, int y, int width, int height)
{
	XFillRectangle(display, *gc->drawable, gc->gc, x, y, width, height);
	XFlush(display);
}

void jtkDrawArc(JtkGC *gc, int x, int y, int width, int height,
					int angle1, int angle2)
{
	XDrawArc(display, *gc->drawable, gc->gc, x, y, width, height,
				angle1, angle2);
	XFlush(display);
}

void jtkFillArc(JtkGC *gc, int x, int y, int width, int height,
					int angle1, int angle2)
{
	XFillArc(display, *gc->drawable, gc->gc, x, y, width, height,
				angle1, angle2);
	XFlush(display);
}

void jtkSetArcMode(JtkGC *gc, JtkArcMode mode)
{
	if(mode == JTK_ARCMODE_PIESLICE)
		XSetArcMode(display, gc->gc, ArcPieSlice);
	if(mode == JTK_ARCMODE_CHORD)
		XSetArcMode(display, gc->gc, ArcChord);
}

void jtkDrawText(JtkGC *gc, int px, int py, char *text, int len)
{
	XRectangle ink, logical;
	
	XmbTextExtents(gc->fs, text, len, &ink, &logical);
	XmbDrawString(display, *gc->drawable, gc->fs, gc->gc,
					px - logical.x, py - logical.y, text, len);
	XFlush(display);
}

void jtkDrawTextWC(JtkGC *gc, int px, int py, JtkWChar *text, int len)
{
	XRectangle ink, logical;

	XwcTextExtents(gc->fs, (wchar_t*)text, len, &ink, &logical);
	XwcDrawString(display, *gc->drawable, gc->fs, gc->gc,
					px - logical.x, py - logical.y, (wchar_t*)text, len);
	XFlush(display);
}

void jtkGetTextSize(JtkGC *gc, char *text, int len, JtkSize *size)
{
	XRectangle ink, logical;
	
	XmbTextExtents(gc->fs, text, len, &ink, &logical);
	size->width = logical.width;
	size->height = logical.height;
}

void jtkGetTextSizeWC(JtkGC *gc, JtkWChar *text, int len, JtkSize *size)
{
	XRectangle ink, logical;
	
	XwcTextExtents(gc->fs, (wchar_t*)text, len, &ink, &logical);
	size->width = logical.width;
	size->height = logical.height;
}

JtkError jtkLockSurface(JtkGC *gc)
{
	if(gc->type != JTK_GCTYPE_SURFACE)
		return JTK_ERROR;
	
	XSync(display, False);
	return JTK_NOERROR;
}

void jtkUnlockSurface(JtkGC *gc)
{
	if(gc->type != JTK_GCTYPE_SURFACE)
		return;

	XSync(display, False);
}

void* jtkGetSurfaceAddr(JtkGC *gc)
{
	if(gc->type == JTK_GCTYPE_SURFACE){
		if((gc->mode == JTK_GCMODE_BUFFER) ||
			(gc->mode == JTK_GCMODE_DOUBLE))
			return gc->data_addr;
	}
	
	return NULL;
}

int jtkGetSurfaceBytesPerLine(JtkGC *gc)
{
	if(gc->type == JTK_GCTYPE_SURFACE){
		if((gc->mode == JTK_GCMODE_BUFFER) ||
			(gc->mode == JTK_GCMODE_DOUBLE))
			return gc->bytes_per_line;
	}
	
	return 0;
}

int jtkGetSurfaceBytesPerPixel(JtkGC *gc)
{
	if(gc->type == JTK_GCTYPE_SURFACE){
		if((gc->mode == JTK_GCMODE_BUFFER) ||
			(gc->mode == JTK_GCMODE_DOUBLE))
			return gc->bytes_per_pixel;
	}
	
	return 0;
}

void jtkReadSurfacePixel(JtkGC *gc, JtkPixel *pixel, int px, int py)
{
	if(gc->type != JTK_GCTYPE_SURFACE)
		return;
		
	if((gc->mode != JTK_GCMODE_BUFFER) && (gc->mode != JTK_GCMODE_DOUBLE))
		return;
	
	if((px < 0) || (py < 0))
		return;
	
	if((px >= gc->buffer_width) || (py >= gc->buffer_height))
		return;
	
	if(gc->bytes_per_pixel == 2){
		unsigned short d;
		unsigned short data;
		unsigned short *p;
		
		p = gc->data_addr;
		data = *(p + (gc->bytes_per_line/2) * py + px);

		d = data & 0xf800;
		d = d >> 8;
		d += d >> 5;
		pixel->r = d;
		
		d = data & 0x07c0;
		d = d >> 3;
		d += d >> 5;
		pixel->g = d;
				
		d = data & 0x001f;
		d = d << 3;
		d += d >> 5;
		pixel->b = d;
		
		pixel->a = 0;
	}
	else if(gc->bytes_per_pixel == 4){
		unsigned long d;
		unsigned long data;
		unsigned long *p;
		
		p = gc->data_addr;
		data = *(p + (gc->bytes_per_line/4) * py + px);
		
		d = data & 0x00ff0000;
		pixel->r = d >> 16;
		d = data & 0x0000ff00;
		pixel->g = d >> 8;
		pixel->b = data & 0x000000ff;
		pixel->a = 0;
	}
}

void jtkWriteSurfacePixel(JtkGC *gc, JtkPixel *pixel, int px, int py)
{
	if(gc->type != JTK_GCTYPE_SURFACE)
		return;
		
	if((gc->mode != JTK_GCMODE_BUFFER) && (gc->mode != JTK_GCMODE_DOUBLE))
		return;
	
	if((px < 0) || (py < 0))
		return;
	
	if((px >= gc->buffer_width) || (py >= gc->buffer_height))
		return;
	
	if(gc->bytes_per_pixel == 2){
		unsigned short d;
		unsigned short data;
		unsigned short *p;
		
		data = 0;
		d = pixel->r & 0xf8;
		d = d << 8;
		data += d;
		d = pixel->g & 0xf8;
		d = d << 3;
		data += d;
		d = pixel->b & 0xf8;
		d = d >> 3;
		data += d;

		p = gc->data_addr;
		*(p + (gc->bytes_per_line/2) * py + px) = data;
	}
	else if(gc->bytes_per_pixel == 4){
		unsigned long data;
		unsigned long *p;
		
		data = pixel->r;
		data = data << 8;
		data += pixel->g;
		data = data << 8;
		data += pixel->b;
		
		p = gc->data_addr;
		*(p + (gc->bytes_per_line/4) * py + px)  = data;
	}
}

void jtkReadSurfaceImage(JtkGC *gc, JtkImage *image,
	int src_x, int src_y, int width, int height,
	int dest_x, int dest_y)
{
	JtkPixel pixel;
	int x, y;

	if(gc->type != JTK_GCTYPE_SURFACE)
		return;
		
	if((gc->mode != JTK_GCMODE_BUFFER) && (gc->mode != JTK_GCMODE_DOUBLE))
		return;
	
	for(x=0; x<width; x++){
		for(y=0; y<height; y++){
			jtkReadSurfacePixel(gc, &pixel, src_x + x, src_y + y);
			jtkPutImagePixel(image, &pixel, dest_x + x, src_y + y);
		}
	}
}

void jtkWriteSurfaceImage(JtkGC *gc, JtkImage *image,
	int src_x, int src_y, int width, int height,
	int dest_x, int dest_y)
{
	JtkPixel pixel;
	int x, y;

	if(gc->type != JTK_GCTYPE_SURFACE)
		return;
		
	if((gc->mode != JTK_GCMODE_BUFFER) && (gc->mode != JTK_GCMODE_DOUBLE))
		return;
	
	for(x=0; x<width; x++){
		for(y=0; y<height; y++){
			jtkGetImagePixel(image, &pixel, src_x + x, src_y + y);
			jtkWriteSurfacePixel(gc, &pixel, dest_x + x, dest_y + y);
		}
	}
}

void jtkCurrent(JtkGC *gc)
{
	if(gc == NULL){
		glXMakeCurrent(display, 0, NULL);
	}else{
		if(gc->type == JTK_GCTYPE_GL){
			if(gc->mode == JTK_GCMODE_DOUBLE)
				glXMakeCurrent(display, gc->screen, gc->glx_context);
		}
	}
}

void jtkFlip(JtkGC *gc)
{
	if(gc == NULL)
		return;
	
	if(gc->mode != JTK_GCMODE_DOUBLE)
		return;
	
	if(gc->type == JTK_GCTYPE_NORMAL){
		XCopyArea(display, gc->buffer, gc->screen, gc->gc, 0, 0,
			gc->buffer_width, gc->buffer_height, 0, 0);
	}
	if(gc->type == JTK_GCTYPE_SURFACE){
		XCopyArea(display, gc->buffer, gc->screen, gc->gc, 0, 0,
			gc->buffer_width, gc->buffer_height, 0, 0);
	}
	if(gc->type == JTK_GCTYPE_GL){
		glXSwapBuffers(display, gc->screen);
	}
	
	XFlush(display);
}

void jtkResizeGC(JtkGC *gc, int width, int height)
{
	if(gc->type == JTK_GCTYPE_NORMAL){
		if((gc->mode == JTK_GCMODE_BUFFER) || (gc->mode == JTK_GCMODE_DOUBLE)){
			change_size_pixmap(gc, width, height);
		}
	}
	if(gc->type == JTK_GCTYPE_SURFACE){
		if((gc->mode == JTK_GCMODE_BUFFER) || (gc->mode == JTK_GCMODE_DOUBLE)){
			change_size_shm_pixmap(gc, width, height);
		}
	}
}

int jtkGetGCWidth(JtkGC *gc)
{
	JtkSize size;

	if(gc->type == JTK_GCTYPE_NORMAL){
		if(gc->mode == JTK_GCMODE_SCREEN){
			jwkGetWindowSize(gc->jw, &size);
			return size.width;
		}
		if(gc->mode == JTK_GCMODE_BUFFER){
			return gc->buffer_width;
		}
		if(gc->mode == JTK_GCMODE_DOUBLE){
			return gc->buffer_width;
		}
	}
	if(gc->type == JTK_GCTYPE_SURFACE){
		if(gc->mode == JTK_GCMODE_BUFFER){
			return gc->buffer_width;
		}
		if(gc->mode == JTK_GCMODE_DOUBLE){
			return gc->buffer_width;
		}
	}
	if(gc->type == JTK_GCTYPE_GL){
		if(gc->mode == JTK_GCMODE_DOUBLE){
			jwkGetWindowSize(gc->jw, &size);
			return size.width;
		}
	}
	
	return 0;
}

int jtkGetGCHeight(JtkGC *gc)
{
	JtkSize size;

	if(gc->type == JTK_GCTYPE_NORMAL){
		if(gc->mode == JTK_GCMODE_SCREEN){
			jwkGetWindowSize(gc->jw, &size);
			return size.height;
		}
		if(gc->mode == JTK_GCMODE_BUFFER){
			return gc->buffer_height;
		}
		if(gc->mode == JTK_GCMODE_DOUBLE){
			return gc->buffer_height;
		}
	}
	if(gc->type == JTK_GCTYPE_SURFACE){
		if(gc->mode == JTK_GCMODE_BUFFER){
			return gc->buffer_height;
		}
		if(gc->mode == JTK_GCMODE_DOUBLE){
			return gc->buffer_height;
		}
	}
	if(gc->type == JTK_GCTYPE_GL){
		if(gc->mode == JTK_GCMODE_DOUBLE){
			jwkGetWindowSize(gc->jw, &size);
			return size.height;
		}
	}
	
	return 0;
}

