#include "jtk.h"

#define LINE_TEXT_SIZE 256

typedef struct LogTextBuffer LogTextBuffer;
struct LogTextBuffer{
	char text[LINE_TEXT_SIZE];
	JtkColor color;
}; 

typedef struct LogBox LogBox;
struct LogBox
{
	JtkWindow *jw;
	JtkGC *gc;

	int buffer_num;
	LogTextBuffer *buffer;
	JtkColor text_color;
};

static void DestroyLogBoxBuffer(LogBox *lb)
{
	if(lb->buffer == NULL)
		return;

	j_free(lb->buffer);
	lb->buffer = NULL;
	lb->buffer_num = 0;
}

static void CreateLogBoxBuffer(LogBox *lb, int line_num)
{
	if(lb->buffer != NULL)
		DestroyLogBoxBuffer(lb);
	
	if(line_num < 1)
		line_num = 1;
	
	lb->buffer = j_malloc(sizeof(LogTextBuffer) * line_num);
	if(lb->buffer == NULL)
		return;
	j_zero(lb->buffer, sizeof(LogTextBuffer) * line_num);
	lb->buffer_num = line_num;
}

static void logbox_callback(JtkWindowEvent *event)
{
	JtkWidget *widget = event->any.widget;
	LogBox *lb = widget->widget;

	switch(event->type){
	case JTK_WINDOW_EVENT_EXPOSE:
		{
			JtkSize window_size;
			JtkSize text_size;
			char *test_text = "abcあいう12３４*!＃＄";
			int i;
			int py;
			
			jtkGetWindowSize(lb->jw, &window_size);
			jtkGetTextSize(lb->gc, test_text, j_strlen(test_text), &text_size);
			for(i=0; i<lb->buffer_num; i++){
				py = window_size.height - (i + 1)*(text_size.height + 2);
				if(py < -text_size.height)
					break;
				jtkSetColor(lb->gc, lb->buffer[i].color);
				jtkDrawText(lb->gc, 5, py,
								lb->buffer[i].text,
								j_strlen(lb->buffer[i].text));
			}
		}
		break;
	default:
		break;
	}
}

static void destroy_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jtkDestroyWindow(lb->jw);
	DestroyLogBoxBuffer(lb);
	j_free(lb);
}

static void map_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jtkMapWindow(lb->jw);
}

static void unmap_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jtkUnmapWindow(lb->jw);
}

static void raise_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	jtkRaiseWindow(lb->jw);
}

static void lower_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	jtkLowerWindow(lb->jw);
}

static void set_logbox_pos(JtkWidget *widget, int px, int py)
{
	LogBox *lb = widget->widget;

	jtkSetWindowPos(lb->jw, px, py);
}

static void set_logbox_size(JtkWidget *widget, int width, int height)
{
	LogBox *lb = widget->widget;

	jtkSetWindowSize(lb->jw, width, height);
}

static void set_logbox_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	jtkSetWindowColor(lb->jw, color);
}

static void set_logbox_border_size(JtkWidget *widget, int size)
{
	LogBox *lb = widget->widget;
	
	jtkSetWindowBorderSize(lb->jw, size);
}

static void set_logbox_border_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	jtkSetWindowBorderColor(lb->jw, color);
}

static void set_logbox_text_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	lb->text_color = color;
}

static void set_logbox_text_lines(JtkWidget *widget, int lines)
{
	LogBox *lb = widget->widget;
	
	CreateLogBoxBuffer(lb, lines);
}

static void set_logbox_text(JtkWidget *widget, char *text)
{
	int i;
	LogBox *lb = widget->widget;
	
	for(i=lb->buffer_num - 2; i>=0; i--){
		j_strcpy(lb->buffer[i+1].text, lb->buffer[i].text);
		lb->buffer[i+1].color = lb->buffer[i].color;
	}
	j_strcpy(lb->buffer[0].text, text);
	lb->buffer[0].color = lb->text_color;
	
	jtkClearWindow(lb->jw, 0, 0, 0, 0, JTK_TRUE);
}

static void clear_logbox_text(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	j_zero(lb->buffer, sizeof(LogTextBuffer) * lb->buffer_num);
	jtkClearWindow(lb->jw, 0, 0, 0, 0, JTK_TRUE);
}

static JtkWidgetFunc logbox_func =
{
	NULL,
	NULL,
	destroy_logbox,
	map_logbox,
	unmap_logbox,
	raise_logbox,
	lower_logbox,
	set_logbox_pos,
	set_logbox_size,
	set_logbox_color,
	set_logbox_border_size,
	set_logbox_border_color,
	set_logbox_text_color,
	set_logbox_text_lines,
	set_logbox_text,
	clear_logbox_text,
};

JtkWidget* jtkCreateLogbox(JtkWidget *parent)
{
	JtkWidget *widget;
	LogBox *lb;

	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	lb = j_malloc(sizeof(LogBox));
	if(lb == NULL){
		j_free(widget);
		return NULL;
	}

	j_zero(widget, sizeof(JtkWidget));
	j_zero(lb, sizeof(LogBox));
	
	widget->widget = lb;
	widget->func = &logbox_func;
	
	lb->jw = jtkCreateSimpleWindow(jtkGetContainer(parent));
	lb->gc = jtkCreateWindowGC(lb->jw);
	lb->text_color = jtkRGB(255,255,255);
	CreateLogBoxBuffer(lb, 10);

	jtkSetWindowColor(lb->jw, jtkRGB(64,64,64));
	jtkSetWindowBorderSize(lb->jw, 1);
	jtkSetWindowBorderColor(lb->jw, jtkRGB(128,128,128));
	jtkSetWidget(lb->jw, logbox_callback, widget);
	jtkSetWindowEventMask(lb->jw, JTK_WINDOW_EVENTMASK_EXPOSE);
	
	return widget;
}
