package org.junitdoc.ui.decisiontable;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jface.util.LocalSelectionTransfer;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.TreeNode;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.Widget;
import org.junitdoc.core.decisiontable.DecisionTableModel;
import org.junitdoc.core.decisiontable.TestMethod;
import org.junitdoc.core.decisiontable.Type;
import org.junitdoc.core.rewriter.ChangeStateHandler;
import org.junitdoc.core.rewriter.IncludeMethodHandler;
import org.junitdoc.core.rewriter.JDTUtils;
import org.junitdoc.core.rewriter.Rewriter;
import org.junitdoc.preferences.PreferenceConstants;
import org.junitdoc.preferences.PreferenceUtils;
import org.junitdoc.ui.Const;
import org.junitdoc.ui.UIMessages;
import org.junitdoc.ui.events.TreeMouseAdapter;
import org.junitdoc.ui.events.TreeMouseEvent;

public class DecisionTableViewer extends TreeViewer {

	private static final int START_INDEX_COLUMN = 2;

	private DecisionTableModel model;

	private boolean isCheckStateCell(int columnIndex) {
		return columnIndex >= START_INDEX_COLUMN;
	}

	private int columnIndexToModelIndex(int columnIndex) {
		return (columnIndex - START_INDEX_COLUMN);
	}

	private int modelIndexToColumnIndex(int modelIndex) {
		return (modelIndex + START_INDEX_COLUMN);
	}

	public DecisionTableViewer(final Composite parent, String decisionTableName) {
		super(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION);

		model = new DecisionTableModel(decisionTableName);

		final Tree tree = getTree();

		GridData gd = new GridData(GridData.FILL_BOTH);
		tree.setLayoutData(gd);
		setAutoExpandLevel(TreeViewer.ALL_LEVELS);

		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);

		setContentProvider(new DecisionTableContentProvider());

		tree.addMouseListener(new TreeMouseAdapter(tree) {

			@Override
			public void mouseUp(TreeMouseEvent event) {
				int columnIndex = event.getSelectedColumnIndex();

				if (columnIndex == 0) {
					Object data = event.getSelectedTreeItem().getData();
					TreeNode treeNode = (TreeNode) data;
					selectAndRevealAnnotationDeclaration(treeNode);

				} else if (isCheckStateCell(columnIndex)) {
					TreeColumn column = event.getSelectedTreeColumn();
					selectAndRevealTestMethod(column);

					// Color whilte = getTree().getDisplay().getSystemColor(
					// SWT.COLOR_WHITE);
					// getTree().setBackground(whilte);
					//
					// Color red = getTree().getDisplay().getSystemColor(
					// SWT.COLOR_GRAY);
					// event.getSelectedTreeItem().setBackground(columnIndex,
					// red);

				}
			}

			@Override
			public void mouseDoubleClick(TreeMouseEvent event) {
				int columnIndex = event.getSelectedColumnIndex();

				if (isCheckStateCell(columnIndex)) {
					Object data = event.getSelectedTreeItem().getData();
					TreeNode treeItem = (TreeNode) data;
					Type type = (Type) treeItem.getValue();

					changeStatus(type, columnIndex);
					selectAndRevealTestMethod(event.getSelectedTreeColumn());
				}
			}
		});

		addDnDSupport();
	}

	private void addDnDSupport() {
		Transfer[] types = new Transfer[] { LocalSelectionTransfer
				.getTransfer() };
		DropTarget target = new DropTarget(getTree(), DND.DROP_DEFAULT
				| DND.DROP_COPY);
		target.setTransfer(types);
		target.addDropListener(new DropTargetAdapter() {

			@Override
			public void dragEnter(DropTargetEvent event) {
				event.detail = DND.DROP_COPY;
			}

			public void dragOver(DropTargetEvent event) {
				// System.out.println("DnD dragOver!!!:" + event);
				//
				// event.feedback = DND.FEEDBACK_EXPAND | DND.FEEDBACK_SCROLL
				// | DND.FEEDBACK_SELECT;
			}

			public void drop(DropTargetEvent event) {
				if (event.data instanceof TreeSelection) {
					TreeSelection selection = (TreeSelection) event.data;
					Object element = selection.getFirstElement();

					if (element instanceof IMethod) {
						IMethod method = (IMethod) element;
						ICompilationUnit unit = method.getCompilationUnit();

						Rewriter.execute(unit, new IncludeMethodHandler(model
								.getName(), method));
					}
				}
			}
		});
	}

	public DecisionTableModel getModel() {
		return model;
	}

	public void changeStatus(Type type, int columnIndex) {
		TestMethod testMethod = model
				.getTestMethod(columnIndexToModelIndex(columnIndex));

		CompilationUnit ast = (CompilationUnit) type.getSource().getRoot();

		ICompilationUnit source = JDTUtils.getICompilationUnit(ast);

		Rewriter.execute(source, new ChangeStateHandler(type, testMethod));

		// We don't need refresh own tree. Because, the DecisionTablesTabPage
		// is listening "elementChanged event". see DecisionTablesTabPage.java
	}

	public void initialize(CompilationUnit ast) {
		getTree().setVisible(false);

		clearAllColumn();
		parseModel(ast);
		makeColumn(model);
		setInput(model);
		resizeColumnWidth();

		getTree().setVisible(true);
	}

	private void parseModel(CompilationUnit ast) {
		model.parse(ast);
	}

	private void resizeColumnWidth() {
		Tree tree = getTree();

		TreeColumn[] columns = tree.getColumns();
		for (TreeColumn treeColumn : columns) {
			treeColumn.pack();
		}
	}

	private void makeColumn(DecisionTableModel model) {
		// create First column
		TreeViewerColumn firstColumn = new TreeViewerColumn(this, SWT.LEFT);
		firstColumn.setLabelProvider(new HeaderCellLabelProvider());
		TreeColumn column = firstColumn.getColumn();
		column.setText(UIMessages.getString("DecisionTableViewer.header.text"));
		// viewerColumn.setEditingSupport(new MyEditingSupport(this));

		// create Second column
		TreeViewerColumn secondColumn = new TreeViewerColumn(this, SWT.LEFT);
		secondColumn.setLabelProvider(new CounterCellLabelProvider());
		secondColumn.getColumn().setText("");

		for (TestMethod testMethod : model.getTestMethodList()) {

			TreeViewerColumn viewerCol = new TreeViewerColumn(this, SWT.LEFT,
					modelIndexToColumnIndex(testMethod.getIndex()));

			viewerCol.setLabelProvider(new CheckStateCellLabelProvider());
			TreeColumn col = viewerCol.getColumn();
			col.setText(testMethod.getName());
			col.setMoveable(true);
			col.addSelectionListener(new TestMethodSelectionListener());

			col.setData(Const.TESTMETHOD_MAPPING_KEY, testMethod);
		}
	}

	private void selectAndRevealTestMethod(TreeColumn treeColumn) {
		TestMethod testMethod = (TestMethod) treeColumn
				.getData(Const.TESTMETHOD_MAPPING_KEY);
		testMethod.selectAndReveal();
	}

	private void selectAndRevealAnnotationDeclaration(TreeNode treeNode) {
		Type type = (Type) treeNode.getValue();
		type.selectAndReveal();
	}

	/**
	 * A listener class for select method in javaEditor when clicking column
	 * header of decision table.
	 * 
	 */
	private class TestMethodSelectionListener extends SelectionAdapter {

		public void widgetSelected(SelectionEvent event) {
			TreeColumn selectedColumn = (TreeColumn) event.getSource();
			selectAndRevealTestMethod(selectedColumn);
		}
	}

	private class HeaderCellLabelProvider extends CellLabelProvider {

		@Override
		public void update(ViewerCell cell) {
			TreeNode treeNode = (TreeNode) cell.getElement();

			Type type = (Type) treeNode.getValue();
			String name = type.getName();

			// If this type is "Conditions" or "Actions", set to cell it's
			// notation.

			if (name.equals(Const.CONDITION_TAG)) {
				cell
						.setText(PreferenceUtils
								.getString(PreferenceConstants.DECISION_TABLE_CONDITION_LABEL));

			} else if (name.equals(Const.ACTION_TAG)) {
				cell
						.setText(PreferenceUtils
								.getString(PreferenceConstants.DECISION_TABLE_ACTION_LABEL));

			} else {
				cell.setText(name);
			}
			// if (type instanceof Condition) {
			// cell.setBackground(parent.getDisplay().getSystemColor(
			// SWT.COLOR_WIDGET_HIGHLIGHT_SHADOW));
			//
			// } else if (type instanceof Action) {
			// cell.setBackground(parent.getDisplay().getSystemColor(
			// SWT.COLOR_WIDGET_LIGHT_SHADOW));
			// }

			Widget item = cell.getItem();

			if (item instanceof TableItem) {
				((TableItem) item).setData(type);
			}
		}
	}

	private class CounterCellLabelProvider extends CellLabelProvider {

		@Override
		public void update(ViewerCell cell) {
			TreeNode treeNode = (TreeNode) cell.getElement();
			Type type = (Type) treeNode.getValue();

			cell.setText(Integer.toString(type.getCheckedTotalCount()));
		}
	}

	private class CheckStateCellLabelProvider extends CellLabelProvider {

		@Override
		public void update(ViewerCell cell) {
			TreeNode treeNode = (TreeNode) cell.getElement();
			Type type = (Type) treeNode.getValue();

			TestMethod testMethod = model
					.getTestMethod(columnIndexToModelIndex(cell
							.getColumnIndex()));

			cell.setText(testMethod.getCheckState(type));
		}
	}

	private void clearAllColumn() {
		TreeColumn[] columns = getTree().getColumns();

		for (TreeColumn column : columns) {
			column.dispose();
		}
	}

}