require 'runit/testcase'

require 'kagemai/kagemai'
require 'kagemai/project'
require 'kagemai/message'
require 'kagemai/message_bundle'

module Kagemai
  remove_const(:VERSION)
  remove_const(:CODENAME)
  VERSION  = 'TEST_VERSION'
  CODENAME = 'TEST_CODENAME'
  Config[:smtp_server] = 'localhost'
end

class TestProject < RUNIT::TestCase
  def setup
    Kagemai::MessageBundle.open('resource', 'ja', 'messages')
    @project_dir = 'test/testfile/projects'
    @project = Kagemai::Project.open(@project_dir, 'p1')
    @project.db_manager.disable_cache

    @m1 = Kagemai::Message.new(@project.report_type)
    @m2 = Kagemai::Message.new(@project.report_type)

    @id_filename = @project_dir + '/p1/spool/id'
    File.open(@id_filename, 'w') do |file|
      file.puts '2'
    end
  end

  def teardown
    File.unlink(@id_filename)

    (2..6).each do |i|
      if FileTest.exist?(@project_dir + "/p1/spool/#{i}")
        Dir.glob(@project_dir + "/p1/spool/#{i}/*") do |f|
          File.unlink(f.untaint)
        end
        Dir.rmdir(@project_dir + "/p1/spool/#{i}")
      end
    end
  end

  def test_open
    assert_equals('p1', @project.id)
    assert_equals('test project 1', @project.name)
  end

  def test_load
    report = @project.load_report(1)
    assert_instance_of(Kagemai::Report, report)
  end

  def test_new_report
    size = @project.size
    report = @project.new_report(@m1)
    assert_instance_of(Kagemai::Report, report)
    assert_equals(size + 1, @project.size)
    assert_equals(1, report.size)
  end

  def test_add_message
    size = @project.size
    report = @project.new_report(@m1)
    assert_equals(size + 1, @project.size)
    assert_equals(1, report.size)

    report2 = @project.add_message(report.id, @m2)
    assert_equals(2, report2.size)
  end

  def test_collect_reports
    m1 = Kagemai::Message.new(@project.report_type)
    m2 = Kagemai::Message.new(@project.report_type)
    m1['status'] = ''
    m2['status'] = ''

    m3 = Kagemai::Message.new(@project.report_type)
    m4 = Kagemai::Message.new(@project.report_type)
    m3['status'] = 'Ѥ'
    m4['status'] = 'Ѥ'

    m5 = Kagemai::Message.new(@project.report_type)
    m5['status'] = 'λ'

    @project.new_report(m1)
    @project.new_report(m2)
    @project.new_report(m3)
    @project.new_report(m4)
    @project.new_report(m5)

    report_hash = @project.collect_reports('status')
    assert_equals(3, report_hash.size)
    assert(report_hash.has_key?(''))
    assert(report_hash.has_key?('Ѥ'))
    assert(report_hash.has_key?('λ'))
    assert_equals(3, report_hash[''].size)
    assert_equals(2, report_hash['Ѥ'].size)
    assert_equals(1, report_hash['λ'].size)
    assert_equals('', report_hash[''][0].attr('status'))
    assert_equals('Ѥ', report_hash['Ѥ'][0].attr('status'))
    assert_equals('λ', report_hash['λ'][0].attr('status'))
  end


  class DummyMailer
    def sendmail(mail, from_addr, *to_addrs)
      @mailsrc = mail.to_s
      @from_addr = from_addr
      @to_addrs = *to_addrs
    end
    attr_reader :mailsrc, :from_addr, :to_addrs
  end

  def test_sendmail()
    mailer = DummyMailer.new
    Kagemai::Mailer.set_mailer(mailer)
    
    @project.instance_eval{
      @admin_address = 'kagemai-admin@daifukuya.com'
      @post_address  = 'kagemai-bts@daifukuya.com'
      @notify_addresses = ['notify1@daifukuya.com']
    }
    
    @m1['email'] = 'fukuoka@daifukuya.com'
    @m1['title'] = 'sendmail test'
    @m1['status'] = ''
    @m1['body'] = 'hello world'
    @m1.time = Time.local(2002, 12, 27, 15, 15, 32)
    @m1.set_option('email_notification', true)
    
    @project.new_report(@m1)
    
    expect_mail = ''
    File.open("#{@project_dir}/p1/expect_mail.txt") {|file| expect_mail = file.read.gsub(/\r/, '')}
    expect_mail = Kconv.kconv(expect_mail, Kconv::JIS, Kconv::EUC).gsub(/\n/, "\r\n")
    
    expect_from_addr = @project.admin_address
    expect_to_addrs  = ['fukuoka@daifukuya.com', 'notify1@daifukuya.com']
    
    assert_equals(expect_mail, mailer.mailsrc)
    assert_equals(expect_from_addr, mailer.from_addr)
    assert_equals(expect_to_addrs, mailer.to_addrs)
  end

end

