# Licence: Ruby's
# Based on lib/ruby/1.8/profiler.rb

module YaProfiler
  Times = if defined? Process.times then Process else Time end
  
  class Method
    @@mid = 0
    def initialize(name, caller = nil)
      @name   = name
      @cost   = 0.0
      @caller = {}
      @callee = {}
      @count  = 0
      
      @id    = @@mid
      @@mid += 1
      
      @self_cost = nil
      
      invoke(caller)
    end
    attr_reader :id, :name, :cost
    
    def add(callee)
      callee.ret()
      @callee[callee.name] = callee
    end
    
    def callee(name)
      method = @callee[name]
      if method then
        method.invoke(self)
      end
      method
    end
    
    def print_profile(out, total, cumm, use_self_cost)
      if @cost == 0.0 then
        @cost = 0.001
      end
      cumm += @cost
      print_self(out, total, cumm, use_self_cost)
      print_caller(out)
      print_callee(out)      
    end
    
    def print_self(out, total, cumm, use_self_cost)
      cost = use_self_cost ? self_cost : @cost
      out.printf "[%04d] ", @id
      out.printf "%6.2f %8.2f  %8.2f %8d ", cost/total*100, cumm, self_cost, @count
      out.printf "%8.2f %8.2f  %s\n", self_cost*1000/@count, @cost*1000/@count, name
    end
    
    def print_caller(out)
    end
    
    def print_callee(out)
      if @callee.size > 0 then
        sorted = @callee.sort_by {|name, callee| callee.cost}.reverse
        sorted.each do |name, callee|
          out.printf "          --> %6.2f %7.2f", callee.cost/@cost*100, callee.cost
          out.printf " [%04d] %s\n", callee.id, callee.name
        end
      end
    end
    
    def eql?(other)
      @name == other.name
    end
    
    def hash()
      @name.hash
    end
    
    def stop()
      ret()
    end
    
    def all()
      all = [self]
      @callee.each do |name, callee|
        all.concat(callee.all)
      end
      all
    end
    
    def self_cost()
      unless @self_cost then
        @self_cost = @cost
        @callee.each {|name, callee| @self_cost -= callee.cost}
      end
      @self_cost
    end    
    
  protected
    def invoke(caller)
      @stime  = Float(Times::times[0])
      @count  += 1
      @caller[caller.name] = caller if caller
    end
    
    def ret()
      @cost += Float(Times::times[0]) - @stime
    end        
  end
  
  # internal values
  PROFILE_PROC = proc{|event, file, line, id, binding, klass|
    if klass.kind_of?(Class) then
      name = klass.to_s + '#'
    elsif klass
      name = klass.to_s + '.'
    else
      name = 'TOP_LEVEL'
    end
    name += id.id2name if id
    
    unless @@excludes.include?(klass) then
      case event
      when "call", "c-call"
        method = @@stack.last.callee(name)
        unless method then
          method = Method.new(name)
        end
        @@stack.push(method)
      when "return", "c-return"
        method = @@stack.pop
        if @@stack.size == 0 then
          @@stack.push(method)
        else
          @@stack.last.add(method)
        end
      end
    end
  }
  
  module_function
  def start_profile(excludes = Set.new)
    @@excludes = excludes
    @@root  = Method.new('TOP_LEVEL')
    @@stack = [@@root]
    @@start = Float(Times::times[0])
    set_trace_func PROFILE_PROC
  end
  
  def stop_profile
    @@root.stop
    set_trace_func nil
  end
  
  def print_profile(f, sort_by_self_cost = false)
    total = Float(Times::times[0]) - @@start
    if total == 0 then total = 0.001 end
    f.printf "         %%    cumulative  self              self     total\n"      
    f.printf " call   time   seconds   seconds    calls  ms/call  ms/call  name\n"
  
    sorted = nil
    if sort_by_self_cost then
      sorted = @@root.all().sort{|a, b|
        a.self_cost == b.self_cost ? a.id <=> b.id : b.self_cost <=> a.self_cost
      }
    else
      sorted = @@root.all().sort{|a, b|
        a.cost == b.cost ? a.id <=> b.id : b.cost <=> a.cost
      }
    end
    
    cumm = 0
    sorted.each do |method|
      method.print_profile(f, total, cumm, sort_by_self_cost)
      cumm += method.self_cost
    end
  end
end
