﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(KsgSubtitleElementConverter))]
    public class KsgSubtitle : ISubtitleElement, IHasFormatDictionary, IHasWordArrangementDictionary, IHasLineArrangementDictionary, IHasPageArrangementDictionary
    {
        public KsgSubtitle()
        {
            try
            {
                Parts = new List<KsgPart>();
                Resolution = new Size(864, 486);
                FrameRate = 30;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void Initialize(Control control)
        {
            try
            {
                _control = control;
                foreach (KsgPart part in Parts)
                {
                    part.Initialize(this);
                }
                RefreshFormatProperty(true, false);
                RefreshWordArrangementProperty(true, false);
                RefreshLineArrangementProperty(true, false);
                RefreshPageArrangementProperty(true, false);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [Category("映像")]
        [DisplayName("解像度")]
        public Size Resolution { get; set; }
        [Category("映像")]
        [DisplayName("フレームレート")]
        public float FrameRate { get; set; }

        [Browsable(false)]
        public List<CatalogItemWrapper<IFormat>> FormatCatalogWrapper
        {
            get
            {
                try
                {
                    List<CatalogItemWrapper<IFormat>> list = new List<CatalogItemWrapper<IFormat>>();
                    foreach (CatalogItem<IFormat> item in CatalogManager.FormatCatalog)
                    {
                        list.Add(new CatalogItemWrapper<IFormat>(item.Name, item.Plugin));
                    }
                    return list;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    CatalogManager.FormatCatalog.Clear();
                    foreach (CatalogItemWrapper<IFormat> item in value)
                    {
                        CatalogManager.FormatCatalog.Add(new CatalogItem<IFormat>(item.Name, item.Plugin));
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public List<CatalogItemWrapper<IWordArrangement>> WordArrangementCatalogWrapper
        {
            get
            {
                try
                {
                    List<CatalogItemWrapper<IWordArrangement>> list = new List<CatalogItemWrapper<IWordArrangement>>();
                    foreach (CatalogItem<IWordArrangement> item in CatalogManager.WordArrangementCatalog)
                    {
                        list.Add(new CatalogItemWrapper<IWordArrangement>(item.Name, item.Plugin));
                    }
                    return list;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    CatalogManager.FormatCatalog.Clear();
                    foreach (CatalogItemWrapper<IWordArrangement> item in value)
                    {
                        CatalogManager.WordArrangementCatalog.Add(new CatalogItem<IWordArrangement>(item.Name, item.Plugin));
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public List<CatalogItemWrapper<ILineArrangement>> LineArrangementCatalogWrapper
        {
            get
            {
                try
                {
                    List<CatalogItemWrapper<ILineArrangement>> list = new List<CatalogItemWrapper<ILineArrangement>>();
                    foreach (CatalogItem<ILineArrangement> item in CatalogManager.LineArrangementCatalog)
                    {
                        list.Add(new CatalogItemWrapper<ILineArrangement>(item.Name, item.Plugin));
                    }
                    return list;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    CatalogManager.FormatCatalog.Clear();
                    foreach (CatalogItemWrapper<ILineArrangement> item in value)
                    {
                        CatalogManager.LineArrangementCatalog.Add(new CatalogItem<ILineArrangement>(item.Name, item.Plugin));
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public List<CatalogItemWrapper<IPageArrangement>> PageArrangementCatalogWrapper
        {
            get
            {
                try
                {
                    List<CatalogItemWrapper<IPageArrangement>> list = new List<CatalogItemWrapper<IPageArrangement>>();
                    foreach (CatalogItem<IPageArrangement> item in CatalogManager.PageArrangementCatalog)
                    {
                        list.Add(new CatalogItemWrapper<IPageArrangement>(item.Name, item.Plugin));
                    }
                    return list;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    CatalogManager.PageArrangementCatalog.Clear();
                    foreach (CatalogItemWrapper<IPageArrangement> item in value)
                    {
                        CatalogManager.PageArrangementCatalog.Add(new CatalogItem<IPageArrangement>(item.Name, item.Plugin));
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public List<KsgPart> Parts { get; set; }
        private Control _control;

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgPage> Pages
        {
            get
            {
                try
                {
                    List<KsgPage> pages = new List<KsgPage>();
                    foreach (KsgPart part in Parts)
                    {
                        pages.AddRange(part.Pages);
                    }
                    return pages;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgLine> Lines
        {
            get
            {
                try
                {
                    List<KsgLine> lines = new List<KsgLine>();
                    foreach (KsgPart part in Parts)
                    {
                        lines.AddRange(part.Lines);
                    }
                    return lines;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgWord> Words
        {
            get
            {
                try
                {
                    List<KsgWord> words = new List<KsgWord>();
                    foreach (KsgPart part in Parts)
                    {
                        words.AddRange(part.Words);
                    }
                    return words;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> MainTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgPart part in Parts)
                    {
                        characters.AddRange(part.MainTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> RubyTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgPart part in Parts)
                    {
                        characters.AddRange(part.RubyTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> Characters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgPart part in Parts)
                    {
                        characters.AddRange(part.Characters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Graphics Graphics
        {
            get
            {
                try
                {
                    return _control.CreateGraphics();
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Color BackColor
        {
            get
            {
                try
                {
                    return _control.BackColor;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return Color.Empty;
                }
            }
        }

        [Category("ページ内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BasePageArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBasePageArrangementName(Pages);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBasePageArrangementName(Pages, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IPageArrangement BasePageArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBasePageArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("ページ内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(PageArrangementTypeListConverter))]
        [XmlIgnore]
        public Type PageArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetPageArrangementType(Pages);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> PageArrangementDictionary { get; set; }

        [Category("行内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseLineArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBaseLineArrangementName(Lines);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseLineArrangementName(Lines, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public ILineArrangement BaseLineArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseLineArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("行内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(LineArrangementTypeListConverter))]
        [XmlIgnore]
        public Type LineArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetLineArrangementType(Lines);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> LineArrangementDictionary { get; set; }

        [Category("単語内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseWordArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangementName(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseWordArrangementName(Words, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IWordArrangement BaseWordArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("単語内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(WordArrangementTypeListConverter))]
        [XmlIgnore]
        public Type WordArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetWordArrangementType(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> WordArrangementDictionary { get; set; }

        [Category("書式")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseFormatName
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormatName(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseFormatName(Characters, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat BaseFormat
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormat(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(FormatTypeListConverter))]
        [XmlIgnore]
        public Type FormatType
        {
            get
            {
                try
                {
                    return Elements.GetFormatType(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> FormatDictionary { get; set; }

        public void ExportToAss(
            List<Ass.AssStyle> styles,
            List<Ass.AssEvent> events)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.ExportToAss(styles, events);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void PageArrange()
        {
            try
            {
                foreach (KsgPage page in Pages)
                {
                    page.PageArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void LineArrange()
        {
            try
            {
                foreach (KsgLine line in Lines)
                {
                    line.LineArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WordArrange()
        {
            try
            {
                foreach (KsgWord word in Words)
                {
                    word.WordArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetFormatProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgPart part in Parts)
                {
                    part.SetFormatProperty(propertyName, value, false);
                }
                RefreshFormatProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshFormatProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgPart part in Parts)
                    {
                        part.RefreshFormatProperty(true, false);
                    }
                }
                FormatDictionary = Elements.GetFormat(Characters);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetWordArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgPart part in Parts)
                {
                    part.SetWordArrangementProperty(propertyName, value, false);
                }
                RefreshWordArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshWordArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgPart part in Parts)
                    {
                        part.RefreshWordArrangementProperty(true, false);
                    }
                }
                WordArrangementDictionary = Elements.GetWordArrangement(Words);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetLineArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgPart part in Parts)
                {
                    part.SetLineArrangementProperty(propertyName, value, false);
                }
                RefreshLineArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshLineArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgPart part in Parts)
                    {
                        part.RefreshLineArrangementProperty(true, false);
                    }
                }
                LineArrangementDictionary = Elements.GetLineArrangement(Lines);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetPageArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgPart part in Parts)
                {
                    part.SetPageArrangementProperty(propertyName, value, false);
                }
                RefreshPageArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshPageArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgPart part in Parts)
                    {
                        part.RefreshPageArrangementProperty(true, false);
                    }
                }
                PageArrangementDictionary = Elements.GetPageArrangement(Pages);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static KsgSubtitle Open(string fileName)
        {
            try
            {
                FileStream stream = new FileStream(fileName, FileMode.Open);
                XmlSerializer serializer = new XmlSerializer(typeof(KsgSubtitle));
                KsgSubtitle subtitle = (KsgSubtitle)serializer.Deserialize(stream);
                stream.Close();
                return subtitle;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public void Save(string fileName)
        {
            try
            {
                FileStream stream = new FileStream(fileName, FileMode.Create);
                XmlSerializer serializer = new XmlSerializer(typeof(KsgSubtitle));
                serializer.Serialize(stream, this);
                stream.Close();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static KsgSubtitle ReadTextFile(string fileName)
        {
            try
            {
                StreamReader reader = new StreamReader(fileName, Encoding.GetEncoding("Shift-JIS"), true);
                KsgSubtitle subtitle = new KsgSubtitle();
                {
                    KsgPart part = new KsgPart();
                    while (!reader.EndOfStream)
                    {
                        string text = reader.ReadLine();
                        KsgPage page = new KsgPage();
                        {
                            KsgLine line = new KsgLine();
                            foreach (char chr in text)
                            {
                                KsgWord word = new KsgWord();
                                {
                                    KsgCharacter character = new KsgCharacter();
                                    character.Char = chr;
                                    word.MainTextCharacters.Add(character);
                                }
                                line.Words.Add(word);
                            }
                            page.Lines.Add(line);
                        }
                        part.Pages.Add(page);
                    }
                    subtitle.Parts.Add(part);
                }
                reader.Close();
                return subtitle;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
