﻿using System;
using System.ComponentModel;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(PlayTimeConverter))]
    public class PlayTime
    {
        [XmlText]
        public double TotalSeconds { get; set; }

        public static PlayTime Zero = new PlayTime(0);

        public PlayTime()
        {
            try
            {
                TotalSeconds = 0;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public PlayTime(double totalSeconds)
        {
            try
            {
                TotalSeconds = totalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static bool IsTimeTag(string value)
        {
            try
            {
                Regex regex = new Regex(@"^\[(\-\d{1}|\d{2}):(\d{2}):(\d{2})\]$");
                Match match = regex.Match(value);
                return match.Success;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static PlayTime FromTimeTag(string timeTag)
        {
            try
            {
                Regex regex = new Regex(@"^\[(\-\d{1}|\d{2}):(\d{2}):(\d{2})\]$");
                Match match = regex.Match(timeTag);
                if (!match.Success)
                {
                    throw new FormatException();
                }
                int minus, minutes, seconds, centiSeconds;
                GroupCollection groups = match.Groups;
                minutes = int.Parse(groups[1].Value);
                minus = minutes < 0 ? -1 : 1;
                minutes = Math.Abs(minutes);
                seconds = int.Parse(groups[2].Value);
                centiSeconds = int.Parse(groups[3].Value);
                return new PlayTime(
                    minus * ((minutes * 60) + seconds + (centiSeconds / 100.0)));
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PlayTime.Zero;
            }
        }

        public string ToTimeTag()
        {
            try
            {
                bool isMinus = TotalSeconds < 0;
                int minutes, seconds, centiSeconds;
                double abstractTotalSeconds = Math.Abs(TotalSeconds);
                minutes = (int)(abstractTotalSeconds / 60);
                seconds = (int)(abstractTotalSeconds - (minutes * 60));
                centiSeconds = (int)((abstractTotalSeconds - (minutes * 60) - seconds) * 100);

                StringBuilder builder = new StringBuilder();
                if (isMinus)
                {
                    builder.Append("-");
                    builder.Append(minutes.ToString("D1"));
                }
                else
                {
                    builder.Append(minutes.ToString("D2"));
                }
                builder.Append(":");
                builder.Append(seconds.ToString("D2"));
                builder.Append(":");
                builder.Append(centiSeconds.ToString("D2"));
                return builder.ToString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return string.Empty;
            }
        }

        public static bool IsAssTimeString(string value)
        {
            try
            {
                Regex regex = new Regex(@"^(\-?)(\d+):(\d{2}):(\d{2})\.(\d+)$");
                Match match = regex.Match(value);
                return match.Success;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static PlayTime FromAssTimeString(string value)
        {
            try
            {
                Regex regex = new Regex(@"^(\-?)(\d+):(\d{2}):(\d{2})\.(\d+)$");
                Match match = regex.Match(value);
                if (!match.Success)
                {
                    throw new FormatException();
                }
                int minus, hours, minutes, seconds;
                double lessThanSeconds;
                GroupCollection groups = match.Groups;
                minus = groups[1].Length > 0 ? -1 : 1;
                hours = int.Parse(groups[2].Value);
                minutes = int.Parse(groups[3].Value);
                seconds = int.Parse(groups[4].Value);
                lessThanSeconds = int.Parse(groups[5].Value) / Math.Pow(10, groups[5].Length);
                return new PlayTime(
                    minus * ((hours * 60 * 60) + (minutes * 60) + seconds + lessThanSeconds));
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PlayTime.Zero;
            }
        }

        public string ToAssTimeString()
        {
            try
            {
                bool isMinus = TotalSeconds < 0;
                int hours, minutes, seconds, milliSeconds;
                double abstractTotalSeconds = Math.Abs(TotalSeconds);
                hours = (int)(abstractTotalSeconds / (60 * 60));
                minutes = (int)((abstractTotalSeconds - (hours * 60 * 60)) / 60);
                seconds = (int)(abstractTotalSeconds - (hours * 60 * 60) - (minutes * 60));
                milliSeconds =
                    (int)((abstractTotalSeconds - (hours * 60 * 60) - (minutes * 60) - seconds) * 1000);

                StringBuilder builder = new StringBuilder();
                if (isMinus)
                {
                    builder.Append("-");
                }
                builder.Append(hours.ToString("D"));
                builder.Append(":");
                builder.Append(minutes.ToString("D2"));
                builder.Append(":");
                builder.Append(seconds.ToString("D2"));
                builder.Append(".");
                builder.Append(milliSeconds.ToString("D3"));
                return builder.ToString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return string.Empty;
            }
        }

        public static PlayTime operator +(PlayTime time, PlayTimeSpan span)
        {
            try
            {
                return new PlayTime(time.TotalSeconds + span.TotalSeconds);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PlayTime.Zero;
            }
        }

        public static PlayTime operator -(PlayTime time, PlayTimeSpan span)
        {
            try
            {
                return new PlayTime(time.TotalSeconds - span.TotalSeconds);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PlayTime.Zero;
            }
        }

        public static PlayTimeSpan operator -(PlayTime time1, PlayTime time2)
        {
            try
            {
                return new PlayTimeSpan(time1.TotalSeconds - time2.TotalSeconds);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PlayTimeSpan.Zero;
            }
        }

        public static bool operator ==(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds == time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static bool operator !=(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds != time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static bool operator <(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds < time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static bool operator <=(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds <= time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static bool operator >(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds > time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public static bool operator >=(PlayTime time1, PlayTime time2)
        {
            try
            {
                return time1.TotalSeconds >= time2.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public override bool Equals(object obj)
        {
            return base.Equals(obj);
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
