﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Reflection;
using System.Text;
using Kasuga.Ass;

namespace Kasuga.StandardPlugins.Formats
{
    public class PreciseDrawingWay
    {
        protected const int FORE_LAYER = 120;
        protected const int BORDER_LAYER = 110;
        protected const int SHADOW_LAYER = 100;

        public static void Draw(
            FormatBase format,
            SortedDictionary<int, Bitmap> bitmaps,
            KsgCharacter character,
            bool isWiped)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return;
                }
                else
                {
                    Size resolution = character.ParentSubtitle.Resolution;
                    if (!bitmaps.ContainsKey(FORE_LAYER))
                    {
                        bitmaps.Add(FORE_LAYER, new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb));
                    }
                    if (!bitmaps.ContainsKey(BORDER_LAYER))
                    {
                        bitmaps.Add(BORDER_LAYER, new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb));
                    }
                    if (!bitmaps.ContainsKey(SHADOW_LAYER))
                    {
                        bitmaps.Add(SHADOW_LAYER, new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb));
                    }

                    //fore layer
                    using (GraphicsPath forePath = format.GetForePath(character))
                    {
                        using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                        {
                            using (Graphics graphics = Graphics.FromImage(bitmap))
                            {
                                using (Brush brush = isWiped ?
                                    new SolidBrush(format.AfterForeColor) :
                                    new SolidBrush(format.BeforeForeColor))
                                {
                                    graphics.FillPath(brush, forePath);
                                    brush.Dispose();
                                }
                                graphics.Dispose();
                            }
                            if (format.BorderWidthX <= 0 && format.BorderWidthY <= 0)
                            {
                                format.Rasterize(bitmap, forePath);
                            }
                            using (Graphics graphics = Graphics.FromImage(bitmaps[FORE_LAYER]))
                            {
                                graphics.DrawImage(bitmap, new Point(0, 0));
                                graphics.Dispose();
                            }
                            bitmap.Dispose();
                        }
                        forePath.Dispose();
                    }

                    //border layer
                    if (format.BorderWidthX > 0 || format.BorderWidthY > 0)
                    {
                        using (GraphicsPath forePath = format.GetForePath(character))
                        {
                            using (GraphicsPath widenPath = format.GetWidenPath(character))
                            {
                                using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                                {
                                    using (Graphics graphics = Graphics.FromImage(bitmap))
                                    {
                                        using (Brush brush = isWiped ?
                                            new SolidBrush(format.AfterBorderColor) :
                                            new SolidBrush(format.BeforeBorderColor))
                                        {
                                            using (Region region = new Region(widenPath))
                                            {
                                                region.Exclude(forePath);
                                                graphics.FillRegion(brush, region);
                                                region.Dispose();
                                            }
                                            brush.Dispose();
                                        }
                                        graphics.Dispose();
                                    }
                                    format.Rasterize(bitmap, widenPath);
                                    using (Graphics graphics = Graphics.FromImage(bitmaps[BORDER_LAYER]))
                                    {
                                        graphics.DrawImage(bitmap, new Point(0, 0));
                                        graphics.Dispose();
                                    }
                                    bitmap.Dispose();
                                }
                                widenPath.Dispose();
                            }
                            forePath.Dispose();
                        }
                    }

                    //shadow layer
                    if (format.ShadowDepthX != 0 || format.ShadowDepthY != 0)
                    {
                        using (GraphicsPath widenPath = format.GetWidenPath(character))
                        {
                            using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                            {
                                using (Graphics graphics = Graphics.FromImage(bitmap))
                                {
                                    using (Brush brush = isWiped ?
                                        new SolidBrush(format.AfterShadowColor) :
                                        new SolidBrush(format.BeforeShadowColor))
                                    {
                                        using (Matrix matrix = new Matrix())
                                        {
                                            matrix.Translate(format.ShadowDepthX, format.ShadowDepthY);
                                            widenPath.Transform(matrix);
                                            matrix.Dispose();
                                        }
                                        graphics.FillPath(brush, widenPath);
                                        brush.Dispose();
                                    }
                                    graphics.Dispose();
                                }
                                format.Rasterize(bitmap, widenPath);
                                using (Graphics graphics = Graphics.FromImage(bitmaps[SHADOW_LAYER]))
                                {
                                    graphics.DrawImage(bitmap, new Point(0, 0));
                                    graphics.Dispose();
                                }
                                bitmap.Dispose();
                            }
                            widenPath.Dispose();
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static AssStyle CreateStyle(
            FormatBase format,
            List<AssStyle> styles,
            bool isWiped,
            FillRegionKind fillRegion)
        {
            try
            {
                Color foreColor;
                Color borderColor;
                float borderWidthX;
                switch (fillRegion)
                {
                    case FillRegionKind.Fore:
                    default:
                        foreColor = isWiped ? format.AfterForeColor : format.BeforeForeColor;
                        borderColor = Color.FromArgb(0, Color.Black);
                        borderWidthX = 0;
                        break;
                    case FillRegionKind.Border:
                        foreColor = Color.FromArgb(0, Color.Black);
                        borderColor = isWiped ? format.AfterBorderColor : format.BeforeBorderColor;
                        borderWidthX = format.BorderWidthX;
                        break;
                    case FillRegionKind.Shadow:
                        foreColor = isWiped ? format.AfterShadowColor : format.BeforeShadowColor;
                        borderColor = isWiped ? format.AfterShadowColor : format.BeforeShadowColor;
                        borderWidthX = format.BorderWidthX;
                        break;
                }

                AssStyle style = new AssStyle(
                    (styles.Count + 1).ToString(),
                    format.FontName,
                    format.FontSize,
                    foreColor,
                    Color.Black,
                    borderColor,
                    Color.Black,
                    format.IsBold,
                    format.IsItalic,
                    format.HasUnderline,
                    format.HasStrikeout,
                    format.ScaleX,
                    format.ScaleY,
                    0,
                    format.RotateZ,
                    AssBorderStyleKind.Normal,
                    borderWidthX,
                    0,
                    ContentAlignment.MiddleCenter,
                    0,
                    0,
                    0,
                    format.GdiCharSet);

                return Util.CheckContainedStyle(styles, style);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public static void GetPositionTag(
            FormatBase format,
            KsgCharacter character,
            out string positionTag,
            out string shadowPositionTag)
        {
            try
            {
                Func<PointF, string> getPositionTag;
                {
                    getPositionTag = (PointF position) =>
                    {
                        StringBuilder builder = new StringBuilder();
                        builder.Append(@"{\pos(").Append(position.X)
                            .Append(",").Append(position.Y).Append("}");
                        return builder.ToString();
                    };
                }
                {
                    Corners corners = format.GetCharacterCorners(character);
                    PointF position = Util.GetIntersectionPoint(corners);
                    positionTag = getPositionTag(position);
                    position.X += format.ShadowDepthX;
                    position.Y += format.ShadowDepthY;
                    shadowPositionTag = getPositionTag(position);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                positionTag = string.Empty;
                shadowPositionTag = string.Empty;
            }
        }

        public static void AddDialogues(
            List<AssEvent> events,
            PlayTime start,
            PlayTime end,
            AssStyle foreStyle,
            AssStyle borderStyle,
            AssStyle shadowStyle,
            string foreTags,
            string borderTags,
            string shadowTags,
            char chr)
        {
            try
            {
                {
                    AssEvent dialogue = new AssEvent(
                        AssEventKind.Dialogue,
                        FORE_LAYER,
                        start,
                        end,
                        foreStyle.Name,
                        string.Empty,
                        0,
                        0,
                        0,
                        string.Empty,
                        foreTags + chr.ToString());
                    events.Add(dialogue);
                }
                {
                    AssEvent dialogue = new AssEvent(
                        AssEventKind.Dialogue,
                        FORE_LAYER,
                        start,
                        end,
                        borderStyle.Name,
                        string.Empty,
                        0,
                        0,
                        0,
                        string.Empty,
                        borderTags + chr.ToString());
                    events.Add(dialogue);
                }
                {
                    AssEvent dialogue = new AssEvent(
                        AssEventKind.Dialogue,
                        FORE_LAYER,
                        start,
                        end,
                        shadowStyle.Name,
                        string.Empty,
                        0,
                        0,
                        0,
                        string.Empty,
                        shadowTags + chr.ToString());
                    events.Add(dialogue);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
