﻿using System;
using System.ComponentModel;
using System.Reflection;
using System.Windows.Forms;

namespace Kasuga
{
    public class PageArrangementPropertyDescriptor : PropertyDescriptor
    {
        public PageArrangementPropertyDescriptor(PropertyDescriptor descriptor)
            : base(descriptor)
        {
            try
            {
                _descriptor = descriptor;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private PropertyDescriptor _descriptor;

        public override object GetValue(object component)
        {
            try
            {
                IHasPageArrangementDictionary element = (IHasPageArrangementDictionary)component;
                return element.PageArrangementDictionary[_descriptor.Name];
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public override void SetValue(object component, object value)
        {
            try
            {
                IHasPageArrangementDictionary element = (IHasPageArrangementDictionary)component;
                element.SetPageArrangementProperty(_descriptor.Name, value, true);
                if (MessageBox.Show("該当ページの構成要素を再配置しますか？", "再配置", MessageBoxButtons.YesNo) == DialogResult.Yes)
                {
                    if (element is KsgSubtitle)
                    {
                        ((KsgSubtitle)element).PageArrange();
                    }
                    else if (element is KsgPart)
                    {
                        ((KsgPart)element).PageArrange();
                    }
                    else if (element is KsgPage)
                    {
                        ((KsgPage)element).PageArrange();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public override bool CanResetValue(object component)
        {
            try
            {
                IHasPageArrangementDictionary element = (IHasPageArrangementDictionary)component;
                return element.BasePageArrangement != null;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public override void ResetValue(object component)
        {
            try
            {
                IHasPageArrangementDictionary element = (IHasPageArrangementDictionary)component;
                IPageArrangement basePageArrangement = element.BasePageArrangement;
                if (basePageArrangement != null)
                {
                    PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(basePageArrangement);
                    PropertyDescriptor descriptor = descriptors.Find(_descriptor.Name, true);
                    SetValue(component, descriptor.GetValue(basePageArrangement));
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public override bool ShouldSerializeValue(object component)
        {
            try
            {
                IHasPageArrangementDictionary element = (IHasPageArrangementDictionary)component;
                IPageArrangement basePageArrangement = element.BasePageArrangement;
                if (basePageArrangement == null)
                {
                    return true;
                }
                else
                {
                    PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(basePageArrangement);
                    PropertyDescriptor descriptor = descriptors.Find(_descriptor.Name, true);
                    return !descriptor.GetValue(basePageArrangement).Equals(GetValue(component));
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return false;
            }
        }

        public override Type ComponentType
        {
            get
            {
                try
                {
                    return typeof(IHasPageArrangementDictionary);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public override bool IsReadOnly
        {
            get
            {
                try
                {
                    return _descriptor.IsReadOnly;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return false;
                }
            }
        }

        public override Type PropertyType
        {
            get
            {
                try
                {
                    return _descriptor.PropertyType;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }
    }
}
