/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-bookmark.h"

#include "gobject-utils.h"
#include "intl.h"

enum {
	CHILDREN_UPDATED_SIGNAL,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_TITLE,
	PROP_URI,
	PROP_DESCRIPTION,
	PROP_LAST_MODIFIED,
	PROP_EDITABLE
};

static void kz_bookmark_class_init   (KzBookmarkClass *klass);
static void kz_bookmark_init         (KzBookmark      *bookmark);
static void kz_bookmark_dispose      (GObject         *object);
static void kz_bookmark_set_property (GObject         *object,
				      guint            prop_id,
				      const GValue    *value,
				      GParamSpec      *pspec);
static void kz_bookmark_get_property (GObject         *object,
				      guint            prop_id,
				      GValue          *value,
				      GParamSpec      *pspec);

static GObjectClass *parent_class = NULL;
static gint kz_bookmark_signals[LAST_SIGNAL] = {0};

static GQuark title_quark       = 0;
static GQuark uri_quark         = 0;
static GQuark description_quark = 0;
static GQuark last_mod_quark    = 0;


KZ_OBJECT_GET_TYPE(kz_bookmark, "KzBookmark", KzBookmark,
		   kz_bookmark_class_init, kz_bookmark_init,
		   G_TYPE_OBJECT)


static void
kz_bookmark_class_init (KzBookmarkClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;

	object_class->dispose = kz_bookmark_dispose;
	object_class->set_property = kz_bookmark_set_property;
	object_class->get_property = kz_bookmark_get_property;

	g_object_class_install_property(
		object_class,
		 PROP_TITLE,
		 g_param_spec_string(
			 "title",
			 _("Title"),
			 _("The title of the bookmark"),
			 NULL,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_URI,
		 g_param_spec_string(
			 "uri",
			 _("URI"),
			 _("The URI of the bookmark"),
			 NULL,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_URI,
		 g_param_spec_string(
			 "description",
			 _("Description"),
			 _("The description of the bookmark"),
			 NULL,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_EDITABLE,
		 g_param_spec_uint(
			 "last-modified",
			 _("Last Modified"),
			 _("Last modification time of the link"),
			 0,
			 G_MAXUINT,
			 0,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_EDITABLE,
		 g_param_spec_boolean(
			 "editable",
			 _("Editable"),
			 _("Whether attributes of the bookmark is writable or not"),
			 TRUE,
			 G_PARAM_READWRITE));

	/* FIXME! add other properties */
	/*
	 * candidates:
	 * 
	 *   * Last-Modified time
	 *   * Last-Accessed time
	 *   * frequency
	 *   * ......
	 */

	title_quark       = g_quark_from_string("KzBookmark::Title");
	uri_quark         = g_quark_from_string("KzBookmark::URI");
	description_quark = g_quark_from_string("KzBookmark::Description");
	last_mod_quark    = g_quark_from_string("KzBookmark::LastModified");
}


static void
kz_bookmark_init (KzBookmark *bookmark)
{
	bookmark->flags = 0;
}


static void
kz_bookmark_dispose (GObject *object)
{
	KzBookmark *bookmark;

	g_return_if_fail(KZ_IS_BOOKMARK(object));

	bookmark = KZ_BOOKMARK(object);

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


#define CHANGE_STR(obj, quark, value) \
{ \
	gchar *str = g_object_get_qdata((obj), (quark)); \
	g_free(str); \
	g_object_set_qdata_full((obj), (quark), (value), \
				(GDestroyNotify) g_free); \
}


static void
kz_bookmark_set_property (GObject *object,
			  guint prop_id,
			  const GValue *value,
			  GParamSpec *pspec)
{
	KzBookmark *bookmark = KZ_BOOKMARK(object);

	switch (prop_id) {
	case PROP_TITLE:
		CHANGE_STR(object, title_quark, g_value_dup_string(value));
		break;
	case PROP_URI:
		CHANGE_STR(object, uri_quark, g_value_dup_string(value));
		break;
	case PROP_DESCRIPTION:
		CHANGE_STR(object, description_quark,
			   g_value_dup_string(value));
		break;
	case PROP_LAST_MODIFIED:
		g_object_set_qdata(object, last_mod_quark,
				   GUINT_TO_POINTER(g_value_get_uint(value)));
		break;
	case PROP_EDITABLE:
		if (g_value_get_boolean(value))
			bookmark->flags |= KZ_BOOKMARK_EDITABLE_FLAG;
		else
			bookmark->flags &= ~KZ_BOOKMARK_EDITABLE_FLAG;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
kz_bookmark_get_property (GObject *object,
		     guint prop_id,
		     GValue *value,
		     GParamSpec *pspec)
{
	KzBookmark *bookmark = KZ_BOOKMARK(object);
	gchar *str;
	guint time;

	switch (prop_id) {
	case PROP_TITLE:
		str = g_object_get_qdata(object, title_quark);
		g_value_set_string(value, str);
		break;
	case PROP_URI:
		str = g_object_get_qdata(object, uri_quark);
		g_value_set_string(value, str);
		break;
	case PROP_DESCRIPTION:
		str = g_object_get_qdata(object, description_quark);
		g_value_set_string(value, str);
		break;
	case PROP_LAST_MODIFIED:
		time = GPOINTER_TO_UINT(g_object_get_qdata(object, last_mod_quark));
		g_value_set_uint(value, time);
		break;
	case PROP_EDITABLE:
		if (bookmark->flags & KZ_BOOKMARK_EDITABLE_FLAG)
			g_value_set_boolean(value, TRUE);
		else
			g_value_set_boolean(value, FALSE);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


KzBookmark *
kz_bookmark_new (void)
{
	KzBookmark *bookmark = KZ_BOOKMARK(g_object_new(KZ_TYPE_BOOKMARK,
							NULL));
	return bookmark;
}


KzBookmark *
kz_bookmark_new_with_attrs (const gchar *title,
			    const gchar *uri,
			    const gchar *description)
{
	KzBookmark *bookmark;

	bookmark = KZ_BOOKMARK(g_object_new(KZ_TYPE_BOOKMARK,
					    "title",       title,
					    "uri",         uri,
					    "description", description,
					    NULL));
	return bookmark;
}


const gchar *
kz_bookmark_get_title (KzBookmark *bookmark)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK(bookmark), NULL);
	return g_object_get_qdata(G_OBJECT(bookmark), title_quark);
}


const gchar *
kz_bookmark_get_uri (KzBookmark *bookmark)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK(bookmark), NULL);
	return g_object_get_qdata(G_OBJECT(bookmark), uri_quark);
}


const gchar *
kz_bookmark_get_description (KzBookmark *bookmark)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK(bookmark), NULL);
	return g_object_get_qdata(G_OBJECT(bookmark), description_quark);
}


guint
kz_bookmark_get_last_modified(KzBookmark *bookmark)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK(bookmark), 0);
	return GPOINTER_TO_UINT(g_object_get_qdata(G_OBJECT(bookmark), last_mod_quark));
}


void
kz_bookmark_set_title (KzBookmark *bookmark,
		       const gchar *title)
{
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));
	g_object_set(G_OBJECT(bookmark), "title", title, NULL);
}


void
kz_bookmark_set_uri (KzBookmark *bookmark,
		     const gchar *uri)
{
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));
	g_object_set(G_OBJECT(bookmark), "uri", uri, NULL);
}


void
kz_bookmark_set_description (KzBookmark *bookmark,
			     const gchar *description)
{
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));
	g_object_set(G_OBJECT(bookmark), "description", description, NULL);
}


void
kz_bookmark_set_last_modified(KzBookmark *bookmark,
			      guint time)
{
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));
	g_object_set(G_OBJECT(bookmark), "last-modified", time, NULL);
}

