/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "kz-meta.h"
#include "gobject-utils.h"
#include "intl.h"
#include "kazehakase.h"
#include "kz-profile.h"
#include "kz-http.h"

enum {
	START_SIGNAL,
	UPDATE_START_SIGNAL,
	UPDATE_COMPLETED_SIGNAL,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_TITLE,
	PROP_URI,
	PROP_INTERVAL,
};

static void kz_meta_class_init      (KzMETAClass *klass);
static void kz_meta_init            (KzMETA *meta);
static void kz_meta_set_property    (GObject *object,
				     guint prop_id,
				     const GValue *value,
				     GParamSpec *pspec);
static void kz_meta_get_property    (GObject *object,
				     guint prop_id,
				     GValue *value,
				     GParamSpec *pspec);

static void kz_meta_file_fetch       (KzMETA *meta);
static void kz_meta_real_items_update(KzMETA *meta);
static void kz_meta_items_dispose    (KzMETA *meta);

static void kz_meta_real_parse_from_string (KzMETA *meta, gpointer user_data,
					    const gchar *buffer, guint length,
					    GError **error);
static void kz_meta_update_complete(KzHTTP *http, KzMETA *meta);

static GObjectClass *parent_class = NULL;
static gint kz_meta_signals[LAST_SIGNAL] = {0};

KZ_OBJECT_GET_TYPE(kz_meta, "KzMETA", KzMETA,
		   kz_meta_class_init, kz_meta_init,
		   G_TYPE_OBJECT)


static void
kz_meta_class_init (KzMETAClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;

	object_class->dispose = kz_meta_dispose;
	object_class->set_property = kz_meta_set_property;
	object_class->get_property = kz_meta_get_property;
	
	klass->update_start         = NULL;
	klass->update_completed     = NULL;
	
	klass->kz_meta_parse_from_string = kz_meta_real_parse_from_string;

	g_object_class_install_property(
		object_class,
		 PROP_TITLE,
		 g_param_spec_string(
			 "title",
			 _("Title"),
			 _("The title used for menu items"),
			 NULL,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_URI,
		 g_param_spec_string(
			 "uri",
			 _("URI"),
			 _("The URI of META file"),
			 NULL,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		 PROP_INTERVAL,
		 g_param_spec_uint(
			 "interval",
			 _("Interval"),
			 _("Update interval"),
			 0,
			 G_MAXUINT,
			 0,
			 G_PARAM_READWRITE));

	kz_meta_signals[UPDATE_START_SIGNAL]
		= g_signal_new ("update_start",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzMETAClass, update_start),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);

	kz_meta_signals[UPDATE_COMPLETED_SIGNAL]
		= g_signal_new ("update_completed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzMETAClass, update_completed),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);
}


static void
kz_meta_init (KzMETA *meta)
{
	meta->items           = NULL;
	meta->title           = NULL;
	meta->uri             = NULL;
	meta->timer_id        = 0;
	meta->update_interval = 0;

	meta->state = KZ_META_NORMAL;
}


void
kz_meta_dispose (GObject *object)
{
	KzMETA *meta;

	g_return_if_fail(KZ_IS_META(object));

	meta = KZ_META(object);

	if (meta->timer_id)
		gtk_timeout_remove(meta->timer_id);
	meta->timer_id = 0;

	kz_meta_items_dispose(meta);
	if (meta->uri)
		g_free(meta->uri);
	if (meta->title)
		g_free(meta->title);

	meta->uri = NULL;
	meta->title = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}

static void
kz_meta_set_property (GObject *object,
		     guint prop_id,
		     const GValue *value,
		     GParamSpec *pspec)
{
	KzMETA *meta = KZ_META(object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_free(meta->title);
		meta->title = g_value_dup_string(value);
		break;
	case PROP_URI:
		g_free(meta->uri);
		meta->uri = g_value_dup_string(value);
		break;
	case PROP_INTERVAL:
		meta->update_interval = g_value_get_uint(value);

		if (meta->timer_id)
			gtk_timeout_remove(meta->timer_id);
		meta->timer_id = 0;

		if (meta->update_interval > 0)
			meta->timer_id
				= gtk_timeout_add(60000 * meta->update_interval,
						  (void *)&kz_meta_update, meta);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
kz_meta_get_property (GObject *object,
		     guint prop_id,
		     GValue *value,
		     GParamSpec *pspec)
{
	KzMETA *meta = KZ_META(object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_value_set_string(value, meta->title);
		break;
	case PROP_URI:
		g_value_set_string(value, meta->uri);
		break;
	case PROP_INTERVAL:
		g_value_set_uint(value, meta->update_interval);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


KzMETA *
kz_meta_new (gchar *uri, gchar *title, guint update_interval)
{
	KzMETA *meta = KZ_META(g_object_new(KZ_TYPE_META,
					 "uri", uri,
					 "title", title,
					 "interval", update_interval,
					 NULL));

	kz_meta_update(meta);

	return meta;
}


/*  calling virtual function */
void
kz_meta_parse_from_string (KzMETA *meta, gpointer user_data,
			   const gchar *buffer, guint length,
			   GError **error)
{
	KZ_META_GET_CLASS(meta)->kz_meta_parse_from_string(meta, user_data,
							   buffer, length,
							   error);
}


static void
kz_meta_real_parse_from_string (KzMETA *meta, gpointer user_data,
				const gchar *buffer, guint length,
				GError **error)
{
	g_warning("meta file real parsing");
}


static void
kz_meta_file_fetch(KzMETA *meta)
{
	KzHTTP *http;
	http = kz_http_new(meta->uri);
	g_signal_connect(G_OBJECT(http), "fetch_completed",
			 G_CALLBACK(kz_meta_update_complete), meta);
	kz_http_file_fetch(http);
}

static void
kz_meta_update_complete(KzHTTP *http, KzMETA *meta)
{	
	if (http->buf->len != 0)
	{
		kz_meta_parse_from_string(meta, NULL,
					  http->buf->str,
					  http->buf->len,
					  NULL);
	}
	kz_http_dispose(G_OBJECT(http));
	meta->state = KZ_META_NORMAL;
	g_signal_emit(G_OBJECT (meta),
		      kz_meta_signals[UPDATE_COMPLETED_SIGNAL],
		      0);
}

static void 
kz_meta_items_update (KzMETA *meta)
{
	g_return_if_fail(KZ_IS_META(meta));

	kz_meta_items_dispose(meta);
	kz_meta_file_fetch(meta);
}


gboolean
kz_meta_update (KzMETA *meta)
{
	if (meta->state == KZ_META_LOADING)
		return TRUE;
	meta->state = KZ_META_LOADING;

	g_signal_emit(G_OBJECT (meta),
		      kz_meta_signals[UPDATE_START_SIGNAL],
		      0);
	kz_meta_items_update(meta);
	return TRUE;
}


void
kz_meta_force_update (KzMETA *meta)
{
	if (meta->state == KZ_META_LOADING)
		return;
	kz_meta_update(meta);
}


void 
kz_meta_items_dispose (KzMETA *meta)
{
	GSList *node;

	if (!meta->items) return;

	for(node = meta->items; node; node = g_slist_next(node))
	{
		KzMETAItem *item = node->data;
		if(item->title)
			g_free(item->title);
		if(item->link)
			g_free(item->link);
		if(item->description)
			g_free(item->description);
	}
	g_slist_free(meta->items);
	meta->items = NULL;
}


void
kz_meta_set_interval (KzMETA *meta, guint interval)
{
	g_return_if_fail(KZ_IS_META(meta));

	g_object_set(G_OBJECT(meta), "interval", interval);
}
