/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-metalist.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gobject-utils.h"
#include "kz-rss.h"
#include "kz-lirs.h"
#include "kz-hinadi.h"
#include "kz-google.h"

enum {
	UPDATED_SIGNAL,
	LAST_SIGNAL
};


struct _KzMetaType {
	const gchar *element_name;
	GType (*get_type) (void);
} meta_types[] = {
	{"rss",  kz_rss_get_type},
	{"lirs", kz_lirs_get_type},
	{"di",   kz_hinadi_get_type},
	{"google",kz_google_get_type},
};

KzMETAList *metalist_single = NULL;

static void kz_meta_list_class_init (KzMETAListClass *klass);
static void kz_meta_list_init       (KzMETAList      *metalist);
static void kz_meta_list_dispose    (GObject        *object);

static gboolean kz_meta_list_parse_from_file   (KzMETAList   *metalist,
					       gpointer     user_data,
					       const gchar *filename,
					       GError     **error);
static gboolean kz_meta_list_parse_from_string (KzMETAList   *metalist,
					       gpointer     user_data,
					       const gchar *buffer,
					       guint        length,
					       GError     **error);

static GObjectClass *parent_class = NULL;
static gint kz_meta_list_signals[LAST_SIGNAL] = {0};


KZ_OBJECT_GET_TYPE(kz_meta_list, "KzMETAList", KzMETAList,
		   kz_meta_list_class_init, kz_meta_list_init,
		   G_TYPE_OBJECT)

static void
kz_meta_list_class_init (KzMETAListClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;

	/* GObject class */
	gobject_class->dispose = kz_meta_list_dispose;

	kz_meta_list_signals[UPDATED_SIGNAL]
		= g_signal_new ("updated",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzMETAListClass, updated),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);
}

static void
kz_meta_list_init (KzMETAList *metalist)
{
	metalist->items = NULL;
}

static void
kz_meta_list_dispose (GObject *object)
{
	KzMETAList *metalist = KZ_META_LIST(object);

	if (metalist->items)
	{
		g_slist_foreach(metalist->items, (GFunc) g_object_unref, NULL);
		g_slist_free(metalist->items);
		metalist->items = NULL;
	}

	if (metalist_single && metalist_single == metalist)
	{
		metalist_single = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


KzMETAList *
kz_meta_list_new (void)
{
	KzMETAList *metalist
		= KZ_META_LIST(g_object_new(KZ_TYPE_META_LIST, NULL));
        gchar *toolbarfile;
	struct stat st;

	/* build the filename */
        toolbarfile = g_strconcat(g_get_home_dir(),
				  "/.kazehakase/toolbar.xml",
				  NULL);
	/* use system wide config file if user's file isn't exist. */
	if (stat(toolbarfile, &st))
	{
		g_free(toolbarfile);
		toolbarfile = g_strconcat(SYSCONFDIR,
					  "/toolbar.xml",
					  NULL);
	}

	metalist->items = NULL;

	/* load toolbar config file */
	kz_meta_list_parse_from_file(metalist, NULL,
				    toolbarfile, NULL);

	g_free(toolbarfile);

	return metalist;
}


KzMETAList *
kz_meta_list_get_instance (void)
{
	if (!metalist_single)
		metalist_single = kz_meta_list_new();
	else
		g_object_ref(metalist_single);

	return metalist_single;
}


void
kz_meta_list_set_list (KzMETAList *metalist, GSList *new_list)
{
	GSList *node, *old_list;

	g_return_if_fail(KZ_IS_META_LIST(metalist));

	/* verify */
	for (node = new_list; node; node = g_slist_next(node))
	{
		KzMETA *meta = node->data;
		g_return_if_fail(KZ_IS_META(meta));
	}

	old_list = metalist->items;

	if (new_list)
	{
		metalist->items = g_slist_copy(new_list);
		g_slist_foreach(metalist->items, (GFunc) g_object_ref, NULL);
	}
	else
	{
		metalist->items = NULL;
	}

	if (old_list)
	{
		g_slist_foreach(old_list, (GFunc) g_object_unref, NULL);
		g_slist_free(old_list);
	}

	g_signal_emit(G_OBJECT (metalist),
		      kz_meta_list_signals[UPDATED_SIGNAL],
		      0);
}


void
kz_meta_list_save (KzMETAList *metalist)
{
	gchar *file;
	GString *xml;
	FILE *fp;
	GSList *node;

	g_return_if_fail(KZ_IS_META_LIST(metalist));

	xml = g_string_new(NULL);

	g_string_append(xml, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	g_string_append(xml, "<metabar>\n");

	/* add sites */
	for (node = metalist->items; node; node = g_slist_next(node))
	{
		KzMETA *meta = node->data;
		gchar *escaped_text[2];
		gint i;
		const gchar *element_name = "rss";

		escaped_text[0] = g_markup_escape_text(meta->uri, -1);
		escaped_text[1] = g_markup_escape_text(meta->title, -1);

		for (i = 0; i < G_N_ELEMENTS(meta_types); i++)
		{
			const gchar *str = meta_types[i].element_name;
			GType gtype = meta_types[i].get_type();

			if (G_TYPE_CHECK_INSTANCE_TYPE(meta, gtype)
			    && str && *str)
			{
				element_name = meta_types[i].element_name;
				break;
			}
		}
		g_string_append_printf(xml, "  <%s ", element_name);
		g_string_append_printf(xml, "uri=\"%s\" title=\"%s\"",
				       escaped_text[0], escaped_text[1]);
		g_free(escaped_text[0]);
		g_free(escaped_text[1]);

		if (meta->update_interval > 0)
			g_string_append_printf(xml, " interval=\"%d\"",
					       meta->update_interval);

		g_string_append(xml, " />\n");
	}

	g_string_append(xml, "</metabar>");

        file = g_strconcat(g_get_home_dir(),
			   "/.kazehakase/toolbar.xml",
			   NULL);
	fp = fopen(file, "w");
	if (fp)
	{
		fwrite(xml->str, xml->len, 1, fp);
		fclose(fp);
	}
	else
	{
		/* error handling */
	}

	g_free(file);

	g_string_free(xml, TRUE);
}


static gint
find_type (const gchar *element_name)
{
	gint i;

	g_return_val_if_fail(element_name && *element_name, -1);

	for (i = 0; i < G_N_ELEMENTS(meta_types); i++)
	{
		if (!strcmp(meta_types[i].element_name, element_name))
			return i;
	}

	return -1;
}



/*****************************************************************************
 *                                                                           *
 *                             Config parser                                 *
 *                                                                           *
 *****************************************************************************/
typedef enum {
	STATE_START,
	STATE_METABAR,
	STATE_META,
	STATE_END
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;

	KzMETAList *metalist;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	int i;
	ParseContext *ctx = user_data;

	if (ctx->state == STATE_START && !strcmp(element_name, "metabar"))
	{
		ctx->state = STATE_METABAR;
	}
	else if (ctx->state == STATE_METABAR)
	{
		gchar *uri = NULL;
		gchar *title = NULL;
		gint update_interval = -1;
		KzMETA *meta = NULL;
		KzMETAList *metalist = ctx->metalist;
		gint type = find_type(element_name);

		g_return_if_fail(type >= 0);
		g_return_if_fail(meta_types[type].get_type);

		ctx->state = STATE_META;

		/* get attributes */
		for (i = 0; attribute_names[i] != NULL; i++)
		{
			if (!strcmp(attribute_names[i], "uri") && !uri)
			{
				uri = g_strdup(attribute_values[i]);
			}
			else if (!strcmp(attribute_names[i], "title") && !title)
			{
				title = g_strdup(attribute_values[i]);
			}
			else if (!strcmp(attribute_names[i], "interval")
				 && update_interval < 0)
			{
				update_interval = atoi(attribute_values[i]);
			}
		}

		/* create new KzMETA object */
		meta = g_object_new(meta_types[type].get_type(),
				    "uri",      uri,
				    "title",    title,
				    "interval", update_interval,
				    NULL);
		if (KZ_IS_META(meta))
		{
			kz_meta_update(meta);
			metalist->items = g_slist_append(metalist->items, meta);
		}
		else
		{
			g_warning("Invalid object type!");
			g_object_unref(meta);
		}

		g_free(uri);
		g_free(title);
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;
	
	switch (ctx->state)
	{
	 case STATE_START:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	 case STATE_METABAR:
		ctx->state = STATE_END;
		break;
	 case STATE_META:
		ctx->state = STATE_METABAR;
		break;
	 case STATE_END:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	NULL,
	NULL,
	NULL
};

static gboolean
kz_meta_list_parse_from_file (KzMETAList *metalist, gpointer user_data,
			     const gchar *filename, GError **error)
{
	gchar *buffer;
	gint length;
	gboolean res;

	if (!g_file_get_contents (filename, &buffer, &length, error))
		return FALSE;

	res = kz_meta_list_parse_from_string(metalist, user_data,
					    buffer, length, error);
	g_free(buffer);

	return res;
}

static gboolean
kz_meta_list_parse_from_string (KzMETAList *metalist, gpointer user_data,
			       const gchar *buffer, guint length,
			       GError **error)
{
	ParseContext ctx = { 0 };
	GMarkupParseContext *context;
	gboolean res = FALSE;

	g_return_val_if_fail(buffer != NULL, FALSE);

	ctx.state   = STATE_START;
	ctx.metalist = metalist;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);

	if (g_markup_parse_context_parse(context, buffer, length, error))
	{
		if (g_markup_parse_context_end_parse(context, error))
			res = TRUE;
	}

	g_markup_parse_context_free (context);

	return res;
}
