/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>

#include "kz-rss.h"
#include "gobject-utils.h"
#include "utils.h"
#include "intl.h"

static void     kz_rss_class_init      (KzRSSClass *klass);
static void     kz_rss_init            (KzRSS *rss);
/* static void     kz_rss_dispose         (GObject *object); */

static gchar   *kz_rss_get_encoding    (gchar *src);
static gchar   *kz_rss_convert_to_utf8 (gchar *src,
					guint src_len,
					guint *dest_len);

static void     kz_rss_parse_from_string (KzMETA *meta, gpointer user_data,
					  const gchar *buffer, guint length,
					  GError **error);

static KzMETAClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_rss, "KzRSS", KzRSS,
		   kz_rss_class_init, kz_rss_init,
		   KZ_TYPE_META)

static void
kz_rss_class_init (KzRSSClass *klass)
{
	GObjectClass *object_class;
	KzMETAClass *meta_class;
	
	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;
	meta_class   = (KzMETAClass *) klass;

	meta_class->kz_meta_parse_from_string = kz_rss_parse_from_string;
}

static void
kz_rss_init (KzRSS *rss)
{
	rss->rss_title       = NULL;
	rss->rss_link        = NULL;
	rss->rss_description = NULL;
	rss->rss_date        = NULL;
}

KzRSS *
kz_rss_new (gchar *uri, gchar *title, guint update_interval)
{
	KzRSS *rss = KZ_RSS(g_object_new(KZ_TYPE_RSS,
					 "uri", uri,
					 "title", title,
					 "interval", update_interval,
					 NULL));
	kz_meta_update((KzMETA*)rss);

	return rss;
}

typedef enum {
	STATE_START   = (1 << 0),
	STATE_ROOT    = (1 << 1),
	STATE_CHANNEL = (1 << 2),
	STATE_ITEM    = (1 << 3),
	STATE_TITLE   = (1 << 4),
	STATE_LINK    = (1 << 5),
	STATE_DESC    = (1 << 6),
	STATE_END     = (1 << 7),
	STATE_ERROR   = (1 << 8)
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;
	ParseState current_state;

	KzMETA *meta;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	ParseContext *ctx = user_data;

	ctx->current_state = STATE_ERROR;
	
	switch (element_name[0])
	{
	 case 'r':
		if (ctx->state == STATE_START && 
		    (!strcmp(element_name, "rdf:RDF") ||
		     !strcmp(element_name, "rss")))
		{
			ctx->state = STATE_ROOT;
			ctx->current_state = STATE_ROOT;
		}
		break;
	 case 'i':
		if ((ctx->state & STATE_ROOT) && !strcmp(element_name, "item")
		    && strcmp(element_name, "items"))
		{
			KzMETAItem *item;

			ctx->state |= STATE_ITEM;
			ctx->current_state = STATE_ITEM;

			/* append new KzRSSItem */
			item = g_new0(KzMETAItem, 1);
			item->meta = ctx->meta;
			item->title = NULL;
			item->link = NULL;
			item->description = NULL;

			if (!((KzMETA*)ctx->meta)->items)
				((KzMETA*)ctx->meta)->items = g_slist_append(((KzMETA*)(ctx->meta))->items, item);
			else
				g_slist_append(((KzMETA*)ctx->meta)->items,item);
		}
		break;
	 case 'c':
		if ((ctx->state & STATE_ROOT) && !strcmp(element_name, "channel"))
		{
			ctx->state |= STATE_CHANNEL;
			ctx->current_state = STATE_CHANNEL;
		}
		break;
	 case 't':
		if (!strcmp(element_name, "title"))
		{
			ctx->state |= STATE_TITLE;
			ctx->current_state = STATE_TITLE;
		}
		break;
	 case 'l':
		if (!strcmp(element_name, "link"))
		{
			ctx->state |= STATE_LINK;
			ctx->current_state = STATE_LINK;
		}
		break;
	 case 'd':
		if (!strcmp(element_name, "description"))
		{
			ctx->state |= STATE_DESC;
			ctx->current_state = STATE_DESC;
		}
		break;
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;

	ctx->state = ctx->state & ~(ctx->current_state);
	ctx->current_state = STATE_ERROR;
}

static void 
text_handler (GMarkupParseContext *context,
	      const gchar          *text,
	      gsize                 text_len,
	      gpointer              user_data,
	      GError              **error)
{
	ParseContext *ctx = user_data;

	if (ctx->current_state == STATE_ERROR)
		return;

	if (ctx->state & STATE_ITEM)
	{
		guint pos = g_slist_length(((KzMETA*)ctx->meta)->items);
		KzMETAItem *item = g_slist_nth_data(((KzMETA*)ctx->meta)->items, pos - 1);

		switch (ctx->current_state)
		{
		 case STATE_TITLE:
			item->title = g_strndup(text, text_len);
			break;
		 case STATE_LINK:
			item->link = g_strndup(text, text_len);
			break;
		 case STATE_DESC:
			item->description = remove_tag((gchar*)text, text_len);
			break;
		default:
			break;
		}
			
	}
	else if (ctx->state & STATE_CHANNEL)
	{
		switch (ctx->current_state)
		{
		 case STATE_TITLE:
			if (((KzRSS*)ctx->meta)->rss_title)
				g_free(((KzRSS*)ctx->meta)->rss_title);
			((KzRSS*)ctx->meta)->rss_title = g_strndup(text, text_len);
			break;
		 case STATE_LINK:
			if (((KzRSS*)ctx->meta)->rss_link)
				g_free(((KzRSS*)ctx->meta)->rss_link);
			((KzRSS*)ctx->meta)->rss_link = g_strndup(text, text_len);
			break;
		 case STATE_DESC:
			if (((KzRSS*)ctx->meta)->rss_description)
				g_free(((KzRSS*)ctx->meta)->rss_description);
			((KzRSS*)ctx->meta)->rss_description = g_strndup(text, text_len);
			break;
		default:
			break;
		}
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	text_handler,
	NULL,
	NULL
};

static void
kz_rss_parse_from_string (KzMETA *meta, gpointer user_data,
			  const gchar *buffer, guint length,
			  GError **error)
{
	gchar *utf8_body = NULL;
	/* gchar *body; */
	guint utf8_len;

	ParseContext ctx = { 0 };
	GMarkupParseContext *context;

	g_return_if_fail(buffer != NULL);

	/* check encoding and convert if need */
	utf8_body = kz_rss_convert_to_utf8((gchar*)buffer, length, &utf8_len);

	ctx.state = STATE_START;
	ctx.current_state = STATE_START;
	ctx.meta = meta;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);
	
	if (utf8_body)
	{
		if (g_markup_parse_context_parse(context, utf8_body, utf8_len, error))
		{
			g_markup_parse_context_end_parse(context, error);
		}
		g_free(utf8_body);
	}
	else
	{
		if (g_markup_parse_context_parse(context, buffer, length, error))
		{
			g_markup_parse_context_end_parse(context, error);
		}
	}

	g_markup_parse_context_free (context);
}

static gchar *
kz_rss_get_encoding (gchar *src)
{
	gchar *encoding = NULL;
	gchar **lines = NULL;
	gchar *prev_pos, *next_pos;
	
	g_return_val_if_fail(src, NULL);

	lines = g_strsplit(src, "\n", 1);
	prev_pos =  strstr(lines[0], "encoding");
	if (prev_pos)
	{
		while (*prev_pos != '"' && *prev_pos != '\0')
		{
			prev_pos++;
		}
		if (prev_pos != '\0')
		{
			prev_pos++;
			next_pos = strchr(prev_pos, '"');
			if (next_pos)
				encoding = g_strndup(prev_pos, 
						     next_pos - prev_pos);
			if (!strcasecmp(encoding, "UTF-8"))
			{
				g_free(encoding);
				encoding = NULL;
			}
		}
	}

	g_strfreev(lines);
	return encoding;
}

static gchar *
kz_rss_convert_to_utf8 (gchar *src, guint src_len, guint *dest_len)
{
	gchar *dest = NULL;
	gchar *encoding = NULL;
	guint read_len;

	encoding = kz_rss_get_encoding(src);

	if (encoding)
	{
		dest = g_convert(src, src_len, "UTF-8", encoding,
				 &read_len, dest_len, NULL);
		g_free(encoding);
	}

	return dest;
}
