/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-metatree.h"

#include "kazehakase.h"
#include "gobject-utils.h"
#include "intl.h"
#include "kz-meta.h"
#include "kz-icons.h"


#define NORMAL_COLOR  NULL
#define LOADING_COLOR "red"


enum
{
	COLUMN_TERMINATOR = -1,
	COLUMN_ICON,
	COLUMN_ICON_SIZE,
	COLUMN_TITLE,
	COLUMN_URI,
	COLUMN_META,
	COLUMN_META_ITEM,
	N_COLUMNS
};


/* Object class methods */
static void kz_meta_tree_class_init (KzMETATreeClass *klass);
static void kz_meta_tree_init       (KzMETATree      *metatree);
static void kz_meta_tree_dispose    (GObject        *obj);

/* KzMETATree private methods */
static void kz_meta_tree_refresh_all  (KzMETATree *metatree);
static void kz_meta_tree_refresh_meta (KzMETATree *metatree,
				       KzMETA     *meta);

static GtkTreeIter *find_node_from_meta     (GtkTreeStore *store,
					     KzMETA        *meta);
static void         remove_all_meta_signals (KzMETATree    *metatree);

/* signal handlers for TreeView */
static void     cb_cursor_changed           (GtkTreeView     *tree_view,
					     KzMETATree       *metatree);
static gboolean cb_tree_view_button_press   (GtkWidget        *widget,
					     GdkEventButton   *event,
					     KzMETATree        *metatree);
static gboolean cb_tree_view_button_release (GtkWidget       *widget,
					     GdkEventButton  *event,
					     KzMETATree       *metatree);
static gboolean cb_scroll_event             (GtkWidget       *widget,
					     GdkEventScroll  *event,
					     KzMETATree       *metatree);

/* signal handlers for KzMETA */
static void cb_meta_update_start     (KzMETA *meta, KzMETATree *metatree);
static void cb_meta_update_completed (KzMETA *meta, KzMETATree *metatree);

/* signal handler for KzMETAList */
static void cb_meta_list_updated (KzMETAList *metalist, KzMETATree *metatree);


static KzSidebarEntry kz_sidebar_meta_tree = 
{
	priority_hint: 0,
	label:         N_("RSS/RDF"),
	icon:          NULL,
	create:        kz_meta_tree_new,
};

static GtkVBoxClass *parent_class = NULL;


KzSidebarEntry *
kz_meta_tree_get_entry (gint idx)
{
	if (idx == 0)
		return &kz_sidebar_meta_tree;
	else
		return NULL;
}


KZ_OBJECT_GET_TYPE(kz_meta_tree, "KzMETATree", KzMETATree,
		   kz_meta_tree_class_init, kz_meta_tree_init,
		   GTK_TYPE_VBOX)


static void
kz_meta_tree_class_init (KzMETATreeClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent(klass);
	gobject_class = (GObjectClass *) klass;

	gobject_class->dispose = kz_meta_tree_dispose;
}


static void
kz_meta_tree_init (KzMETATree *metatree)
{
	GtkTreeStore *store;
	GtkWidget *widget, *scrwin;
	GtkTreeView *tree_view;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *column;

	/* create scrolled window and tree */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(metatree), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	store = gtk_tree_store_new(N_COLUMNS,
				   G_TYPE_STRING,
				   GTK_TYPE_ICON_SIZE,
				   G_TYPE_STRING,
				   G_TYPE_STRING,
				   KZ_TYPE_META,
				   G_TYPE_POINTER);

	widget = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	tree_view = GTK_TREE_VIEW(widget);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree_view), TRUE);
	gtk_container_add(GTK_CONTAINER(scrwin), GTK_WIDGET(tree_view));
	gtk_widget_show(GTK_WIDGET(tree_view));

	/* append a column */
	column = gtk_tree_view_column_new();
	cell = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(column, cell, FALSE);
	gtk_tree_view_column_add_attribute(column, cell,
					   "stock-id", COLUMN_ICON);
	gtk_tree_view_column_add_attribute(column, cell,
					   "stock-size", COLUMN_ICON_SIZE);
	cell = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, cell, TRUE);
	gtk_tree_view_column_add_attribute(column, cell,
					   "text", COLUMN_TITLE);
	gtk_tree_view_column_set_title(column, _("Title"));
	gtk_tree_view_append_column(tree_view, column);

	/* set signal handlers */
	g_signal_connect(G_OBJECT(tree_view), "cursor-changed",
			 G_CALLBACK(cb_cursor_changed), metatree);
	g_signal_connect(G_OBJECT(tree_view), "button-press-event",
			 G_CALLBACK(cb_tree_view_button_press), metatree);
	g_signal_connect(G_OBJECT(tree_view), "button-release-event",
			 G_CALLBACK(cb_tree_view_button_release), metatree);
	g_signal_connect(G_OBJECT(tree_view), "scroll-event",
			 G_CALLBACK(cb_scroll_event), metatree);

	/* initialize struct */
	metatree->sidebar   = NULL;
	metatree->tree_view = GTK_TREE_VIEW(tree_view);
	metatree->store     = store;
	metatree->metalist   = kz_meta_list_get_instance();

	/* set signal handler for KzMETAList */
	g_signal_connect(G_OBJECT(metatree->metalist), "updated",
			 G_CALLBACK(cb_meta_list_updated), metatree);

	/* set nodes */
	kz_meta_tree_refresh_all(metatree);

	gtk_tree_view_expand_all(metatree->tree_view);
}


static void
kz_meta_tree_dispose (GObject *obj)
{
	KzMETATree *metatree = KZ_META_TREE(obj);

	if (metatree->metalist)
	{
		remove_all_meta_signals(metatree);
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(metatree->metalist),
			 G_CALLBACK(cb_meta_list_updated), metatree);
		g_object_unref(metatree->metalist);
		metatree->metalist = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(obj);
}


GtkWidget *
kz_meta_tree_new (KzSidebar *sidebar)
{
	KzMETATree *metatree;

	metatree = KZ_META_TREE(g_object_new(KZ_TYPE_META_TREE, NULL));
	metatree->sidebar = sidebar;

	return GTK_WIDGET(metatree);
}


/*****************************************************************************
 *                                                                           *
 *                       KzMETATree private methods                           *
 *                                                                           *
 *****************************************************************************/
static void
kz_meta_tree_refresh_all (KzMETATree *metatree)
{
	GSList *node;

	g_return_if_fail (KZ_IS_META_TREE(metatree));

	remove_all_meta_signals(metatree);
	gtk_tree_store_clear(metatree->store);

	for (node = metatree->metalist->items; node; node = g_slist_next(node))
	{
		KzMETA *meta = node->data;
		GtkTreeIter iter;
		const gchar *stock;

		if (!meta) continue;

		if (meta->state == KZ_META_LOADING)
			stock = KZ_STOCK_RED;
		else
			stock = KZ_STOCK_GREEN;

		gtk_tree_store_append(metatree->store, &iter, NULL);
		/* FIXME!! increase refcount? or weekref? */
		gtk_tree_store_set(metatree->store, &iter,
				   COLUMN_ICON,      stock,
				   COLUMN_ICON_SIZE, GTK_ICON_SIZE_MENU,
				   COLUMN_TITLE,     meta->title,
				   COLUMN_URI,       meta->uri,
				   COLUMN_META,      meta,
				   COLUMN_TERMINATOR);

		g_signal_connect(G_OBJECT(meta), "update-start",
				 G_CALLBACK(cb_meta_update_start), metatree);
		g_signal_connect(G_OBJECT(meta), "update-completed",
				 G_CALLBACK(cb_meta_update_completed), metatree);

		kz_meta_tree_refresh_meta(metatree, meta);
	}
}


static void
kz_meta_tree_refresh_meta (KzMETATree *metatree, KzMETA *meta)
{
	GtkTreeIter *parent_iter, iter;
	GtkTreePath *treepath;
	GSList *node;
	gboolean expand;

	g_return_if_fail(metatree);
	g_return_if_fail(KZ_IS_META(meta));

	parent_iter = find_node_from_meta(metatree->store, meta);
	if (!parent_iter) return;

	treepath = gtk_tree_model_get_path(GTK_TREE_MODEL(metatree->store),
					   parent_iter);
	expand = gtk_tree_view_row_expanded(metatree->tree_view, treepath);

	/* clean children */
	while (gtk_tree_model_iter_children(GTK_TREE_MODEL(metatree->store),
					    &iter, parent_iter))
	{
		gtk_tree_store_remove(metatree->store, &iter);
	}

	/* append children */
	for (node = meta->items; node; node = g_slist_next(node))
	{
		KzMETAItem *item = node->data;

		if (!item) continue;

		gtk_tree_store_append(metatree->store, &iter, parent_iter);
		gtk_tree_store_set(metatree->store, &iter,
				   COLUMN_TITLE,    item->title,
				   COLUMN_URI,      item->link,
				   COLUMN_META,      meta,
				   COLUMN_META_ITEM, item,
				   COLUMN_TERMINATOR);
	}

	gtk_tree_view_expand_row(metatree->tree_view, treepath, TRUE);

	gtk_tree_path_free(treepath);
	gtk_tree_iter_free(parent_iter);
}


typedef struct _FindNodeMETA
{
	KzMETA       *meta;
	GtkTreeIter *iter;
} FindNodeMETA;


static gboolean
find_node_func (GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter,
		FindNodeMETA *data)
{
	KzMETA *meta = NULL;
	KzMETAItem *item = NULL;

	if (!iter) return FALSE;

	gtk_tree_model_get(model, iter,
			   COLUMN_META,      &meta,
			   COLUMN_META_ITEM, &item,
			   COLUMN_TERMINATOR);
	if (meta && data->meta && meta == data->meta && !item)
	{
		data->iter = gtk_tree_iter_copy(iter);
		return TRUE;
	}

	return FALSE;
}


static GtkTreeIter *
find_node_from_meta (GtkTreeStore *store, KzMETA *meta)
{
	FindNodeMETA data;

	data.iter = NULL;
	data.meta  = meta;

	gtk_tree_model_foreach(GTK_TREE_MODEL(store),
			       (GtkTreeModelForeachFunc) find_node_func,
			       &data);

	return data.iter;
}


static gboolean
remove_meta_signal_func (GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data)
{
	KzMETA *meta = NULL;
	KzMETATree *metatree;

	g_return_val_if_fail(KZ_IS_META_TREE(data), FALSE);
	metatree = KZ_META_TREE(data);

	g_return_val_if_fail(iter, FALSE);

	gtk_tree_model_get(model, iter,
			   COLUMN_META, &meta,
			   COLUMN_TERMINATOR);

	if (meta)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(meta),
			 G_CALLBACK(cb_meta_update_start),
			 metatree);
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(meta),
			 G_CALLBACK(cb_meta_update_completed),
			 metatree);
	}

	return FALSE;
}


static void
remove_all_meta_signals (KzMETATree *metatree)
{
	g_return_if_fail(KZ_IS_META_TREE(metatree));

	gtk_tree_model_foreach(GTK_TREE_MODEL(metatree->store),
			       remove_meta_signal_func,
			       metatree);
}


/*****************************************************************************
 *                                                                           *
 *                               Callbacks                                   *
 *                                                                           *
 *****************************************************************************/
static void
cb_cursor_changed(GtkTreeView *tree_view, KzMETATree *metatree)
{
}


static gboolean
cb_tree_view_button_press (GtkWidget *widget, GdkEventButton *event,
			   KzMETATree *metatree)
{
	GtkTreePath *treepath = NULL;
	GtkTreeIter iter;
	GtkTreeViewColumn *column;
	gboolean success;
	KzWindow *kz;
	KzMETA *meta = NULL;
	KzMETAItem *item = NULL;
	const gchar *link = NULL;

	g_return_val_if_fail(GTK_IS_TREE_VIEW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_META_TREE(metatree), FALSE);

	kz = metatree->sidebar->kz;

	/* get node  */
	success = gtk_tree_view_get_path_at_pos(metatree->tree_view,
						event->x, event->y,
						&treepath, &column,
						NULL, NULL);
	/* get URI */
	if (success)
	{
		gtk_tree_model_get_iter(GTK_TREE_MODEL(metatree->store),
					&iter, treepath);
		gtk_tree_model_get(GTK_TREE_MODEL(metatree->store), &iter,
				   COLUMN_META,      &meta,
				   COLUMN_META_ITEM, &item,
				   COLUMN_TERMINATOR);
		if (item)
			link = item->link;
/*		else if (meta)
			link = meta->meta_link; */
	}

	/* open URI or show popup menu */
	if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
	{
		if (link && *link)
			kz_window_load_url(kz, link);
	}
	else if (event->button == 2)
	{
		if (link && *link)
			kz_window_open_new_tab(kz, link);
	}
	else if (event->button == 3)
	{
		g_print("show popup menu.\n");
	}

	if (treepath)
		gtk_tree_path_free(treepath);

	return FALSE;
}


static gboolean
cb_tree_view_button_release (GtkWidget *widget, GdkEventButton *event,
			     KzMETATree *metatree)
{
	return FALSE;
}


static gboolean
cb_scroll_event (GtkWidget *widget, GdkEventScroll *event, KzMETATree *metatree)
{
	gboolean retval = FALSE;

	g_return_val_if_fail(GTK_IS_TREE_VIEW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_META_TREE(metatree), FALSE);

	switch (event->direction) {
	case GDK_SCROLL_UP:
		g_signal_emit_by_name(GTK_TREE_VIEW(widget), "move-cursor",
				      GTK_MOVEMENT_DISPLAY_LINES, -1, &retval);
		break;
	case GDK_SCROLL_DOWN:
		g_signal_emit_by_name(GTK_TREE_VIEW(widget), "move-cursor",
				      GTK_MOVEMENT_DISPLAY_LINES, 1, &retval);
		break;
	case GDK_SCROLL_LEFT:
	case GDK_SCROLL_RIGHT:
		break;
	default:
		g_warning ("Invalid scroll direction!");
		break;
	}

	return retval;
}


static void
cb_meta_update_start (KzMETA *meta, KzMETATree *metatree)
{
	GtkTreeIter *iter;

	g_return_if_fail(KZ_IS_META_TREE(metatree));
	g_return_if_fail(KZ_IS_META(meta));

	/* update icon */
	iter = find_node_from_meta(metatree->store, meta);
	if (!iter) return;

	gtk_tree_store_set(metatree->store, iter,
			   COLUMN_ICON, KZ_STOCK_RED,
			   COLUMN_TERMINATOR);

	gtk_tree_iter_free(iter);
}


static void
cb_meta_update_completed (KzMETA *meta, KzMETATree *metatree)
{
	GtkTreeIter *iter;

	g_return_if_fail(KZ_IS_META_TREE(metatree));
	g_return_if_fail(KZ_IS_META(meta));

	iter = find_node_from_meta(metatree->store, meta);
	if (!iter) return;

	/* update icon */
	gtk_tree_store_set(metatree->store, iter,
			   COLUMN_ICON, KZ_STOCK_GREEN,
			   COLUMN_TERMINATOR);

	gtk_tree_iter_free(iter);

	/* update children */
	kz_meta_tree_refresh_meta (metatree, meta);
}


static void
cb_meta_list_updated (KzMETAList *metalist, KzMETATree *metatree)
{
	g_return_if_fail(KZ_IS_META_LIST(metalist));
	g_return_if_fail(KZ_IS_META_TREE(metatree));

	kz_meta_tree_refresh_all(metatree);
}
