/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *  Copyright (C) 2004 Hidetaka Iwai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "migemo.h"
#if USE_MIGEMO

#include "kazehakase.h"

static gchar* migemo_strescape        (const gchar *source);
static gchar* migemo_strchomp         (gchar *string);

gboolean
migemo_init (void)
{
	gboolean ret;
	GSpawnFlags flags;
	gint argc;
	gchar **argv = NULL;
	gchar *migemo_command, *migemo_options;
	
	if (!migemo)
		migemo = g_new0(Migemo, 1);

	migemo_options = KZ_CONF_GET_STR("Global", "migemo_options");
	if (!migemo_options)
		migemo_options = g_strdup("-t egrep -d /usr/share/migemo/migemo-dict");


	migemo_command = g_strdup_printf ("kzmigemo %s", migemo_options);

	g_shell_parse_argv(migemo_command,
			   &argc,
			   &argv,
			   NULL);

	flags = G_SPAWN_SEARCH_PATH;
	ret = g_spawn_async_with_pipes(NULL,
				       argv,
				       NULL,
				       flags,
				       NULL,
				       NULL,
				       &migemo->pid,
				       &migemo->input,
				       &migemo->output,
				       &migemo->error,
				       NULL);	
	g_strfreev(argv);
	g_free(migemo_options);
	g_free(migemo_command);
	
	if (!ret)
	{
		g_free(migemo);
		migemo = NULL;
	}

	return ret;
}

void
migemo_exit (void)
{
#if (GLIB_MAJOR_VERSION == 2) && (GLIB_MINOR_VERSION > 3)
	g_spawn_close_pid(migemo->pid);
#endif	
	g_free(migemo);
}

gchar *
migemo_get_word(const gchar *body, const gchar *text)
{
	GIOChannel *in;
	GIOChannel *out;
	GError *e = NULL;
	gchar *body_escaped = NULL, *write_buf = NULL, *read_buf = NULL;
	gsize bytes;
	
	g_return_val_if_fail(migemo, NULL);
	g_return_val_if_fail(body, NULL);
	g_return_val_if_fail(text, NULL);
	g_return_val_if_fail(g_utf8_validate(body, -1, NULL), NULL);
	
	in  = g_io_channel_unix_new(migemo->input);
	out = g_io_channel_unix_new(migemo->output);
	
	g_io_channel_set_encoding(in, NULL, NULL);
	g_io_channel_set_encoding(out, NULL, NULL);
	

	body_escaped = migemo_strescape (body);
	
	if(body_escaped)
	{
		write_buf = g_strconcat (body_escaped, "\t", text, "\n", NULL);
		if(write_buf)
		{
			g_io_channel_write_chars(in,
						 write_buf,
						 strlen(write_buf),
						 &bytes,
						 NULL);
			g_io_channel_flush(in, NULL);

			g_io_channel_read_line(out,
					       &read_buf,
					       &bytes,
					       NULL,
					       &e);
			if (e)
			{
				g_warning("%s", e->message);
				g_error_free(e);
				e = NULL;
			}
			
			if (read_buf)
			{
				(void)migemo_strchomp(read_buf);
			}

			g_free(write_buf);
		}

		g_free(body_escaped);
	}
	g_io_channel_unref(in);
	g_io_channel_unref(out);

	return read_buf;
}

static gchar *
migemo_strescape (const gchar *source)
{
	const gchar *p;
	gchar *dest;
	gchar *q;

	g_return_val_if_fail (source != NULL, NULL);

	p = source;
	q = dest = g_malloc (strlen (source) + 1);

	while (*p)
	{
		switch (*p)
		{
		case '\n':
			*q++ = ' ';
			break;
		case '\r':
			*q++ = ' ';
			break;
		case '\t':
			*q++ = ' ';
			break;
		default:
			*q++ = *p;
			break;
		}
		p++;
	}
	*q = 0;
	return dest;
}

static gchar*
migemo_strchomp (gchar *string)
{
	gsize len;

	g_return_val_if_fail (string != NULL, NULL);

	len = strlen (string);
	while (len--)
	{
		if ((guchar)string[len] == '\n')
			string[len] = '\0';
		else
			break;
	}

	return string;
}

#endif /* USE_MIGEMO */
