/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include "config.h"
#include "kz-home.h"
#include "gobject-utils.h"

static void kz_home_button_class_init (KzHomeButtonClass *class);
static void kz_home_button_init       (KzHomeButton *kz);
static void kz_home_button_destroy    (GtkObject *object);

static GtkWidget *kz_home_menu_new   (KzHomeButton *kzhome);
static void kz_home_file_load        (KzHomeButton *kzhome);
static void kz_home_file_save        (KzHomeButton *kzhome);
static void kz_home_menu_update      (KzHomeButton *kzhome);

/* submenu */
static void kz_home_menu_submenu_popup(KzHomeList *kzhomelist, GdkEventButton *event);
static void kz_home_menu_submenu_delete(GtkMenuItem *menuite, KzHomeList *kzhomelist);

/* callbacks */
static void kz_home_button_press_cb  (GtkWidget *widget, GdkEventButton *event,
				      KzHomeButton *kzhome);
static void kz_home_menu_add_press_cb(GtkWidget *widget, GdkEventButton *event,
				      KzHomeButton *kzhome);
static void kz_home_menu_press_cb    (GtkWidget *widget, GdkEventButton *event,
				      KzHomeList *kzhomelist);

static GtkButtonClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_home_button, "KzHomeButton", KzHomeButton,
		   kz_home_button_class_init, kz_home_button_init,
		   GTK_TYPE_BUTTON)

typedef enum {
	STATE_START   = (1 << 0),
	STATE_HOME    = (1 << 1),
	STATE_SITE    = (1 << 2),
	STATE_END     = (1 << 7),
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;

	KzHomeButton *kzhome;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	ParseContext *ctx = user_data;
	int i;

	switch (element_name[0])
	{
	 case 'h':
		if (ctx->state == STATE_START && !strcmp(element_name, "home"))
		{
			ctx->state = STATE_HOME;
		}
		break;
	 case 's':
		if ((ctx->state & STATE_HOME) && !strcmp(element_name, "site")) 
		{
			KzHomeList *list = NULL;

			list = g_new0(KzHomeList, 1);
			ctx->state = STATE_SITE;

			list->kzhome = ctx->kzhome;
			for (i = 0; attribute_names[i] != NULL; i++)
			{
				if (!strcmp(attribute_names[i], "uri"))
					list->uri = g_strdup(attribute_values[i]);
				else if (!strcmp(attribute_names[i], "title"))
					list->title = g_strdup(attribute_values[i]);
			}
			g_slist_append(ctx->kzhome->home_list, list);
		}
		break;
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;
	
	switch (ctx->state)
	{
	 case STATE_START:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	 case STATE_HOME:
		ctx->state = STATE_END;
		break;
	 case STATE_SITE:
		ctx->state = STATE_HOME;
		break;
	 case STATE_END:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	NULL,
	NULL,
	NULL
};

static gboolean
kz_home_parse_from_string (KzHomeButton *kzhome, gpointer user_data,
			   const gchar *buffer, guint length,
			   GError **error)
{
	ParseContext ctx = { 0 };
	GMarkupParseContext *context;
	gboolean res = TRUE;

	g_return_val_if_fail(buffer != NULL, FALSE);

	ctx.state = STATE_START;
	ctx.kzhome = kzhome;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);

	if (g_markup_parse_context_parse(context, buffer, length, error))
	{
		if (!g_markup_parse_context_end_parse(context, error))
			res = FALSE;
	}
	else
		res = FALSE;

	g_markup_parse_context_free (context);

	return res;
}

static gboolean
kz_home_parse_from_file (KzHomeButton *kzhome, gpointer user_data,
			 const gchar *filename, GError **error)
{
	gchar *buffer;
	gint length;
	gboolean res;

	if (!g_file_get_contents (filename, &buffer, &length, error))
		return FALSE;

	res = kz_home_parse_from_string(kzhome, user_data, buffer, length, error);
	g_free(buffer);

	return res;
}

static void
kz_home_button_class_init (KzHomeButtonClass *class)
{
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;

	/* GtkObject signals */
	object_class->destroy = kz_home_button_destroy;
}

static void
kz_home_button_init (KzHomeButton *kzhome)
{
	/* widgets */
	kzhome->kz = NULL;
	kzhome->PopupMenu = NULL;
	
	/* lairs list  */
	kzhome->home_list = NULL;
}

/* create new home button */
GtkWidget *
kz_home_button_new (KzWindow *kz)
{
	KzHomeButton *kzhome = KZ_HOME_BUTTON(g_object_new(kz_home_button_get_type(), NULL));
	GtkIconSize isize = gtk_toolbar_get_icon_size(GTK_TOOLBAR(kz->kztoolBar));
	
	kzhome->kz = kz;
	kzhome->home_list = g_slist_alloc();
	
	/* create button */
	gtk_container_add(GTK_CONTAINER(kzhome),
			  gtk_image_new_from_stock(GTK_STOCK_HOME,isize));
	gtk_button_set_relief(GTK_BUTTON(kzhome), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(kzhome), "button_press_event",
			 G_CALLBACK(kz_home_button_press_cb), kzhome);
	/* load lairs list file */
	kz_home_file_load(kzhome);
	
	/* create lairs menu */
	kzhome->PopupMenu = kz_home_menu_new(kzhome);
	gtk_widget_show_all(GTK_WIDGET(kzhome));
	
	return GTK_WIDGET(kzhome);
}

static void
kz_home_button_destroy (GtkObject *object)
{
	g_slist_free(KZ_HOME_BUTTON(object)->home_list);
	KZ_HOME_BUTTON(object)->home_list = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void 
kz_home_file_load (KzHomeButton *kzhome)
{
        gchar *homefile;
	
        homefile = g_strdup_printf("%s/.%s/home.xml", g_get_home_dir(), PACKAGE);
	kz_home_parse_from_file(kzhome, NULL, homefile, NULL);
	g_free(homefile);
}

static void
kz_home_file_save (KzHomeButton *kzhome)
{
	gchar *homefile;
	GString *xml;
	FILE *fp;
	
	KzHomeList *kzhomelist;
	GSList *list;
	guint list_num, n;
	
	xml = g_string_new(NULL);
	
	g_string_append(xml, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	g_string_append(xml, "<home>\n");
	
	/* add sites */
	list = kzhome->home_list;
	list_num = g_slist_length(kzhome->home_list);
	for (n = 1; n < list_num; n++)
	{
		kzhomelist = (KzHomeList*) g_slist_nth_data(list, n);
		g_string_append_printf(xml, "  <site uri=\"%s\" title=\"%s\" />\n",
				       g_markup_escape_text(kzhomelist->uri,-1),
				       kzhomelist->title);
	}
	
	xml = g_string_append(xml, "</home>");

        homefile = g_strdup_printf("%s/.%s/home.xml", g_get_home_dir(), PACKAGE);
	fp = fopen(homefile, "w");
	if (fp != NULL)
	{
		fwrite(xml->str, xml->len, 1, fp);
		fclose(fp);
	}
	g_free(homefile);

	g_string_free(xml, TRUE);
}

/* create lairs menu */
static GtkWidget *
kz_home_menu_new(KzHomeButton *kzhome)
{
	GtkWidget *menu, *menu_item;
	GSList *list;
	KzHomeList *kzhomelist;
	guint list_num, n;
	
	menu = gtk_menu_new();	
	/* Add lairs */
	menu_item = gtk_menu_item_new_with_label("Add lairs");
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_home_menu_add_press_cb),
			 kzhome);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_separator_menu_item_new());
	
	/* create lairs menu */
	list = kzhome->home_list;
	list_num = g_slist_length(kzhome->home_list);
	for (n = 1; n < list_num; n++)
	{
		kzhomelist = (KzHomeList*) g_slist_nth_data(list, n);

		menu_item = gtk_menu_item_new_with_label(kzhomelist->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu),
				      menu_item);
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(kz_home_menu_press_cb),
				 kzhomelist);
	}
	gtk_widget_show_all(menu);
	return menu;
}

static void
kz_home_menu_submenu_popup(KzHomeList *kzhomelist, GdkEventButton *event)
{
	GtkWidget *submenu, *menu_item;
	
	submenu = gtk_menu_new();

	menu_item = gtk_menu_item_new_with_label("Remove from list");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "activate",
			 G_CALLBACK(kz_home_menu_submenu_delete),
			 kzhomelist);
	gtk_widget_show_all(submenu);
	gtk_menu_popup(GTK_MENU(submenu), NULL, NULL, NULL,
		       NULL, event->button, event->time);
}

static void
kz_home_menu_submenu_delete(GtkMenuItem *menuitem, KzHomeList *kzhomelist)
{
	g_slist_remove(kzhomelist->kzhome->home_list, kzhomelist);
	
	/* save home.xml */
	kz_home_file_save(kzhomelist->kzhome);
	
	kz_home_menu_update(kzhomelist->kzhome);
	
	g_free(kzhomelist);
}

static void
kz_home_menu_update(KzHomeButton *kzhome)
{
	/* update lairs menu */
	if (kzhome->PopupMenu != NULL)
	{	
		gtk_widget_destroy(kzhome->PopupMenu);
		g_free(kzhome->PopupMenu);
	}
	kzhome->PopupMenu = kz_home_menu_new(kzhome);
}

static void
kz_home_button_press_cb(GtkWidget *widget, GdkEventButton *event,
			KzHomeButton *kzhome)
{
	KzWindow *kz;

	kz = kzhome->kz;
	
	switch ( event->button ) {
	 case 1: /* open in current tab */
		if (g_slist_length(kzhome->home_list) > 1)
			kz_window_load_url(kz, ((KzHomeList *)g_slist_nth_data(kzhome->home_list, 1))->uri);
		break;
	 case 2: /* open in new tab */
		if (g_slist_length(kzhome->home_list) > 1)
			kz_window_open_new_tab(kz, ((KzHomeList *)g_slist_nth_data(kzhome->home_list, 1))->uri);
		break;
	 case 3: /* popup secret menu */
		gtk_menu_popup(GTK_MENU(kzhome->PopupMenu), NULL, NULL, NULL,
			       NULL, event->button, event->time);
		break;
	}
}

static void
kz_home_menu_add_press_cb(GtkWidget *widget, GdkEventButton *event,
			  KzHomeButton *kzhome)
{
	GSList *list;
	
	KzWindow *kz;
	KzHomeList *kzhomelist = NULL;

	kz = kzhome->kz;
	
	kzhomelist = g_new0(KzHomeList, 1);
	
	kzhomelist->kzhome = kzhome;

	kzhomelist->title = kz_window_get_title(kz);
	kzhomelist->uri   = kz_window_get_uri(kz);

	/* append new site */
	list = g_slist_last(kzhome->home_list);
	g_slist_append(list, kzhomelist);
	
	/* save home.xml */
	kz_home_file_save(kzhome);
	
	kz_home_menu_update(kzhome);
}

static void
kz_home_menu_press_cb(GtkWidget *widget, GdkEventButton *event,
		      KzHomeList *kzhomelist)
{
	KzWindow *kz;

	kz = kzhomelist->kzhome->kz;
	
	if (kzhomelist->uri)
	{
		switch ( event->button ) {
		 case 1:
			kz_window_load_url(kz, kzhomelist->uri);
			break;
		 case 2:
			kz_window_open_new_tab(kz, kzhomelist->uri);
			break;
		 case 3: /* popup menu(remove etc.) */
			kz_home_menu_submenu_popup(kzhomelist, event);
			break;
		}
	}
}
