/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark.h 790 2004-03-20 09:40:30Z makeinu $
 */

#ifndef __KZ_BOOKMARK_H__
#define __KZ_BOOKMARK_H__

#include <glib-object.h>

G_BEGIN_DECLS

#define KZ_TYPE_BOOKMARK		(kz_bookmark_get_type ())
#define KZ_BOOKMARK(obj)		(G_TYPE_CHECK_INSTANCE_CAST ((obj), KZ_TYPE_BOOKMARK, KzBookmark))
#define KZ_BOOKMARK_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST ((klass),  KZ_TYPE_BOOKMARK, KzBookmarkClass))
#define KZ_IS_BOOKMARK(obj)		(G_TYPE_CHECK_INSTANCE_TYPE ((obj), KZ_TYPE_BOOKMARK))
#define KZ_IS_BOOKMARK_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass),  KZ_TYPE_BOOKMARK))
#define KZ_BOOKMARK_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS ((obj),  KZ_TYPE_BOOKMARK, KzBookmarkClass))

#define KZ_TYPE_BOOKMARK_TYPE		(kz_bookmark_type_get_type())

typedef struct _KzBookmark         KzBookmark;
typedef struct _KzBookmarkClass    KzBookmarkClass;
typedef struct _KzBookmarkFileType KzBookmarkFileType;

typedef enum {
	KZ_BOOKMARK_NORMAL,
	KZ_BOOKMARK_FOLDER,
	KZ_BOOKMARK_PURE_FOLDER,
	KZ_BOOKMARK_SEPARATOR,
	KZ_BOOKMARK_N_TYPES
} KzBookmarkType;

typedef enum {
	KZ_BOOKMARK_EDITABLE_FLAG    = 1 << 0,  /* not used yet */
	KZ_BOOKMARK_FOLDED_FLAG      = 1 << 1,
	KZ_BOOKMARK_FILE_FLAG        = 1 << 2,
	KZ_BOOKMARK_SMART_FLAG       = 1 << 3
} KzBookmarkFlag;

#define KZ_BOOKMARK_SORT_LAST_MODIFIED "last-modified"

typedef enum {
	KZ_BOOKMARK_STATE_NORMAL,
	KZ_BOOKMARK_STATE_LOADING,
	KZ_BOOKMARK_STATE_SAVING
} KzBookmarkState;

struct _KzBookmark
{
	GObject         parent;
	KzBookmarkType  type;
	KzBookmarkFlag  flags;
};

struct _KzBookmarkClass
{
	GObjectClass parent_class;

	/* -- signals -- */
	void (*insert_child)       (KzBookmark  *bookmark,
				    KzBookmark  *child,
				    KzBookmark  *sibling);
	void (*remove_child)       (KzBookmark  *bookmark,
				    KzBookmark  *child);
	void (*move_child)         (KzBookmark  *bookmark,
				    KzBookmark  *parent,
				    KzBookmark  *sibling);
	void (*children_reordered) (KzBookmark  *boomark);

	void (*load_start)         (KzBookmark  *bookmark);
	void (*load_completed)     (KzBookmark  *bookmark);
	void (*save_start)         (KzBookmark  *bookmark);
	void (*save_completed)     (KzBookmark  *bookmark);

	void (*error)              (KzBookmark  *bookmark,
				    const gchar *message);
};

struct _KzBookmarkFileType
{
	gint         priority_hint;
	const gchar *file_type;
	gboolean   (*is_supported) (KzBookmark  *bookmark,
				    const gchar *buffer);
	void       (*init)         (KzBookmark  *bookmark);
	gboolean   (*from_string)  (KzBookmark  *bookmark,
				    const gchar *buffer,
				    guint        length,
				    GError     **error);
	gchar     *(*to_string)    (KzBookmark  *bookmark);
};


/* GType */
GType		kz_bookmark_type_get_type	(void);
GType		kz_bookmark_get_type		(void);

/* boomark item creating functions */
KzBookmark     *kz_bookmark_new			(void);
KzBookmark     *kz_bookmark_new_with_attrs	(const gchar *title,
						 const gchar *uri,
						 const gchar *description);
KzBookmark     *kz_bookmark_pure_folder_new	(void);
KzBookmark     *kz_bookmark_separator_new	(void);
KzBookmark     *kz_bookmark_file_new		(const gchar *location,
						 const gchar *title,
						 const gchar *file_type);

KzBookmark     *kz_bookmark_file_create_new	(const gchar *location,
						 const gchar *title,
						 const gchar *file_type);

/* set/get properties */
/*
 * "title" means user defined title.
 * "document_title" is the original document title.
 * "location" means location of bookmark file itself.
 */
const gchar    *kz_bookmark_get_link		(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_title		(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_document_title	(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_description	(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_location	(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_file_type	(KzBookmark  *bookmark);
guint		kz_bookmark_get_last_modified	(KzBookmark  *bookmark);
guint		kz_bookmark_get_interval	(KzBookmark  *bookmark);
gboolean        kz_bookmark_is_file             (KzBookmark  *bookmark);
gboolean        kz_bookmark_is_editable         (KzBookmark  *bookmark);
gboolean        kz_bookmark_file_is_editable    (KzBookmark  *bookmark_file);

void		kz_bookmark_set_link		(KzBookmark  *bookmark,
						 const gchar *uri);
void		kz_bookmark_set_title		(KzBookmark  *bookmark,
						 const gchar *title);
void		kz_bookmark_set_document_title	(KzBookmark  *bookmark,
						 const gchar *doc_title);
void		kz_bookmark_set_description	(KzBookmark  *bookmark,
						 const gchar *description);
void		kz_bookmark_set_location	(KzBookmark  *bookmark,
						 const gchar *location);
void		kz_bookmark_set_file_type	(KzBookmark  *bookmark,
						 const gchar *file_type);
void		kz_bookmark_set_last_modified	(KzBookmark  *bookmark,
						 guint        time);
void		kz_bookmark_set_interval	(KzBookmark  *bookmark,
						 guint        interval);

/* separator interface */
gboolean	kz_bookmark_is_separator	(KzBookmark  *bookmark);

/* folder interface */
gboolean	kz_bookmark_is_folder		(KzBookmark  *bookmark);
gboolean	kz_bookmark_is_pure_folder	(KzBookmark  *bookmark);
gboolean	kz_bookmark_get_folded		(KzBookmark  *bookmark);
void		kz_bookmark_set_folded		(KzBookmark  *bookmark,
						 gboolean     folded);
void		kz_bookmark_insert_before	(KzBookmark  *bookmark,
						 KzBookmark  *child,
						 KzBookmark  *sibling);
void		kz_bookmark_append		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_prepend		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_remove		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_remove_all		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_get_parent		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_get_parent_file	(KzBookmark  *bookmark);
GList	       *kz_bookmark_get_children	(KzBookmark  *bookmark);
gboolean        kz_bookmark_has_children        (KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_next		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_prev		(KzBookmark  *bookmark);
void		kz_bookmark_sort		(KzBookmark  *bookmark,
						 const gchar *type);
void		kz_bookmark_regist_sort_func	(const gchar *type,
						 GCompareFunc *func);

/* load/save interface */
/* async */
gboolean	kz_bookmark_load_start		 (KzBookmark *bookmark);
gboolean	kz_bookmark_load_in_memory_start (KzBookmark *bookmark);
void		kz_bookmark_load_stop		 (KzBookmark *bookmark);
gboolean	kz_bookmark_save_start		 (KzBookmark *bookmark);
void		kz_bookmark_save_stop		 (KzBookmark *bookmark);
/* sync */
void		kz_bookmark_load		 (KzBookmark *bookmark);
void		kz_bookmark_save		 (KzBookmark *bookmark);

KzBookmarkState kz_bookmark_get_state		 (KzBookmark *bookmark);


G_END_DECLS

#endif /* __KZ_BOOKMARK_H__ */
