/* -*- mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-sidebar.c 1932 2005-02-14 05:45:06Z ikezoe $
 */

#include "kz-sidebar.h"

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "gobject-utils.h"
#include "kz-bookmarks-sidebar.h"
#include "kz-downloader-sidebar.h"
#include "kz-tabtree.h"
#include "kz-thumbnails-sidebar.h"


#define ADD_ENTRIES(func) 						\
{									\
	KzSidebarEntry *entry;						\
	gint i;								\
	for (i = 0; (entry = func(i)); i++)				\
	{								\
		n_sidebar_entries++;					\
		sidebar_entries = g_realloc(sidebar_entries,		\
					    sizeof(KzSidebarEntry)	\
					    * n_sidebar_entries);	\
		sidebar_entries[n_sidebar_entries - 1] = entry;		\
	}								\
}


enum {
	PROP_0,
	PROP_KZ_WINDOW
};


static void     kz_sidebar_class_init     (KzSidebarClass *klass);
static void     kz_sidebar_init           (KzSidebar *sidebar);
static void     kz_sidebar_dispose        (GObject       *object);
static void     kz_sidebar_set_property   (GObject       *object,
					   guint          prop_id,
					   const GValue  *value,
					   GParamSpec    *pspec);
static void     kz_sidebar_get_property   (GObject       *object,
					   guint          prop_id,
					   GValue        *value,
					   GParamSpec    *pspec);

static gboolean kz_sidebar_change_content (KzSidebar *sidebar,
					   KzSidebarEntry *entry);

static void     cb_option_menu_changed    (GtkComboBox   *option_menu,
					   KzSidebar     *sidebar);
static void     cb_close_button           (GtkWidget     *widget,
					   KzSidebar     *sidebar);

static GtkVBoxClass *parent_class = NULL;

static KzSidebarEntry **sidebar_entries   = NULL;
static gint             n_sidebar_entries = 0;


KZ_OBJECT_GET_TYPE(kz_sidebar, "KzSidebar", KzSidebar,
		   kz_sidebar_class_init, kz_sidebar_init,
		   GTK_TYPE_VBOX)

static void
kz_sidebar_class_init (KzSidebarClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent(klass);
	gobject_class = (GObjectClass *) klass;

	gobject_class->dispose      = kz_sidebar_dispose;
	gobject_class->set_property = kz_sidebar_set_property;
	gobject_class->get_property = kz_sidebar_get_property;

	g_object_class_install_property
		(gobject_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The parent kazehakase window"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));

	ADD_ENTRIES(kz_bookmarks_sidebar_get_entry);
	ADD_ENTRIES(kz_tab_tree_get_entry);
	ADD_ENTRIES(kz_downloader_sidebar_get_entry);
	ADD_ENTRIES(kz_thumbnails_sidebar_get_entry);
}

static void
kz_sidebar_init (KzSidebar *sidebar)
{
	GtkWidget *hbox, *option_menu, *close_button, *image;
	gint i;

	/* top hbox */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(sidebar), hbox, FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(hbox));

	/* option menu */
	option_menu = gtk_combo_box_new_text();
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(option_menu));

	sidebar->content = NULL;
	sidebar->current_entry = NULL;

	for (i = 0; i < n_sidebar_entries; i++)
	{
		gtk_combo_box_append_text(GTK_COMBO_BOX(option_menu),
					  _(sidebar_entries[i]->label));
	}
	g_signal_connect (option_menu, "changed",
			  G_CALLBACK (cb_option_menu_changed), sidebar);

	/* close button */
	close_button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(close_button), GTK_RELIEF_NONE);
	gtk_box_pack_end(GTK_BOX(hbox), close_button, FALSE, FALSE, 0);
	gtk_widget_show(close_button);

	image = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
					 GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(close_button), image);
	gtk_widget_show(image);

	g_signal_connect(close_button, "clicked",
			 G_CALLBACK(cb_close_button), sidebar);

	sidebar->kz            = NULL;
	sidebar->hbox          = hbox;
	sidebar->option_menu   = option_menu;
	sidebar->close_button  = close_button;
	sidebar->content       = NULL;
	sidebar->current_entry = NULL;
}


static void
kz_sidebar_dispose (GObject *obj)
{
	KzSidebar *sidebar = KZ_SIDEBAR(obj);

	if (sidebar->kz)
		g_object_unref(sidebar->kz);
	sidebar->kz = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(obj);
}


static void
kz_sidebar_set_property (GObject         *object,
			 guint            prop_id,
			 const GValue    *value,
			 GParamSpec      *pspec)
{
	KzSidebar *sidebar = KZ_SIDEBAR(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		sidebar->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_sidebar_get_property (GObject         *object,
			 guint            prop_id,
			 GValue          *value,
			 GParamSpec      *pspec)
{
	KzSidebar *sidebar = KZ_SIDEBAR(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, sidebar->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


GtkWidget *
kz_sidebar_new (KzWindow *kz)
{
	KzSidebar *sidebar;

	sidebar = g_object_new(KZ_TYPE_SIDEBAR,
			       "kz-window", kz,
			       NULL);

	kz_sidebar_set_current(sidebar, sidebar_entries[0]->label);

	return GTK_WIDGET(sidebar);
}

const gchar *
kz_sidebar_get_current (KzSidebar *sidebar)
{
	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), NULL);
	g_return_val_if_fail(sidebar->current_entry, NULL);

	return sidebar->current_entry->label;
}


static gboolean
kz_sidebar_change_content (KzSidebar *sidebar, KzSidebarEntry *entry)
{
	GtkWidget *widget;

	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), FALSE);
	g_return_val_if_fail(entry, FALSE);
	g_return_val_if_fail(entry->create, FALSE);

	widget = entry->create(sidebar);
	if (!GTK_IS_WIDGET(widget))
		return FALSE;

	if (sidebar->content)
		gtk_widget_destroy(sidebar->content);

	sidebar->content = widget;

	gtk_box_pack_start(GTK_BOX(sidebar), sidebar->content,
			   TRUE, TRUE, 0);
	gtk_widget_show(sidebar->content);

	sidebar->current_entry = entry;

	return TRUE;
}


gboolean
kz_sidebar_set_current (KzSidebar *sidebar, const gchar *label)
{
	gint i, num = n_sidebar_entries;

	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), FALSE);
	g_return_val_if_fail(label && *label, FALSE);

	for (i = 0; i < num; i++)
	{
		gboolean ok;

		if (strcmp(sidebar_entries[i]->label, label)) continue;

		ok = kz_sidebar_change_content(sidebar, sidebar_entries[i]);
		if (!ok) continue;

		/* set option menu */
		g_signal_handlers_block_by_func(sidebar->option_menu,
						G_CALLBACK(cb_option_menu_changed),
						sidebar);
		gtk_combo_box_set_active(GTK_COMBO_BOX(sidebar->option_menu), i);
		g_signal_handlers_unblock_by_func(sidebar->option_menu,
						  G_CALLBACK(cb_option_menu_changed),
						  sidebar);

		return TRUE;
	}

	return FALSE;
}

static void
cb_option_menu_changed (GtkComboBox *option_menu, KzSidebar *sidebar)
{
	gint num;

	g_return_if_fail (KZ_IS_SIDEBAR(sidebar));

	num = gtk_combo_box_get_active(option_menu);
	g_return_if_fail(num >= 0 && num < n_sidebar_entries);

	kz_sidebar_change_content(sidebar, sidebar_entries[num]);
}

static void
cb_close_button (GtkWidget *widget, KzSidebar *sidebar)
{
	g_return_if_fail(KZ_IS_SIDEBAR(sidebar));
	gtk_widget_hide(GTK_WIDGET(sidebar));
}
