/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n.h>
#include <gmodule.h>

#include "kazehakase.h"
#include "glib-utils.h"
#include "kz-search.h"

typedef struct _KzSearchModule KzSearchModule;
typedef struct _KzSearchModuleClass KzSearchModuleClass;

struct _KzSearchModule
{
	GTypeModule  object;
	GModule *library;

	void             (*init)     (GTypeModule    *module);
	void             (*exit)     (void);
	KzSearch        *(*create)   (void);
	gchar *path;
};

struct _KzSearchModuleClass
{
	GTypeModuleClass parent_class;
};

GType kz_search_module_get_type (void) G_GNUC_CONST;

G_DEFINE_TYPE (KzSearchModule, kz_search_module, G_TYPE_TYPE_MODULE)

#define KZ_TYPE_SEARCH_MODULE      (kz_search_module_get_type ())
#define KZ_SEARCH_MODULE(module)   (G_TYPE_CHECK_INSTANCE_CAST ((module), KZ_TYPE_SEARCH_MODULE, KzSearchModule))

static GSList *loaded_search;

static gboolean
kz_search_module_load (GTypeModule *module)
{
	KzSearchModule *s_module = KZ_SEARCH_MODULE(module); 
	gpointer initp, exitp, createp;

	s_module->library = g_module_open(s_module->path, G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);
	if (!s_module->library)
	{
		g_warning(g_module_error());
		return FALSE;
	}

	/* exsact symbols from the lib */
	if (!g_module_symbol(s_module->library, "kz_search_module_init", &initp) ||
	    !g_module_symbol(s_module->library, "kz_search_module_exit", &exitp) ||
	    !g_module_symbol(s_module->library, "kz_search_module_create", &createp))
	{
		g_warning(g_module_error());
		g_module_close(s_module->library);

		return FALSE;
	}

	s_module->init   = initp;
	s_module->exit   = exitp;
	s_module->create = createp;

	s_module->init(module);

	return TRUE;
}
 
static void
kz_search_module_unload (GTypeModule *module)
{
	KzSearchModule *s_module = KZ_SEARCH_MODULE(module);

	s_module->exit();

	g_module_close(s_module->library);
	s_module->library = NULL;

	s_module->init   = NULL;
	s_module->exit   = NULL;
	s_module->create = NULL;
}

static void
kz_search_module_finalize (GObject *object)
{
	KzSearchModule *module = KZ_SEARCH_MODULE(object);

	g_free(module->path);

	G_OBJECT_CLASS(kz_search_module_parent_class)->finalize(object);
}
static void
kz_search_module_class_init (KzSearchModuleClass *klass)
{
	GTypeModuleClass *module_class = G_TYPE_MODULE_CLASS(klass);
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	module_class->load   = kz_search_module_load;
	module_class->unload = kz_search_module_unload;

	gobject_class->finalize = kz_search_module_finalize;
} 

static void
kz_search_module_init (KzSearchModule *module)
{
}

static KzSearch *
kz_search_module_create (KzSearchModule *module)
{
	KzSearch *search = NULL;
	if (g_type_module_use(G_TYPE_MODULE(module)))
	{
		search = module->create();
		g_type_module_unuse(G_TYPE_MODULE(module));
		return search;
	}
	return NULL;
}
 
KzSearch *
kz_search_new (const gchar *name)
{
	GSList *l;
	gchar *module_path;
	KzSearchModule *module;
	KzSearch *search = NULL;

	for (l = loaded_search; l; l = l->next)
	{
		module = l->data;

		if (strcmp(G_TYPE_MODULE (module)->name, name) == 0)
			return kz_search_module_create(module);
	}

	if (g_module_supported())
	{
		module_path = g_module_build_path(KZ_SEARCH_MODULEDIR, name);

		if (module_path)
		{
			module = g_object_new(KZ_TYPE_SEARCH_MODULE, NULL);

			g_type_module_set_name(G_TYPE_MODULE(module), name);
			module->path = g_strdup(module_path);

			loaded_search = g_slist_prepend(loaded_search,
					module);

			search = kz_search_module_create(module);
			g_free(module_path);
		}
	}
 
	return search;
}

typedef struct _KzSearchPrivate	KzSearchPrivate;
struct _KzSearchPrivate
{
};

#define KZ_SEARCH_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_SEARCH, KzSearchPrivate))

G_DEFINE_ABSTRACT_TYPE(KzSearch, kz_search, G_TYPE_OBJECT)

static void
kz_search_class_init (KzSearchClass *klass)
{
	GObjectClass *object_class;

	kz_search_parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;

	klass->get_search_result_html     = NULL;
	klass->get_search_result_bookmark = NULL;
	klass->optimize_index             = NULL;
	klass->make_index                 = NULL;
	klass->exist_index_dir            = NULL;

	g_type_class_add_private (object_class, sizeof(KzSearchPrivate));
}


static void
kz_search_init (KzSearch *search)
{
}

gchar *
kz_search_get_search_result_html (KzSearch *search, const gchar *text)
{
	KzSearchClass *klass;

	g_return_val_if_fail (KZ_IS_SEARCH (search), NULL);

	klass = KZ_SEARCH_GET_CLASS (search);
	if (klass->get_search_result_html)
		return klass->get_search_result_html (search, text);
	else
		return NULL;
}

KzBookmark *
kz_search_get_search_result_bookmark (KzSearch *search, const gchar *text)
{
	KzSearchClass *klass;

	g_return_val_if_fail (KZ_IS_SEARCH(search), NULL);

	klass = KZ_SEARCH_GET_CLASS(search);
	if (klass->get_search_result_bookmark)
		return klass->get_search_result_bookmark(search, text);
	else
		return NULL;
}

gboolean
kz_search_register_document (KzSearch *search, const gchar *uri, const gchar *title, const gchar *contents, GTime mtime)
{
	KzSearchClass *klass;

	g_return_val_if_fail (KZ_IS_SEARCH (search), FALSE);

	klass = KZ_SEARCH_GET_CLASS (search);
	if (klass->register_document)
		return klass->register_document (search, uri, title, contents, mtime);
	else
		return FALSE;
}

gboolean
kz_search_unregister_document (KzSearch *search, const gchar *uri)
{
	KzSearchClass *klass;

	g_return_val_if_fail (KZ_IS_SEARCH (search), FALSE);

	klass = KZ_SEARCH_GET_CLASS (search);
	if (klass->unregister_document)
		return klass->unregister_document (search, uri);
	else
		return FALSE;
}

GPid
kz_search_optimize_index (KzSearch *search)
{
	KzSearchClass *klass;

	g_return_val_if_fail(KZ_IS_SEARCH(search), 0);

	klass = KZ_SEARCH_GET_CLASS(search);
	if (klass->optimize_index)
		return klass->optimize_index(search);
	else
		return 0;
}

void
kz_search_make_index (KzSearch *search)
{
	KzSearchClass *klass;

	g_return_if_fail(KZ_IS_SEARCH(search));

	klass = KZ_SEARCH_GET_CLASS(search);
	if (klass->make_index)
		klass->make_index(search);
}

gboolean
kz_search_exist_index_dir (KzSearch *search)
{
	KzSearchClass *klass;

	g_return_val_if_fail(KZ_IS_SEARCH(search), FALSE);

	klass = KZ_SEARCH_GET_CLASS(search);
	if (klass->exist_index_dir)
		return klass->exist_index_dir(search);
	else
		return FALSE;
}

