/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-session.c 2780 2007-01-05 11:37:13Z ikezoe $
 */

#include "kz-session.h"

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

typedef struct _KzSessionPrivate	KzSessionPrivate;
struct _KzSessionPrivate
{
	gboolean auto_save;
	gboolean frozen;
};

#define KZ_SESSION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_SESSION, KzSessionPrivate))

static void kz_bookmark_file_update  (KzBookmarkFile *bookmark);

static void cb_profile_changed (KzProfile *profile,
				const gchar *section, const gchar *key,
				const gchar *old_value,
				KzSession *session);

G_DEFINE_TYPE(KzSession, kz_session, KZ_TYPE_BOOKMARK_FILE)

static void
kz_session_class_init (KzSessionClass *klass)
{
	KzBookmarkFileClass *bookmark_file_class;
	GObjectClass *object_class;

	object_class = (GObjectClass *) klass;
	bookmark_file_class = (KzBookmarkFileClass *) klass;

	bookmark_file_class->update  = kz_bookmark_file_update;

	g_type_class_add_private(object_class, sizeof(KzSessionPrivate));
}

static void
kz_session_init (KzSession *session)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	priv->auto_save = FALSE;
	priv->frozen    = FALSE;
}

KzBookmark *
kz_session_new (const gchar *location, const gchar *title)
{
	KzSession *session;

	session = g_object_new(KZ_TYPE_SESSION,
			     "location",               location,
			     "title",                  title,
			     "file-type",              "XBEL",
			     NULL);

	if (!g_file_test(location, G_FILE_TEST_EXISTS))
		kz_bookmark_file_initialize(KZ_BOOKMARK_FILE(session));

	return KZ_BOOKMARK(session);
}


void
kz_session_set_auto_save (KzSession *session, gboolean auto_save)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	priv->auto_save = auto_save;
}

void
kz_session_freeze (KzSession *session)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	priv->frozen = TRUE;
}

void
kz_session_thaw (KzSession *session)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	priv->frozen = FALSE;
}

void
kz_session_set_profile (KzSession *session, KzProfile *profile)
{
	gboolean auto_save;
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	kz_profile_get_value(profile, "Session", "save",
			     &auto_save, sizeof(gboolean), KZ_PROFILE_VALUE_TYPE_BOOL);
	priv->auto_save = auto_save;
	g_signal_connect(profile,
			 "changed::Session",
			 G_CALLBACK(cb_profile_changed), session);
}

static void
cb_profile_changed (KzProfile *profile,
		    const gchar *section, const gchar *key,
		    const gchar *old_value,
		    KzSession *session)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(session);

	if (key && strcmp(key, "save"))
	{
		gboolean auto_save;
		kz_profile_get_value(profile, "Session", "save",
				&auto_save, sizeof(gboolean), KZ_PROFILE_VALUE_TYPE_BOOL);
		priv->auto_save = auto_save;
	}
}

static void
kz_bookmark_file_update (KzBookmarkFile *bookmark)
{
	KzSessionPrivate *priv = KZ_SESSION_GET_PRIVATE(bookmark);

	if (priv->frozen)
		return;

	if (priv->auto_save)
		kz_bookmark_file_save(bookmark);
}

