/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie <ashie@homa.ne.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark-base-menu-item.c 3869 2009-10-04 01:39:33Z ikezoe $
 */

#include "kz-bookmark-base-menu-item.h"

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "utils.h"

enum {
	PROP_0,
	PROP_BOOKMARK,
	PROP_KZ_WINDOW
};

/* object class methods */
static GObject* constructor      (GType           type,
                                  guint           n_props,
                                  GObjectConstructParam *props);
static void     dispose          (GObject        *object);
static void     set_property     (GObject        *object,
                                  guint           prop_id,
                                  const GValue   *value,
                                  GParamSpec     *pspec);
static void     get_property     (GObject        *object,
                                  guint           prop_id,
                                  GValue         *value,
                                  GParamSpec     *pspec);

static void 	connect_bookmark_signals                    (KzBookmarkBaseMenuItem *item,
	       						     KzBookmark             *bookmark);
static void 	disconnect_bookmark_signals                 (KzBookmarkBaseMenuItem *item,
	       						     KzBookmark             *bookmark);
G_DEFINE_TYPE(KzBookmarkBaseMenuItem, kz_bookmark_base_menu_item, GTK_TYPE_IMAGE_MENU_ITEM)

static void
kz_bookmark_base_menu_item_class_init (KzBookmarkBaseMenuItemClass *klass)
{
	GObjectClass *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->dispose      = dispose;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;
	gobject_class->constructor  = constructor;
	
	g_object_class_install_property
		(gobject_class,
		 PROP_BOOKMARK,
		 g_param_spec_object ("bookmark",
				      _("Bookmark"),
				      _("The boomark to show"),
				      KZ_TYPE_BOOKMARK,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property
		(gobject_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a home button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_bookmark_base_menu_item_init (KzBookmarkBaseMenuItem *menu)
{
	menu->kz            = NULL;
	menu->bookmark      = NULL;
}


static GObject*
constructor (GType                  type,
             guint                  n_props,
             GObjectConstructParam *props)
{
	KzBookmarkBaseMenuItem *item;
	GObject *object;
	const gchar *desc;
	gchar *tooltip_text = NULL;
	KzBookmark *bookmark;
	GObjectClass *klass = G_OBJECT_CLASS(kz_bookmark_base_menu_item_parent_class);

	object = klass->constructor(type, n_props, props);
	item = KZ_BOOKMARK_BASE_MENU_ITEM(object);
	bookmark = item->bookmark;
	
	/* tooltip */
	desc = kz_bookmark_get_description(bookmark);
	if (desc)
		tooltip_text = remove_tag(desc, strlen(desc));
	
	if (tooltip_text)
	{
		gtk_widget_set_tooltip_text(GTK_WIDGET(item), tooltip_text);
		g_free(tooltip_text);
	}
	connect_bookmark_signals(item, bookmark);

	return object;
}

static void
dispose (GObject *object)
{
	KzBookmarkBaseMenuItem *menu;

	if (G_OBJECT_CLASS (kz_bookmark_base_menu_item_parent_class)->dispose)
		G_OBJECT_CLASS (kz_bookmark_base_menu_item_parent_class)->dispose(object);

	menu = KZ_BOOKMARK_BASE_MENU_ITEM(object);

	if (menu->bookmark)
	{
		disconnect_bookmark_signals(menu, menu->bookmark);
		g_object_unref(menu->bookmark);
	}

	if (menu->kz)
		g_object_unref(menu->kz);

	menu->kz       = NULL;
	menu->bookmark = NULL;
}


static void
set_property (GObject         *object,
              guint            prop_id,
              const GValue    *value,
              GParamSpec      *pspec)
{
	KzBookmarkBaseMenuItem *menu = KZ_BOOKMARK_BASE_MENU_ITEM(object);
  
	switch (prop_id)
	{
	case PROP_BOOKMARK:
		menu->bookmark = g_value_dup_object(value);
		break;
	case PROP_KZ_WINDOW:
		menu->kz = g_value_dup_object(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject         *object,
              guint            prop_id,
              GValue          *value,
              GParamSpec      *pspec)
{
	KzBookmarkBaseMenuItem *item = KZ_BOOKMARK_BASE_MENU_ITEM(object);

	switch (prop_id)
	{
	case PROP_BOOKMARK:
		g_value_set_object(value, item->bookmark);
		break;
	case PROP_KZ_WINDOW:
		g_value_set_object(value, item->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
cb_notify_title (GObject *object, GParamSpec *pspec,
		 GtkWidget *widget)
{
	const gchar *title;

	title = kz_bookmark_get_title(KZ_BOOKMARK(object));

	if (GTK_IS_LABEL(GTK_BIN(widget)->child))
		gtk_label_set_text(GTK_LABEL(GTK_BIN(widget)->child), title);
}


static void
cb_notify_desc (GObject *object, GParamSpec *pspec,
		GtkWidget *widget)
{
	const gchar *desc;
	gchar *tooltip_text = NULL;
	
	desc = kz_bookmark_get_description(KZ_BOOKMARK(object));
	if (desc)
		tooltip_text = remove_tag(desc, strlen(desc));
	
	if (tooltip_text)
	{
		gtk_widget_set_tooltip_text(widget, tooltip_text);
		g_free(tooltip_text);
	}
}


static void
connect_bookmark_signals (KzBookmarkBaseMenuItem *item, KzBookmark *bookmark)
{
	g_signal_connect(bookmark, "notify::title",
			 G_CALLBACK(cb_notify_title),
			 item);
	g_signal_connect(bookmark, "notify::description",
			 G_CALLBACK(cb_notify_desc),
			 item);
}

static void
disconnect_bookmark_signals (KzBookmarkBaseMenuItem *item, KzBookmark *bookmark)
{
	g_signal_handlers_disconnect_by_func(bookmark,
					     G_CALLBACK(cb_notify_title),
					     item);
	g_signal_handlers_disconnect_by_func(bookmark,
					     G_CALLBACK(cb_notify_desc),
					     item);
}

