//
//  GameViewController.swift
//  KCD
//
//  Created by Hori,Masaki on 2016/12/31.
//  Copyright © 2016年 Hori,Masaki. All rights reserved.
//

import Cocoa
import WebKit
import JavaScriptCore

fileprivate extension Selector {
    static let reloadContent = #selector(GameViewController.reloadContent(_:))
    static let deleteCacheAndReload = #selector(GameViewController.deleteCacheAndReload(_:))
    static let screenShot = #selector(GameViewController.screenShot(_:))
}

func viewHierarchy(_ view: NSView) {
    print(view)
    view.subviews.forEach { viewHierarchy($0) }
}

class GameViewController: NSViewController {
    private static let gamePageURL = "http://www.dmm.com/netgame/social/-/gadgets/=/app_id=854854/"
    private static let loginPageURLPrefix = "https://www.dmm.com/my/-/login/=/"
    
    var webView: WKWebView!
    
    override var nibName: String! {
        return "GameViewController"
    }
    
    fileprivate var flashTopLeft = NSPoint(x: 500, y: 20)
    private var clipView: NSClipView {
        return view as! NSClipView  // swiftlint:disable:this force_cast
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        let pref = WKPreferences()
        pref.plugInsEnabled = true
        let config = WKWebViewConfiguration()
        config.preferences = pref
        config.applicationNameForUserAgent = AppDelegate.shared.appNameForUserAgent
        let frame = CGRect(x: 0, y: 0, width: 2000, height: 2_000)
        webView = WKWebView(frame: frame, configuration: config)
        webView.navigationDelegate = self
        
        clipView.documentView = webView
        
        adjustFlash()
        
        webView.allowsBackForwardNavigationGestures = false
        webView.allowsMagnification = false
        webView.allowsLinkPreview = false
        
        loadURLString(urlString: GameViewController.gamePageURL)
    }
    
    func loadURLString(urlString: String) {
        guard let url = URL(string: urlString) else { return }
        print("load \(urlString)")
        
        let req = URLRequest(url: url)
        webView.load(req)
    }
    func adjustFlash() {
        clipView.scroll(flashTopLeft)
    }
    
    @IBAction func reloadContent(_ sender: AnyObject?) {
        guard let _ = webView.url
            else {
                loadURLString(urlString: GameViewController.gamePageURL)
                return
        }
        // ゲームページでない場合はゲームページを表示する
        if let url = webView?.url, url.absoluteString != GameViewController.gamePageURL {
            loadURLString(urlString: GameViewController.gamePageURL)
            return
        }
        if let url = webView.url, url.absoluteString.hasPrefix(GameViewController.loginPageURLPrefix) {
            webView.reload(sender)
            return
        }
        
        adjustFlash()
        
        let prevDate = UserDefaults.standard.prevReloadDate
        if let prevDate = prevDate {
            let now = Date(timeIntervalSinceNow: 0.0)
            if now.timeIntervalSince(prevDate) < 1 * 60 {
                let untilDate = prevDate.addingTimeInterval(1 * 60)
                let date = DateFormatter.localizedString(from: untilDate, dateStyle: .none, timeStyle: .medium)
                let alert = NSAlert()
                alert.messageText = NSLocalizedString("Reload interval is too short?", comment: "")
                let format = NSLocalizedString("Reload interval is too short.\nWait until %@.", comment: "")
                alert.informativeText = String(format: format, date)
                alert.runModal()
                
                return
            }
        }
        
        webView.reload(sender)
        
        UserDefaults.standard.prevReloadDate = Date(timeIntervalSinceNow: 0.0)
    }
    @IBAction func deleteCacheAndReload(_ sender: AnyObject?) {
        let panel = ProgressPanel()
        guard let window = view.window,
            let panelWindow = panel.window
            else { return }
        panel.title = ""
        panel.message = NSLocalizedString("Deleting caches...", comment: "Deleting caches...")
        panel.animate = true
        
        window.beginSheet(panelWindow) { _ in NSSound(named: "Submarine")?.play() }
        
        AppDelegate.shared.clearCache()
        
        window.endSheet(panelWindow)
    }
    @IBAction func screenShot(_ sender: AnyObject?) {
        let frame = webView.visibleRect
        let screenshotBorder = UserDefaults.standard.screenShotBorderWidth
        let f = frame.insetBy(dx: -screenshotBorder, dy: -screenshotBorder)
        guard let rep = webView.bitmapImageRepForCachingDisplay(in: f) else { return }
        webView.cacheDisplay(in: frame, to: rep)
        AppDelegate.shared.registerScreenshot(rep, fromOnScreen: .zero)
    }
    
    override func validateMenuItem(_ menuItem: NSMenuItem) -> Bool {
        if menuItem.action == .reloadContent {
            guard let frameURL = webView.url?.absoluteString
                else { return true }
            switch frameURL {
            case GameViewController.gamePageURL:
                menuItem.title = NSLocalizedString("Reload", comment: "Reload menu, reload")
            case let s where s.hasPrefix(GameViewController.loginPageURLPrefix):
                menuItem.title = NSLocalizedString("Reload", comment: "Reload menu, reload")
            default:
                menuItem.title = NSLocalizedString("Back To Game", comment: "Reload menu, back to game")
            }
            return true
        }
        if menuItem.action == .deleteCacheAndReload {
            return true
        }
        if menuItem.action == .screenShot {
            return true
        }
        return false
    }
}

extension GameViewController: WKNavigationDelegate {
    func webView(_ webView: WKWebView, didFinish navigation: WKNavigation!) {
        webView.evaluateJavaScript(
            ["var iframe = document.getElementById('game_frame');",
             "var validIframe = 0;",
             "var atop = 0;",
             "var aleft = 0;",
             "var aWidfth = 0;",
             "if(iframe) {",
             "    validIframe = 1;",
             "    var rect = iframe.getBoundingClientRect();",
             "    atop = rect.top;",
             "    aleft = rect.left;",
             "    aWidth = rect.width;",
             "}",
             "var dict = {valid:validIframe, top:atop, left:aleft, width:aWidth};",
             "dict;"]
                .reduce("", +)
        ) { (dict, error) in
            error.map { print($0) }
            
            guard let param = dict as? [String: Any] else { return }
            guard let valid = param["valid"] as? Int else { return }
            guard valid == 1 else { return }
            guard let top = param["top"] as? Int,
                let left = param["left"] as? Int,
                let width = param["width"] as? Int
                else { return }
            
            DispatchQueue.main.async { [weak self] in
                guard let `self` = self else { return }
                
                self.flashTopLeft = NSPoint(x: CGFloat(left + (width - 800) / 2),
                                            y: CGFloat(top + 16))
                self.adjustFlash()
            }
        }
    }
}

extension GameViewController: WebFrameLoadDelegate, WebUIDelegate {
    private static let excludeMenuItemTag = [
        WebMenuItemTagOpenLinkInNewWindow,
        WebMenuItemTagDownloadLinkToDisk,
        WebMenuItemTagOpenImageInNewWindow,
        WebMenuItemTagOpenFrameInNewWindow,
        WebMenuItemTagGoBack,
        WebMenuItemTagGoForward,
        WebMenuItemTagStop,
        WebMenuItemTagReload
    ]
    
    func webView(_ sender: WebView!, didFinishLoadFor frame: WebFrame!) {
        guard let path = frame.dataSource?.initialRequest.url?.path else { return }
        
        let handler: (JSContext?, JSValue?) -> Void = { (_, exception) in
            if let exception = exception {
                print("Caught exception in evaluteScript -> \(exception)")
            } else {
                print("Caught exception in evaluteScript")
            }
        }
        
        if path.hasSuffix("gadgets/ifr") {
            guard let context = frame.javaScriptContext else { return }
            context.exceptionHandler = handler
            context.evaluateScript(
                ["var emb = document.getElementById('flashWrap');",
                 "var rect = emb.getBoundingClientRect();",
                 "var atop = rect.top;",
                 "var aleft = rect.left;"]
                    .reduce("", +)
            )
            let top = context.objectForKeyedSubscript("atop").toDouble()
            let left = context.objectForKeyedSubscript("aleft").toDouble()
            flashTopLeft = NSPoint(x: CGFloat(left), y: webView.frame.size.height - CGFloat(top) - 480)
        }
        if path.hasSuffix("app_id=854854") {
            guard let context = frame.javaScriptContext else { return }
            context.exceptionHandler = handler
            context.evaluateScript(
                ["var iframe = document.getElementById('game_frame');",
                 "var validIframe = 0;",
                 "if(iframe) {",
                 "    validIframe = 1;",
                 "    var rect = iframe.getBoundingClientRect();",
                 "    var atop = rect.top;",
                 "    var aleft = rect.left;",
                 "}"]
                    .reduce("", +)
            )
            let validIframe = context.objectForKeyedSubscript("validIframe").toInt32()
            guard validIframe != 0 else { return }
            let top = context.objectForKeyedSubscript("atop").toDouble()
            let left = context.objectForKeyedSubscript("aleft").toDouble()
            flashTopLeft = NSPoint(x: flashTopLeft.x + CGFloat(left), y: flashTopLeft.y - CGFloat(top))
            adjustFlash()
        }
    }
    
    func webView(_ sender: WebView!,
                 contextMenuItemsForElement element: [AnyHashable: Any]!,
                 defaultMenuItems: [Any]!) -> [Any]! {
        guard let menuItems = defaultMenuItems as? [NSMenuItem] else { return [] }
        return menuItems.flatMap {
            GameViewController.excludeMenuItemTag.contains($0.tag) ? nil : $0
        }
    }
}
